package io.sendon.contacts;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

import io.sendon.Log;
import io.sendon.base.SendonClient;
import io.sendon.base.SendonJsonResponse;
import io.sendon.contacts.request.ContactBuilder;
import io.sendon.contacts.request.CreateBlocklistRequest;
import io.sendon.contacts.request.GetBlocklistRequest;
import io.sendon.contacts.response.AddBlocklist;
import io.sendon.contacts.response.DeleteBlocklist;
import io.sendon.contacts.response.GetBlocklist;

/**
 * SendonContacts provides methods to interact with the Sendon Contacts API.
 * It allows adding, retrieving, and deleting phone numbers from the blocklist.
 */
public class SendonContacts extends SendonClient {

  /**
   * Initializes the SendonContacts client with user credentials.
   *
   * @param userId The user ID for authentication.
   * @param apiKey The API key for authentication.
   */
  public SendonContacts(String userId, String apiKey) {
    super(userId, apiKey);
  }

  /**
   * Initializes the SendonContacts client with user credentials and an option to use OkHttp.
   *
   * @param userId The user ID for authentication.
   * @param apiKey The API key for authentication.
   * @param useOkHttp Whether to use OkHttp for HTTP requests.
   */
  public SendonContacts(String userId, String apiKey, boolean useOkHttp) {
    super(userId, apiKey, useOkHttp);
  }

  /**
   * Adds a phone number to the blocklist.
   *
   * @param phoneNumber The phone number to be added to the blocklist.
   * @return An AddBlocklist object containing the API response, or null if an error occurs.
   */
  public AddBlocklist addBlocklist(String phoneNumber) {
    return addBlocklist(phoneNumber, null, null, null, null, null);
  }

  /**
   * Adds a phone number to the blocklist with optional parameters.
   *
   * @param phoneNumber The phone number to be added to the blocklist.
   * @param messageType The message type (SMS, KAKAO, RCS). Default is SMS.
   * @param senderNumber The sender number to block from.
   * @param kakaoChannelId The Kakao channel ID for Kakao message blocking.
   * @param rcsChatbotId The RCS chatbot ID for RCS message blocking.
   * @param rcsBrandId The RCS brand ID for RCS message blocking.
   * @return An AddBlocklist object containing the API response, or null if an error occurs.
   */
  public AddBlocklist addBlocklist(String phoneNumber, String messageType, String senderNumber, 
                                 String kakaoChannelId, String rcsChatbotId, String rcsBrandId) {
    CreateBlocklistRequest request = new CreateBlocklistRequest(phoneNumber, messageType, senderNumber, 
                                                              kakaoChannelId, rcsChatbotId, rcsBrandId);
    return addBlocklist(request);
  }

  /**
   * Adds a phone number to the blocklist using a request object.
   *
   * @param request The CreateBlocklistRequest object containing all parameters.
   * @return An AddBlocklist object containing the API response, or null if an error occurs.
   */
  public AddBlocklist addBlocklist(CreateBlocklistRequest request) {
    Log.d("addBlocklist: " + request.phoneNumber + ", messageType=" + request.messageType + ", senderNumber=" + request.senderNumber);

    Gson gson = new Gson();
    String requestJson = gson.toJson(request);

    try {
      SendonJsonResponse sendonResponse = parseJsonResponse(post("/v2/contacts/blocklist", requestJson));
      return new AddBlocklist(sendonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Retrieves the blocklist with pagination support.
   *
   * @param cursor The cursor for pagination.
   * @param limit The maximum number of entries to retrieve.
   * @return A GetBlocklist object containing the API response, or null if an error occurs.
   */
  public GetBlocklist getBlocklist(int cursor, int limit) {
    return getBlocklist(null, null, null, null, null, cursor, limit);
  }

  /**
   * Retrieves the blocklist with advanced filtering and pagination support.
   *
   * @param senderNumber Filter by specific sender number.
   * @param kakaoChannelIds Array of Kakao channel IDs to filter by.
   * @param blockTypes Array of block types (WEB, API, ARS) to filter by.
   * @param startDate Start date for filtering (YYYY-MM-DD HH:mm:ss format).
   * @param endDate End date for filtering (YYYY-MM-DD HH:mm:ss format).
   * @param cursor The cursor for pagination (default: 0).
   * @param limit The maximum number of entries to retrieve (default: 20).
   * @return A GetBlocklist object containing the API response, or null if an error occurs.
   */
  public GetBlocklist getBlocklist(String senderNumber, String[] kakaoChannelIds, String[] blockTypes,
                                 String startDate, String endDate, Integer cursor, Integer limit) {
    GetBlocklistRequest request = new GetBlocklistRequest(senderNumber, kakaoChannelIds, blockTypes,
                                                        startDate, endDate, cursor, limit);
    return getBlocklist(request);
  }

  /**
   * Retrieves the blocklist using a request object.
   *
   * @param request The GetBlocklistRequest object containing all filter and pagination parameters.
   * @return A GetBlocklist object containing the API response, or null if an error occurs.
   */
  public GetBlocklist getBlocklist(GetBlocklistRequest request) {
    StringBuilder queryParams = new StringBuilder();
    boolean hasParam = false;

    if (request.senderNumber != null) {
      queryParams.append("senderNumber=").append(request.senderNumber);
      hasParam = true;
    }

    if (request.kakaoChannelIds != null && request.kakaoChannelIds.length > 0) {
      for (String channelId : request.kakaoChannelIds) {
        if (hasParam) queryParams.append("&");
        queryParams.append("kakaoChannelIds=").append(channelId);
        hasParam = true;
      }
    }

    if (request.blockTypes != null && request.blockTypes.length > 0) {
      for (String blockType : request.blockTypes) {
        if (hasParam) queryParams.append("&");
        queryParams.append("blockType=").append(blockType);
        hasParam = true;
      }
    }

    if (request.startDate != null) {
      if (hasParam) queryParams.append("&");
      queryParams.append("startDate=").append(request.startDate);
      hasParam = true;
    }

    if (request.endDate != null) {
      if (hasParam) queryParams.append("&");
      queryParams.append("endDate=").append(request.endDate);
      hasParam = true;
    }

    if (request.cursor != null) {
      if (hasParam) queryParams.append("&");
      queryParams.append("cursor=").append(request.cursor);
      hasParam = true;
    }

    if (request.limit != null) {
      if (hasParam) queryParams.append("&");
      queryParams.append("limit=").append(request.limit);
      hasParam = true;
    }

    String url = "/v2/contacts/blocklist";
    if (hasParam) {
      url += "?" + queryParams.toString();
    }

    Log.d("getBlocklist: " + url);

    try {
      SendonJsonResponse sendonResponse = parseJsonResponse(get(url));
      return new GetBlocklist(sendonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Deletes a phone number from the blocklist by its block ID.
   *
   * @param blocklistId The ID of the blocklist entry to delete.
   * @return A DeleteBlocklist object containing the API response, or null if an error occurs.
   */
  public DeleteBlocklist deleteBlocklist(long blocklistId) {
    Log.d("deleteBlocklist: " + blocklistId);

    JsonObject bodyJson = new JsonObject();

    try {
      SendonJsonResponse sendonResponse = parseJsonResponse(this.post("/v2/contacts/blocklist/" + blocklistId + "/delete", bodyJson.toString()));
      return new DeleteBlocklist(sendonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Creates a new builder for creating blocklist entries.
   * Provides a fluent API for configuring blocklist creation requests.
   *
   * @return A new {@link ContactBuilder.CreateBlocklistBuilder} instance.
   */
  public static ContactBuilder.CreateBlocklistBuilder createBlocklistBuilder() {
    return ContactBuilder.createBlocklist();
  }

  /**
   * Creates a new builder for retrieving blocklist entries.
   * Provides a fluent API for configuring blocklist retrieval requests with filtering and pagination.
   *
   * @return A new {@link ContactBuilder.GetBlocklistBuilder} instance.
   */
  public static ContactBuilder.GetBlocklistBuilder getBlocklistBuilder() {
    return ContactBuilder.getBlocklist();
  }
}