package io.rudin.cdi.entitymanager.repository;

import java.io.Serializable;
import java.util.List;

import javax.enterprise.inject.Instance;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Root;

import io.rudin.cdi.entitymanager.api.crud.CRUD;
import io.rudin.cdi.entitymanager.api.entitymanager.EntityManagerCloser;

public abstract class Repository<E, K extends Serializable> implements CRUD<E, K> {

	protected abstract Class<E> getEntityClass();

	protected abstract String getIDFieldName();

	protected abstract Instance<EntityManager> getEntityManagerInstance();

	protected void read(){}

	protected void write(){}

	@Override
	public List<E> findAll(){
		read();

		try (EntityManagerCloser emc = new EntityManagerCloser(getEntityManagerInstance())){
			EntityManager em = emc.getEntityManager();

			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<E> query = cb.createQuery(getEntityClass());

			Root<E> root = query.from(getEntityClass());
			CriteriaQuery<E> select = query.select(root);
			select.where();

			return em.createQuery(query).getResultList();

		}
	}

	@Override
	public E findById(K id) {
		read();

		try (EntityManagerCloser emc = new EntityManagerCloser(getEntityManagerInstance())){
			EntityManager em = emc.getEntityManager();
			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<E> query = cb.createQuery(getEntityClass());

			Root<E> root = query.from(getEntityClass());
			CriteriaQuery<E> select = query.select(root);
			select.where(cb.equal(root.get(getIDFieldName()), id));

			return em
					.createQuery(query)
					.getResultList()
					.stream()
					.findFirst()
					.orElse(null);
		}

	}


	@Override
	public E save(E item) {
		write();
		try (EntityManagerCloser emc = new EntityManagerCloser(getEntityManagerInstance())){
			EntityManager em = emc.getEntityManager();

			EntityTransaction tx = em.getTransaction();

			tx.begin();
			E entity = em.merge(item);
			tx.commit();
			return entity;
		}
	}


	@Override
	public void remove(E item) {
		write();

		try (EntityManagerCloser emc = new EntityManagerCloser(getEntityManagerInstance())){
			EntityManager em = emc.getEntityManager();

			EntityTransaction tx = em.getTransaction();
			tx.begin();
			em.remove(item);
			tx.commit();
		}
	}
}
