package io.rudin.cdi.entitymanager.bean;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.context.spi.CreationalContext;
import javax.enterprise.inject.Any;
import javax.enterprise.inject.Default;
import javax.enterprise.inject.spi.Bean;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.InjectionPoint;
import javax.enterprise.util.AnnotationLiteral;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.rudin.cdi.entitymanager.api.PersistenceConfiguration;
import io.rudin.cdi.entitymanager.api.PersistenceConfigurationProperties;
import io.rudin.cdi.entitymanager.config.PersistenceContextConfiguration;

public class EntityManagerFactoryBean implements Bean<EntityManagerFactory>
{
	private static final Logger logger = LoggerFactory.getLogger(EntityManagerFactoryBean.class);


	public EntityManagerFactoryBean(BeanManager bm, PersistenceContextConfiguration config)
	{
		this.bm = bm;
		this.config = config;
		
		logger.debug("Created new factory-bean for qualifiers: {}", config.getQualifiers());
	}

	private final BeanManager bm;

	private final PersistenceContextConfiguration config;

	private EntityManagerFactory factory;

	private PersistenceConfiguration configuration;

	@SuppressWarnings("serial")
	@Override
	public synchronized EntityManagerFactory create(CreationalContext<EntityManagerFactory> creationalContext)
	{
		if (factory == null)
		{
			Set<Annotation> qualifiers = new HashSet<>();
			if (config.getQualifiers().isEmpty())
				qualifiers.add(new AnnotationLiteral<Default>() {});
			else
				qualifiers.addAll(config.getQualifiers());
			
			Set<Bean<?>> beans = bm.getBeans(PersistenceConfiguration.class, qualifiers.toArray(new Annotation[qualifiers.size()]));
			configuration = (PersistenceConfiguration) bm.getReference(beans.iterator().next(), PersistenceConfiguration.class, creationalContext);
			
			logger.debug("Got config class: {}", configuration.getClass().getName());


			Map<String, String> defaultConfig = new HashMap<String, String>();
			defaultConfig.put("hibernate.connection.provider_class", "org.hibernate.connection.C3P0ConnectionProvider");
			defaultConfig.put("hibernate.c3p0.min_size", "5");
			defaultConfig.put("hibernate.c3p0.max_size", "100");
			defaultConfig.put("hibernate.c3p0.timeout", "100");
			defaultConfig.put("hibernate.c3p0.max_statements", "50");
			defaultConfig.put("hibernate.c3p0.idle_test_period", "1000");
			defaultConfig.put("hibernate.c3p0.validate", "true");

			defaultConfig.put("javax.persistence.jdbc.driver", configuration.getDriver());
			defaultConfig.put("javax.persistence.jdbc.url", configuration.getUrl());
			defaultConfig.put("javax.persistence.jdbc.user", configuration.getUsername());
			defaultConfig.put("javax.persistence.jdbc.password", configuration.getPassword());

		
			Set<Annotation> propertyQualifiers = new HashSet<>();
			if (config.getQualifiers().isEmpty())
				propertyQualifiers.add(new AnnotationLiteral<Any>() {});
			else
				propertyQualifiers.addAll(config.getQualifiers());
			
			propertyQualifiers.add(new AnnotationLiteral<PersistenceConfigurationProperties>() {});
			
			Set<Bean<?>> propertyBeans = bm.getBeans(Object.class, propertyQualifiers.toArray(new Annotation[propertyQualifiers.size()]));
			Iterator<Bean<?>> iterator = propertyBeans.iterator();
			
			while (iterator.hasNext())
			{
				Bean<?> next = iterator.next();
				
				@SuppressWarnings("unchecked")
				Map<String, String> map = (Map<String, String>) bm.getReference(next, Object.class, creationalContext);
				//TODO: throw error if not a map
				
				logger.debug("Got additional properties: {}", map);
				defaultConfig.putAll(map);
			}

			logger.info("Creating entitymanager factory for pu: {}", configuration.getUnitName());
			factory = Persistence.createEntityManagerFactory(configuration.getUnitName(), defaultConfig);
		}

		return factory;
	}

	@Override
	public synchronized void destroy(EntityManagerFactory instance, CreationalContext<EntityManagerFactory> creationalContext) {
		if (factory != null)
		{
			logger.info("Shutting down entitymanager factory for pu: {}", configuration.getUnitName());
			factory.close();
		}
	}

	@Override
	public Set<Type> getTypes() {
		Set<Type> set = new HashSet<>();
		set.add(Object.class);
		set.add(EntityManagerFactory.class);

		return set;
	}

	@SuppressWarnings("serial")
	@Override
	public Set<Annotation> getQualifiers() {
		Set<Annotation> set = new HashSet<>();
		
		if (config.getQualifiers().isEmpty())
			set.add(new AnnotationLiteral<Default>() {});
		else
			set.addAll(config.getQualifiers());

		return set;
	}

	@Override
	public Class<? extends Annotation> getScope() {
		return ApplicationScoped.class;
	}

	@Override
	public String getName() {
		return "entitymanagerfactory("+config.getQualifiers()+")";
	}

	@Override
	public Set<Class<? extends Annotation>> getStereotypes() {
		return Collections.emptySet();
	}

	@Override
	public boolean isAlternative() {
		return false;
	}

	@Override
	public Class<?> getBeanClass() {
		return EntityManagerFactory.class;
	}

	@Override
	public Set<InjectionPoint> getInjectionPoints() {
		return Collections.emptySet();
	}

	@Override
	public boolean isNullable() {
		return false;
	}

}
