package io.rudin.cdi.entitymanager.bean;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.enterprise.context.Dependent;
import javax.enterprise.context.spi.CreationalContext;
import javax.enterprise.inject.Any;
import javax.enterprise.inject.Default;
import javax.enterprise.inject.spi.Bean;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.InjectionPoint;
import javax.enterprise.util.AnnotationLiteral;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.rudin.cdi.entitymanager.api.PersistenceConfiguration;
import io.rudin.cdi.entitymanager.config.PersistenceContextConfiguration;
import io.rudin.cdi.entitymanager.status.PersistenceStatusImpl;

public class EntityManagerBean implements Bean<EntityManager>
{
	private static final Logger logger = LoggerFactory.getLogger(EntityManagerBean.class);

	public EntityManagerBean(BeanManager bm, PersistenceContextConfiguration config, EntityManagerFactoryBean factoryBean, PersistenceStatusBean statusBean)
	{
		this.bm = bm;
		this.config = config;
		this.factoryBean = factoryBean;
		this.statusBean = statusBean;
	}

	private final BeanManager bm;

	private final PersistenceStatusBean statusBean;

	private final EntityManagerFactoryBean factoryBean;

	private final PersistenceContextConfiguration config;

	private PersistenceConfiguration configuration;


	@Override
	public EntityManager create(CreationalContext<EntityManager> creationalContext)
	{
		EntityManagerFactory emf = (EntityManagerFactory) bm.getReference(factoryBean, EntityManagerFactory.class, creationalContext);
		PersistenceStatusImpl status = (PersistenceStatusImpl) bm.getReference(statusBean, PersistenceStatusImpl.class, creationalContext);

		Annotation[] qualifiers = config.getQualifiers().toArray(new Annotation[config.getQualifiers().size()]);
		Set<Bean<?>> beans = bm.getBeans(PersistenceConfiguration.class, qualifiers);
		configuration = (PersistenceConfiguration) bm.getReference(beans.iterator().next(), PersistenceConfiguration.class, creationalContext);

		logger.debug("Got config class: {}", configuration.getClass().getName());

		status.incrementOpenCount(1);
		EntityManager entityManager = emf.createEntityManager();
		logger.debug("Created new entitymanager: {}", entityManager);

		return entityManager;
	}

	@Override
	public void destroy(EntityManager instance, CreationalContext<EntityManager> creationalContext)
	{
		PersistenceStatusImpl status = (PersistenceStatusImpl) bm.getReference(statusBean, PersistenceStatusImpl.class, creationalContext);

		if (instance != null && instance.isOpen())
		{
			logger.debug("Disposed entitymanager: {}", instance);
			status.incrementOpenCount(-1);
			instance.close();
		}
	}

	@Override
	public Set<Type> getTypes()
	{
		Set<Type> set = new HashSet<>();
		set.add(Object.class);
		set.add(EntityManager.class);

		return set;
	}

	@SuppressWarnings("serial")
	@Override
	public Set<Annotation> getQualifiers() {
		Set<Annotation> set = new HashSet<>();

		if (config.getQualifiers().isEmpty())
		{
			set.add(new AnnotationLiteral<Default>() {});
			set.add(new AnnotationLiteral<Any>() {});
		}
		else
			set.addAll(config.getQualifiers());

		return set;
	}

	@Override
	public Class<? extends Annotation> getScope() {
		return Dependent.class;
	}

	@Override
	public String getName() {
		return "entitymanager("+config.getQualifiers()+")";
	}

	@Override
	public Set<Class<? extends Annotation>> getStereotypes() {
		return Collections.emptySet();
	}

	@Override
	public boolean isAlternative() {
		return false;
	}

	@Override
	public Class<?> getBeanClass() {
		return EntityManagerFactory.class;
	}

	@Override
	public Set<InjectionPoint> getInjectionPoints() {
		return Collections.emptySet();
	}

	@Override
	public boolean isNullable() {
		return false;
	}

}
