package io.rudin.cdi.entitymanager;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.enterprise.event.Observes;
import javax.enterprise.inject.spi.AfterBeanDiscovery;
import javax.enterprise.inject.spi.AnnotatedType;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.Extension;
import javax.enterprise.inject.spi.ProcessAnnotatedType;
import javax.inject.Qualifier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.rudin.cdi.entitymanager.api.PersistenceConfiguration;
import io.rudin.cdi.entitymanager.bean.EntityManagerBean;
import io.rudin.cdi.entitymanager.bean.EntityManagerFactoryBean;
import io.rudin.cdi.entitymanager.bean.PersistenceStatusBean;
import io.rudin.cdi.entitymanager.config.PersistenceContextConfiguration;

public class EntityManagerExtension implements Extension
{

	private static final Logger logger = LoggerFactory.getLogger(EntityManagerExtension.class);
	
	private final List<PersistenceContextConfiguration> contextList = new ArrayList<PersistenceContextConfiguration>();
	
	<X extends PersistenceConfiguration> void observeType(@Observes ProcessAnnotatedType<X> pat)
	{
		//collect config beans
		AnnotatedType<X> type = pat.getAnnotatedType();
		Class<X> javaClass = type.getJavaClass();

		if (javaClass == PersistenceConfiguration.class)
			//interface
			return;
		
		logger.debug("Found config: {}", javaClass.getName());

		Set<Annotation> annotations = type.getAnnotations();
		
		Set<Annotation> qualifiers = new HashSet<Annotation>();
		for (Annotation a: annotations)
			if (isQualifier(a))
				qualifiers.add(a);
		
		logger.debug("	with qualifiers: {}", qualifiers);
		
		PersistenceContextConfiguration configuration = new PersistenceContextConfiguration();
		configuration.setQualifiers(qualifiers);
		
		contextList.add(configuration);
	}
	
	void afterBeanDiscovery(@Observes AfterBeanDiscovery abd, BeanManager bm)
	{
		//Register em beans
		for (PersistenceContextConfiguration ctx: contextList)
		{
			PersistenceStatusBean statusBean = new PersistenceStatusBean(bm, ctx);
			abd.addBean(statusBean);
			
			EntityManagerFactoryBean factoryBean = new EntityManagerFactoryBean(bm, ctx);
			abd.addBean(factoryBean);
			
			EntityManagerBean entityManagerBean = new EntityManagerBean(bm, ctx, factoryBean, statusBean);
			abd.addBean(entityManagerBean);
			
		}
	}
	
	private boolean isQualifier(Annotation e)
	{
		return e.annotationType().getAnnotation(Qualifier.class) != null;
	}
	
	
}
