/*
 * QALIPSIS
 * Copyright (C) 2025 AERIS IT Solutions GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package io.qalipsis.api.steps

import io.qalipsis.api.constraints.PositiveOrZeroDuration
import java.time.Duration

/**
 * Interface to mark a [StepSpecification] as a singleton. A singleton step is one that is not executed in minions-related context
 * but acts on his own.
 *
 * It is often the case for the datasource steps, which are not executed once per minion
 * (implying potentially millions of requests) but only once periodically, broadcasting its result to all the minions.
 *
 * @author Eric Jessé
 */
interface SingletonStepSpecification {

    /**
     * Configuration of the singleton.
     */
    val singletonConfiguration: SingletonConfiguration

    /**
     * Specifies whether the step should be really executed as a singleton.
     */
    val isReallySingleton: Boolean
        get() = (singletonConfiguration.type != SingletonType.SEQUENTIAL)
}

data class SingletonConfiguration(
    /**
     * Nature of the singleton.
     */
    var type: SingletonType,

    /**
     * Size of the buffer to keep data before the minions are ready to read them.
     * Unlimited by default.
     */
    var bufferSize: Int = -1,

    /**
     * Time to idle of a subscription. Once idle a subscription passed this duration, it is automatically cancelled.
     * When set to `Duration.ZERO` (default) or less, there is no timeout.
     */
    var idleTimeout: @PositiveOrZeroDuration Duration = Duration.ZERO
)

/**
 * Defines how a singleton [io.qalipsis.api.steps.Step] will distribute the data to its children.
 *
 * @author Eric Jessé
 */
enum class SingletonType {
    /**
     * Each record is provided only once to a unique minion. The singleton acts as a FIFO-Provider.
     */
    UNICAST,

    /**
     * All the records are provided to all the minions. The singleton acts as a Pub/Sub-Provider.
     */
    BROADCAST,

    /**
     * Like [BROADCAST] but looping to the beginning when reaching the end of the values.
     */
    LOOP,

    /**
     * Disables the side-execution of the step and integrates it into its parent execution workflow.
     */
    SEQUENTIAL

}

/**
 * Interface for singleton that can forward the data by distributing each message individually to the requesting minions.
 *
 * @author Eric Jessé
 */
interface UnicastSpecification : SingletonStepSpecification {

    /**
     * Distributes each record to a unique one minion. The distribution is fair: each minion will get the next available
     * record in the order of asking.
     *
     * @param bufferSize Size of the buffer to keep the data before the first subscription.
     * @param idleTimeout Time to idle of a subscription.
     */
    fun unicast(bufferSize: Int = -1, idleTimeout: Duration = Duration.ZERO) {
        singletonConfiguration.type = SingletonType.UNICAST
        singletonConfiguration.bufferSize = bufferSize
        singletonConfiguration.idleTimeout = idleTimeout
    }
}

/**
 * Interface for singleton that can forward the data by distributing each message to all the minions.
 *
 * @author Eric Jessé
 */
interface BroadcastSpecification : SingletonStepSpecification {

    /**
     * Distributes each record to all the minions. The first record a minion will get in the latest generated by
     * the source at that moment.
     *
     * For each minion, a cursor is kept in order to evaluate the position of the next record to read. If a given minion
     * did not polled a value twice in a time frame defined by [idleTimeout], the cursor is cancelled and any
     * further poll will fail.
     *
     * A value of `Duration.ZERO` (default) defines no limitation but implies more memory consumption.
     *
     * @param bufferSize Size of the buffer to keep to pass past data to new subscriptions.
     * @param idleTimeout Time to idle of a subscription.
     */
    fun broadcast(bufferSize: Int = -1, idleTimeout: Duration = Duration.ZERO) {
        singletonConfiguration.type = SingletonType.BROADCAST
        singletonConfiguration.bufferSize = bufferSize
        singletonConfiguration.idleTimeout = idleTimeout
    }

}

/**
 * Interface for singleton that can forward the data again and again to all the minions.
 *
 * @author Eric Jessé
 */
interface LoopableSpecification : SingletonStepSpecification {

    /**
     * Distributes each record to a unique one minion. The distribution is fair: each minion will get the next available
     * record in the order of asking.
     *
     * For each minion, a cursor is kept in order to evaluate the position of the next record to read. If a given minion
     * did not polled a value twice in a time frame defined by [idleTimeout], the cursor is cancelled and any
     * further poll will fail.
     *
     * A value of `Duration.ZERO` (default) defines no limitation but implies more memory consumption.
     *
     * @param idleTimeout Time to idle of a subscription.
     */
    fun loop(idleTimeout: Duration = Duration.ZERO) {
        singletonConfiguration.type = SingletonType.LOOP
        singletonConfiguration.idleTimeout = idleTimeout
        singletonConfiguration.bufferSize = -1
    }

}

/**
 * Interface for singleton that can also run as non-singleton.
 *
 * @author Eric Jessé
 */
interface NoSingletonSpecification : SingletonStepSpecification {

    /**
     * Disables the singleton-execution of the step and integrates it into its parent execution workflow.
     */
    fun sequential() {
        singletonConfiguration.type = SingletonType.SEQUENTIAL
    }

}
