/*
 * QALIPSIS
 * Copyright (C) 2025 AERIS IT Solutions GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package io.qalipsis.api.steps

import cool.graph.cuid.Cuid
import io.micronaut.core.annotation.Introspected
import io.qalipsis.api.context.CorrelationRecord
import io.qalipsis.api.context.StepName
import io.qalipsis.api.scenario.ScenarioSpecification
import java.time.Duration
import javax.validation.constraints.NotBlank

/**
 * Specification for a [io.qalipsis.core.factory.steps.LeftJoinStep].
 *
 * @author Eric Jessé
 */
@Introspected
data class InnerJoinStepSpecification<INPUT, OUTPUT>(
    val primaryKeyExtractor: (CorrelationRecord<INPUT>) -> Any?,
    val secondaryKeyExtractor: (CorrelationRecord<out Any?>) -> Any?,
    val secondaryStepName: @NotBlank StepName
) : AbstractStepSpecification<INPUT?, OUTPUT?, InnerJoinStepSpecification<INPUT, OUTPUT>>() {
    var cacheTimeout: Duration = Duration.ZERO
}

/**
 * Correlates the records coming from an earlier step with the ones generated by the specified specification.
 * Correlation keys are used in order to be able to map the record from each source together.
 *
 * @param using function to execute on the records from the local earlier step to extract the correlation key.
 * @param on specification of the step that should generate the remote records.
 * @param having function to execute on the remote records to extract the correlation key.
 *
 * @author Eric Jessé
 */
private fun <INPUT, OTHER_INPUT> StepSpecification<*, INPUT, *>.innerJoin(
    using: (CorrelationRecord<INPUT>) -> Any?,
    on: (scenario: ScenarioSpecification) -> StepSpecification<*, OTHER_INPUT, *>,
    having: (CorrelationRecord<OTHER_INPUT>) -> Any?
): InnerJoinStepSpecification<INPUT, Pair<INPUT, OTHER_INPUT>> {

    // Since the relationship is performed in the step name, we generate one in case it is not specified by the user.
    val secondaryStep = on(scenario)
    if (secondaryStep.name.isBlank()) {
        secondaryStep.name = Cuid.createCuid()
    }
    // We force the step to be known by the scenario.
    scenario.register(secondaryStep)

    @Suppress("UNCHECKED_CAST")
    val step = InnerJoinStepSpecification<INPUT, Pair<INPUT, OTHER_INPUT>>(
        using,
        having as CorrelationRecord<*>.() -> Any?,
        secondaryStep.name
    )
    this.add(step)
    return step
}

/**
 * Correlates the records coming from an earlier step with the ones generated by step with the specified name.
 * Correlation keys are used in order to be able to map the record from each source together.
 *
 * @param using function to execute on the records from the local earlier step to extract the correlation key.
 * @param on name of the step, which should have be specified earlier in the same scenario.
 * @param having function to execute on the remote records to extract the correlation key.
 *
 * @author Eric Jessé
 */
private fun <INPUT, OTHER_INPUT> StepSpecification<*, INPUT, *>.innerJoin(
    using: (CorrelationRecord<INPUT>) -> Any?,
    on: String,
    having: (CorrelationRecord<OTHER_INPUT>) -> Any?
): InnerJoinStepSpecification<INPUT, Pair<INPUT, OTHER_INPUT>> {

    @Suppress("UNCHECKED_CAST")
    val step = InnerJoinStepSpecification<INPUT, Pair<INPUT, OTHER_INPUT>>(
        using,
        having as CorrelationRecord<out Any?>.() -> Any?,
        on
    )
    this.add(step)
    return step
}

interface InnerJoinUsingSpecification<INPUT> {

    /**
     * Specifies how to extract the correlation key from the left side.
     */
    fun <KEY> using(block: (CorrelationRecord<INPUT>) -> KEY?): InnerJoinOnSpecification<INPUT, KEY>
}

internal class InnerJoinUsingSpecificationImpl<INPUT>(
    private val previousStep: StepSpecification<*, INPUT, *>
) : InnerJoinUsingSpecification<INPUT> {

    override fun <KEY> using(block: (CorrelationRecord<INPUT>) -> KEY?): InnerJoinOnSpecification<INPUT, KEY> {
        return InnerJoinOnSpecificationImpl(previousStep, block)
    }
}

interface InnerJoinOnSpecification<INPUT, KEY> {

    /**
     * Specifies the step that generates the records for the right side.
     */
    fun <OTHER_INPUT> on(block: (scenario: ScenarioSpecification) -> StepSpecification<*, OTHER_INPUT, *>): InnerJoinHavingSpecification<INPUT, KEY, OTHER_INPUT>

    /**
     * Specifies the name of the step that generates the records for the right side.
     */
    fun <OTHER_INPUT> on(stepName: StepName): InnerJoinHavingSpecification<INPUT, KEY, OTHER_INPUT>

}

internal class InnerJoinOnSpecificationImpl<INPUT, KEY>(
    private val previousStep: StepSpecification<*, INPUT, *>,
    private val using: (CorrelationRecord<INPUT>) -> Any?
) : InnerJoinOnSpecification<INPUT, KEY> {

    override fun <OTHER_INPUT> on(block: (scenario: ScenarioSpecification) -> StepSpecification<*, OTHER_INPUT, *>): InnerJoinHavingSpecification<INPUT, KEY, OTHER_INPUT> {
        return InnerJoinHavingSpecificationImpl(previousStep, using, block)
    }

    override fun <OTHER_INPUT> on(stepName: StepName): InnerJoinHavingSpecification<INPUT, KEY, OTHER_INPUT> {
        return InnerJoinHavingWithNameSpecificationImpl(previousStep, using, stepName)
    }

}

interface InnerJoinHavingSpecification<INPUT, KEY, OTHER_INPUT> {

    /**
     * Specifies how to extract the correlation key from the left side.
     */
    fun having(block: (CorrelationRecord<OTHER_INPUT>) -> KEY?): InnerJoinStepSpecification<INPUT, Pair<INPUT, OTHER_INPUT>>
}

internal class InnerJoinHavingSpecificationImpl<INPUT, KEY, OTHER_INPUT>(
    private val previousStep: StepSpecification<*, INPUT, *>,
    private val using: (CorrelationRecord<INPUT>) -> Any?,
    private val on: (scenario: ScenarioSpecification) -> StepSpecification<*, OTHER_INPUT, *>
) : InnerJoinHavingSpecification<INPUT, KEY, OTHER_INPUT> {

    override fun having(block: (CorrelationRecord<OTHER_INPUT>) -> KEY?): InnerJoinStepSpecification<INPUT, Pair<INPUT, OTHER_INPUT>> {
        return previousStep.innerJoin(using, on, block)
    }
}


internal class InnerJoinHavingWithNameSpecificationImpl<INPUT, KEY, OTHER_INPUT>(
    private val previousStep: StepSpecification<*, INPUT, *>,
    private val using: (CorrelationRecord<INPUT>) -> Any?,
    private val on: String
) : InnerJoinHavingSpecification<INPUT, KEY, OTHER_INPUT> {

    override fun having(block: (CorrelationRecord<OTHER_INPUT>) -> KEY?): InnerJoinStepSpecification<INPUT, Pair<INPUT, OTHER_INPUT>> {
        return previousStep.innerJoin(using, on, block)
    }
}

/**
 * Initiates the creation of an inner join step.
 *
 * @author Eric Jessé
 */
fun <INPUT> StepSpecification<*, INPUT, *>.innerJoin(): InnerJoinUsingSpecification<INPUT> {
    return InnerJoinUsingSpecificationImpl(this)
}