/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.manager.syslog;

import jakarta.persistence.TypedQuery;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import org.openremote.container.persistence.PersistenceService;
import org.openremote.container.timer.TimerService;
import org.openremote.container.util.MapAccess;
import org.openremote.manager.event.ClientEventService;
import org.openremote.manager.syslog.SyslogResourceImpl;
import org.openremote.manager.web.ManagerWebService;
import org.openremote.model.Container;
import org.openremote.model.ContainerService;
import org.openremote.model.syslog.SyslogCategory;
import org.openremote.model.syslog.SyslogConfig;
import org.openremote.model.syslog.SyslogEvent;
import org.openremote.model.syslog.SyslogLevel;
import org.openremote.model.util.Pair;

public class SyslogService
extends Handler
implements ContainerService {
    public static final String OR_SYSLOG_LOG_LEVEL = "OR_SYSLOG_LOG_LEVEL";
    public static final SyslogLevel OR_SYSLOG_LOG_LEVEL_DEFAULT = SyslogLevel.INFO;
    public static final String OR_SYSLOG_MAX_AGE_DAYS = "OR_SYSLOG_MAX_AGE_DAYS";
    public static final int OR_SYSLOG_MAX_AGE_DAYS_DEFAULT = 5;
    private static final Logger LOG = Logger.getLogger(SyslogService.class.getName());
    protected ScheduledExecutorService scheduledExecutorService;
    protected PersistenceService persistenceService;
    protected ClientEventService clientEventService;
    protected SyslogConfig config;
    protected final List<SyslogEvent> batch = new ArrayList<SyslogEvent>();
    protected ScheduledFuture flushBatchFuture;
    protected ScheduledFuture deleteOldFuture;

    public int getPriority() {
        return 1000;
    }

    public void init(Container container) throws Exception {
        this.scheduledExecutorService = container.getScheduledExecutor();
        if (container.hasService(ClientEventService.class) && container.hasService(PersistenceService.class)) {
            LOG.info("Syslog service enabled");
            this.clientEventService = (ClientEventService)container.getService(ClientEventService.class);
            this.persistenceService = (PersistenceService)container.getService(PersistenceService.class);
        } else {
            LOG.info("Syslog service disabled, missing required services");
        }
        if (this.clientEventService != null) {
            this.clientEventService.addSubscriptionAuthorizer((realm, auth, subscription) -> subscription.isEventType(SyslogEvent.class) && auth != null && (auth.isSuperUser() || auth.hasResourceRole("read:logs", "openremote")));
        }
        if (container.hasService(ManagerWebService.class)) {
            ((ManagerWebService)container.getService(ManagerWebService.class)).addApiSingleton((Object)new SyslogResourceImpl(this));
        }
        int maxAgeDays = MapAccess.getInteger((Map)container.getConfig(), (String)OR_SYSLOG_MAX_AGE_DAYS, (int)5);
        SyslogLevel syslogLevel = Optional.ofNullable(MapAccess.getString((Map)container.getConfig(), (String)OR_SYSLOG_LOG_LEVEL, null)).map(SyslogLevel::valueOf).orElse(OR_SYSLOG_LOG_LEVEL_DEFAULT);
        this.config = new SyslogConfig(syslogLevel, SyslogCategory.values(), maxAgeDays * 24 * 60);
    }

    public void start(Container container) throws Exception {
        if (this.persistenceService != null) {
            TimerService timerService = (TimerService)container.getService(TimerService.class);
            this.flushBatchFuture = this.scheduledExecutorService.scheduleAtFixedRate(this::flushBatch, 10L, 3L, TimeUnit.SECONDS);
            if (this.config.getStoredMaxAgeMinutes() > 0) {
                long initialDelay = ChronoUnit.MILLIS.between(timerService.getNow(), timerService.getNow().truncatedTo(ChronoUnit.DAYS).plus(27L, ChronoUnit.HOURS));
                this.deleteOldFuture = this.scheduledExecutorService.scheduleAtFixedRate(this::purgeSyslog, initialDelay, Duration.ofDays(1L).toMillis(), TimeUnit.MILLISECONDS);
            }
        }
    }

    protected void purgeSyslog() {
        try {
            this.persistenceService.doTransaction(em -> em.createQuery("delete from SyslogEvent e where e.timestamp <= :date").setParameter("date", (Object)Date.from(Instant.now().minus(this.config.getStoredMaxAgeMinutes(), ChronoUnit.MINUTES))).executeUpdate());
        }
        catch (Throwable e) {
            LOG.log(Level.WARNING, "Failed to purge syslog events", e);
        }
    }

    public void stop(Container container) throws Exception {
        if (this.flushBatchFuture != null) {
            this.flushBatchFuture.cancel(true);
            this.flushBatchFuture = null;
        }
        if (this.deleteOldFuture != null) {
            this.deleteOldFuture.cancel(true);
            this.deleteOldFuture = null;
        }
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() throws SecurityException {
    }

    @Override
    public void publish(LogRecord record) {
        SyslogEvent syslogEvent = SyslogCategory.mapSyslogEvent((LogRecord)record);
        if (syslogEvent != null) {
            try {
                this.store(syslogEvent);
            }
            catch (Exception e) {
                LOG.log(Level.SEVERE, "Failed to store syslog event", e);
            }
            try {
                if (this.clientEventService != null) {
                    this.clientEventService.publishEvent(syslogEvent);
                }
            }
            catch (Exception e) {
                LOG.log(Level.SEVERE, "Failed to send syslog event to subscribed clients", e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setConfig(SyslogConfig config) {
        List<SyslogEvent> list = this.batch;
        synchronized (list) {
            LOG.info("Using: " + String.valueOf(config));
            this.config = config;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SyslogConfig getConfig() {
        List<SyslogEvent> list = this.batch;
        synchronized (list) {
            return this.config;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearStoredEvents() {
        if (this.persistenceService == null) {
            return;
        }
        List<SyslogEvent> list = this.batch;
        synchronized (list) {
            this.persistenceService.doTransaction(em -> em.createQuery("delete from SyslogEvent e").executeUpdate());
        }
    }

    public Pair<Long, List<SyslogEvent>> getEvents(SyslogLevel level, int perPage, int page, Instant from, Instant to, List<SyslogCategory> categories, List<String> subCategories) {
        if (this.persistenceService == null) {
            return null;
        }
        if (to == null) {
            to = Instant.now();
        }
        if (from == null) {
            from = to.minus(1L, ChronoUnit.HOURS);
        }
        Date fromDate = Date.from(from);
        Date toDate = Date.from(to);
        AtomicLong count = new AtomicLong();
        List events = (List)this.persistenceService.doReturningTransaction(em -> {
            StringBuilder sb = new StringBuilder("from SyslogEvent e where e.timestamp >= :from and e.timestamp <= :to");
            if (level != null) {
                sb.append(" and e.level >= :level");
            }
            if (categories != null && !categories.isEmpty()) {
                sb.append(" and e.category in :categories");
            }
            if (subCategories != null && !subCategories.isEmpty()) {
                sb.append(" and e.subCategory in :subCategories");
            }
            TypedQuery countQuery = em.createQuery("select count(e.id) " + sb.toString(), Long.class);
            countQuery.setParameter("from", (Object)fromDate);
            countQuery.setParameter("to", (Object)toDate);
            if (level != null) {
                countQuery.setParameter("level", (Object)level);
            }
            if (categories != null && !categories.isEmpty()) {
                countQuery.setParameter("categories", (Object)categories);
            }
            if (subCategories != null && !subCategories.isEmpty()) {
                countQuery.setParameter("subCategories", (Object)subCategories);
            }
            count.set((Long)countQuery.getSingleResult());
            if (count.get() == 0L) {
                return Collections.emptyList();
            }
            sb.append(" order by e.timestamp desc");
            TypedQuery query = em.createQuery("select e " + sb.toString(), SyslogEvent.class);
            query.setParameter("from", (Object)fromDate);
            query.setParameter("to", (Object)toDate);
            if (level != null) {
                query.setParameter("level", (Object)level);
            }
            if (categories != null && !categories.isEmpty()) {
                query.setParameter("categories", (Object)categories);
            }
            if (subCategories != null && !subCategories.isEmpty()) {
                query.setParameter("subCategories", (Object)subCategories);
            }
            query.setMaxResults(perPage);
            if (page > 1) {
                query.setFirstResult((page - 1) * perPage + 1);
            }
            return query.getResultList();
        });
        return new Pair((Object)count.get(), (Object)events);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void store(SyslogEvent syslogEvent) {
        boolean isLoggable;
        if (this.persistenceService == null) {
            return;
        }
        if (this.persistenceService.getEntityManagerFactory() == null) {
            return;
        }
        boolean bl = isLoggable = this.config.getStoredLevel().isLoggable(syslogEvent) && Arrays.asList(this.config.getStoredCategories()).contains(syslogEvent.getCategory());
        if (isLoggable) {
            List<SyslogEvent> list = this.batch;
            synchronized (list) {
                this.batch.add(syslogEvent);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void flushBatch() {
        if (this.persistenceService == null) {
            return;
        }
        List<SyslogEvent> list = this.batch;
        synchronized (list) {
            ArrayList<SyslogEvent> transientEvents = new ArrayList<SyslogEvent>(this.batch);
            this.batch.clear();
            if (transientEvents.size() == 0) {
                return;
            }
            LOG.finest("Flushing syslog batch: " + transientEvents.size());
            try {
                this.persistenceService.doTransaction(em -> {
                    try {
                        for (SyslogEvent e : transientEvents) {
                            em.persist((Object)e);
                        }
                    }
                    catch (RuntimeException ex) {
                        LOG.info("Error flushing syslog to database, some events are lost: " + String.valueOf(ex));
                    }
                    finally {
                        em.flush();
                    }
                });
            }
            catch (Exception e) {
                LOG.log(Level.SEVERE, "Exception occurred whilst flushing the syslog", e);
            }
        }
    }

    public String toString() {
        return this.getClass().getSimpleName() + "{}";
    }
}

