/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.manager.notification;

import jakarta.persistence.Query;
import jakarta.persistence.TypedQuery;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.model.ThreadsDefinition;
import org.openremote.container.message.MessageBrokerService;
import org.openremote.container.persistence.PersistenceService;
import org.openremote.container.security.AuthContext;
import org.openremote.container.timer.TimerService;
import org.openremote.manager.asset.AssetStorageService;
import org.openremote.manager.notification.NotificationHandler;
import org.openremote.manager.notification.NotificationProcessingException;
import org.openremote.manager.notification.NotificationResourceImpl;
import org.openremote.manager.security.ManagerIdentityService;
import org.openremote.manager.web.ManagerWebService;
import org.openremote.model.Container;
import org.openremote.model.ContainerService;
import org.openremote.model.asset.Asset;
import org.openremote.model.notification.Notification;
import org.openremote.model.notification.NotificationSendResult;
import org.openremote.model.notification.RepeatFrequency;
import org.openremote.model.notification.SentNotification;
import org.openremote.model.query.UserQuery;
import org.openremote.model.query.filter.StringPredicate;
import org.openremote.model.util.TextUtil;
import org.openremote.model.util.TimeUtil;

public class NotificationService
extends RouteBuilder
implements ContainerService {
    public static final String NOTIFICATION_QUEUE = "direct://NotificationQueue";
    private static final Logger LOG = Logger.getLogger(NotificationService.class.getName());
    protected TimerService timerService;
    protected PersistenceService persistenceService;
    protected AssetStorageService assetStorageService;
    protected ManagerIdentityService identityService;
    protected MessageBrokerService messageBrokerService;
    protected ExecutorService executorService;
    protected Map<String, NotificationHandler> notificationHandlerMap = new HashMap<String, NotificationHandler>();

    public int getPriority() {
        return 1000;
    }

    public void init(Container container) throws Exception {
        this.timerService = (TimerService)container.getService(TimerService.class);
        this.persistenceService = (PersistenceService)container.getService(PersistenceService.class);
        this.assetStorageService = (AssetStorageService)container.getService(AssetStorageService.class);
        this.identityService = (ManagerIdentityService)container.getService(ManagerIdentityService.class);
        this.messageBrokerService = (MessageBrokerService)container.getService(MessageBrokerService.class);
        this.executorService = container.getExecutor();
        ((MessageBrokerService)container.getService(MessageBrokerService.class)).getContext().addRoutes((RoutesBuilder)this);
        container.getServices(NotificationHandler.class).forEach(notificationHandler -> this.notificationHandlerMap.put(notificationHandler.getTypeName(), (NotificationHandler)notificationHandler));
        ((ManagerWebService)container.getService(ManagerWebService.class)).addApiSingleton((Object)new NotificationResourceImpl(this, (MessageBrokerService)container.getService(MessageBrokerService.class), (AssetStorageService)container.getService(AssetStorageService.class), (ManagerIdentityService)container.getService(ManagerIdentityService.class)));
    }

    public void start(Container container) throws Exception {
    }

    public void stop(Container container) throws Exception {
    }

    public void configure() throws Exception {
        ((ThreadsDefinition)this.from(NOTIFICATION_QUEUE).routeId("NotificationQueue").threads().executorService(this.executorService).process(exchange -> {
            Notification notification = (Notification)exchange.getIn().getBody(Notification.class);
            if (notification == null) {
                throw new NotificationProcessingException(NotificationProcessingException.Reason.MISSING_NOTIFICATION, "Notification must be set");
            }
            LOG.finest("Processing: " + notification.getName());
            if (notification.getMessage() == null) {
                throw new NotificationProcessingException(NotificationProcessingException.Reason.MISSING_CONTENT, "Notification message must be set");
            }
            Notification.Source source = (Notification.Source)exchange.getIn().getHeader(Notification.HEADER_SOURCE, () -> null, Notification.Source.class);
            if (source == null) {
                throw new NotificationProcessingException(NotificationProcessingException.Reason.MISSING_SOURCE);
            }
            NotificationHandler handler = this.notificationHandlerMap.get(notification.getMessage().getType());
            if (handler == null) {
                throw new NotificationProcessingException(NotificationProcessingException.Reason.UNSUPPORTED_MESSAGE_TYPE, "No handler for message type: " + notification.getMessage().getType());
            }
            if (!handler.isValid()) {
                throw new NotificationProcessingException(NotificationProcessingException.Reason.NOTIFICATION_HANDLER_CONFIG_ERROR, "Handler is not valid: " + handler.getTypeName());
            }
            if (!handler.isMessageValid(notification.getMessage())) {
                throw new NotificationProcessingException(NotificationProcessingException.Reason.INVALID_MESSAGE);
            }
            String realm = null;
            String userId = null;
            String assetId = null;
            AtomicReference<String> sourceId = new AtomicReference<String>("");
            boolean isSuperUser = false;
            boolean isRestrictedUser = false;
            switch (source) {
                case INTERNAL: {
                    isSuperUser = true;
                    break;
                }
                case CLIENT: {
                    AuthContext authContext = (AuthContext)exchange.getIn().getHeader("AUTH_CONTEXT", AuthContext.class);
                    if (authContext == null) {
                        throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS);
                    }
                    realm = authContext.getAuthenticatedRealmName();
                    userId = authContext.getUserId();
                    sourceId.set(userId);
                    isSuperUser = authContext.isSuperUser();
                    isRestrictedUser = this.identityService.getIdentityProvider().isRestrictedUser(authContext);
                    break;
                }
                case GLOBAL_RULESET: {
                    isSuperUser = true;
                    break;
                }
                case REALM_RULESET: {
                    realm = (String)exchange.getIn().getHeader(Notification.HEADER_SOURCE_ID, String.class);
                    sourceId.set(realm);
                    break;
                }
                case ASSET_RULESET: {
                    assetId = (String)exchange.getIn().getHeader(Notification.HEADER_SOURCE_ID, String.class);
                    sourceId.set(assetId);
                    Asset<?> asset = this.assetStorageService.find(assetId, false);
                    realm = asset.getRealm();
                }
            }
            LOG.fine("Sending " + notification.getMessage().getType() + " notification '" + notification.getName() + "': '" + String.valueOf(source) + ":" + sourceId.get() + "' -> " + String.valueOf(notification.getTargets()));
            this.checkAccess(source, sourceId.get(), notification.getTargets(), realm, userId, isSuperUser, isRestrictedUser, assetId);
            List<Object> mappedTargetsList = handler.getTargets(source, sourceId.get(), notification.getTargets(), notification.getMessage());
            if (mappedTargetsList == null || mappedTargetsList.isEmpty()) {
                throw new NotificationProcessingException(NotificationProcessingException.Reason.MISSING_TARGETS, "Notification targets must be set");
            }
            if (LOG.isLoggable(Level.FINER)) {
                LOG.finer("Notification targets mapped from: [" + (notification.getTargets() != null ? notification.getTargets().stream().map(Object::toString).collect(Collectors.joining(",")) : "null") + "to: [" + mappedTargetsList.stream().map(Object::toString).collect(Collectors.joining(",")) + "]");
            }
            if (!(TextUtil.isNullOrEmpty((String)notification.getName()) || TextUtil.isNullOrEmpty((String)notification.getRepeatInterval()) && notification.getRepeatFrequency() == null)) {
                mappedTargetsList = mappedTargetsList.stream().filter(target -> this.okToSendNotification(source, (String)sourceId.get(), (Notification.Target)target, notification)).collect(Collectors.toList());
            }
            AtomicReference error = new AtomicReference();
            mappedTargetsList.forEach(target -> {
                Exception notificationError = (Exception)this.persistenceService.doReturningTransaction(em -> {
                    SentNotification sentNotification = new SentNotification().setName(notification.getName()).setType(notification.getMessage().getType()).setSource(source).setSourceId((String)sourceId.get()).setTarget(target.getType()).setTargetId(target.getId()).setMessage(notification.getMessage()).setSentOn(Date.from(this.timerService.getNow()));
                    sentNotification = (SentNotification)em.merge((Object)sentNotification);
                    long id = sentNotification.getId();
                    try {
                        handler.sendMessage(id, source, (String)sourceId.get(), (Notification.Target)target, notification.getMessage());
                        NotificationSendResult result = NotificationSendResult.success();
                        LOG.fine("Notification sent '" + id + "': " + String.valueOf(target));
                        sentNotification.setMessage(notification.getMessage());
                    }
                    catch (Exception e) {
                        NotificationProcessingException notificationProcessingException = e instanceof NotificationProcessingException ? (NotificationProcessingException)e : new NotificationProcessingException(NotificationProcessingException.Reason.SEND_FAILURE, e.getMessage());
                        LOG.warning("Notification failed '" + id + "': " + String.valueOf(target) + ", reason=" + String.valueOf(notificationProcessingException));
                        sentNotification.setError(TextUtil.isNullOrEmpty((String)notificationProcessingException.getMessage()) ? "Unknown error" : notificationProcessingException.getMessage());
                        NotificationProcessingException notificationProcessingException2 = notificationProcessingException;
                        return notificationProcessingException2;
                    }
                    finally {
                        em.merge((Object)sentNotification);
                    }
                    return null;
                });
                if (notificationError != null && error.get() == null) {
                    error.set(notificationError);
                }
            });
            exchange.getMessage().setBody((Object)(error.get() == null ? 1 : 0));
            if (error.get() != null) {
                throw (Exception)error.get();
            }
        })).onException(Exception.class).logStackTrace(false).handled(true).process(exchange -> exchange.getMessage().setBody((Object)false));
    }

    public boolean sendNotification(Notification notification) {
        return this.sendNotification(notification, Notification.Source.INTERNAL, "");
    }

    public void sendNotificationAsync(Notification notification, Notification.Source source, String sourceId) {
        Map headers = Map.ofEntries(Map.entry(Notification.HEADER_SOURCE, source), Map.entry(Notification.HEADER_SOURCE_ID, sourceId));
        this.messageBrokerService.getFluentProducerTemplate().withBody((Object)notification).withHeaders(headers).to(NOTIFICATION_QUEUE).send();
    }

    public boolean sendNotification(Notification notification, Notification.Source source, String sourceId) {
        Map headers = Map.ofEntries(Map.entry(Notification.HEADER_SOURCE, source), Map.entry(Notification.HEADER_SOURCE_ID, sourceId));
        return (Boolean)this.messageBrokerService.getFluentProducerTemplate().withBody((Object)notification).withHeaders(headers).to(NOTIFICATION_QUEUE).request(Boolean.class);
    }

    public void setNotificationDelivered(long id) {
        this.setNotificationDelivered(id, this.timerService.getCurrentTimeMillis());
    }

    public void setNotificationDelivered(long id, long timestamp) {
        this.persistenceService.doTransaction(entityManager -> {
            Query query = entityManager.createQuery("UPDATE SentNotification SET deliveredOn=:timestamp WHERE id =:id");
            query.setParameter("id", (Object)id);
            query.setParameter("timestamp", (Object)new Date(timestamp));
            query.executeUpdate();
        });
    }

    public void setNotificationAcknowledged(long id, String acknowledgement) {
        this.setNotificationAcknowledged(id, acknowledgement, this.timerService.getCurrentTimeMillis());
    }

    public void setNotificationAcknowledged(long id, String acknowledgement, long timestamp) {
        this.persistenceService.doTransaction(entityManager -> {
            Query query = entityManager.createQuery("UPDATE SentNotification SET acknowledgedOn=:timestamp, acknowledgement=:acknowledgement WHERE id =:id");
            query.setParameter("id", (Object)id);
            query.setParameter("timestamp", (Object)new Date(timestamp));
            query.setParameter("acknowledgement", (Object)acknowledgement);
            query.executeUpdate();
        });
    }

    public SentNotification getSentNotification(Long notificationId) {
        return (SentNotification)this.persistenceService.doReturningTransaction(em -> (SentNotification)em.find(SentNotification.class, (Object)notificationId));
    }

    public List<SentNotification> getNotifications(List<Long> ids, List<String> types, Long fromTimestamp, Long toTimestamp, List<String> realmIds, List<String> userIds, List<String> assetIds) throws IllegalArgumentException {
        StringBuilder builder = new StringBuilder();
        builder.append("select n from SentNotification n where 1=1");
        ArrayList<Object> parameters = new ArrayList<Object>();
        this.processCriteria(builder, parameters, ids, types, fromTimestamp, toTimestamp, realmIds, userIds, assetIds, false);
        builder.append(" order by n.sentOn asc");
        return (List)this.persistenceService.doReturningTransaction(entityManager -> {
            TypedQuery query = entityManager.createQuery(builder.toString(), SentNotification.class);
            IntStream.rangeClosed(1, parameters.size()).forEach(i -> query.setParameter(i, parameters.get(i - 1)));
            return query.getResultList();
        });
    }

    public void removeNotification(Long id) {
        this.persistenceService.doTransaction(entityManager -> entityManager.createQuery("delete SentNotification where id = :id").setParameter("id", (Object)id).executeUpdate());
    }

    public void removeNotifications(List<Long> ids, List<String> types, Long fromTimestamp, Long toTimestamp, List<String> realmIds, List<String> userIds, List<String> assetIds) throws IllegalArgumentException {
        StringBuilder builder = new StringBuilder();
        builder.append("delete from SentNotification n where 1=1");
        ArrayList<Object> parameters = new ArrayList<Object>();
        this.processCriteria(builder, parameters, ids, types, fromTimestamp, toTimestamp, realmIds, userIds, assetIds, true);
        this.persistenceService.doTransaction(entityManager -> {
            Query query = entityManager.createQuery(builder.toString());
            IntStream.rangeClosed(1, parameters.size()).forEach(i -> query.setParameter(i, parameters.get(i - 1)));
            query.executeUpdate();
        });
    }

    protected void processCriteria(StringBuilder builder, List<Object> parameters, List<Long> ids, List<String> types, Long fromTimestamp, Long toTimestamp, List<String> realmIds, List<String> userIds, List<String> assetIds, boolean isRemove) {
        boolean hasIds = ids != null && !ids.isEmpty();
        boolean hasTypes = types != null && !types.isEmpty();
        boolean hasRealms = realmIds != null && !realmIds.isEmpty();
        boolean hasUsers = userIds != null && !userIds.isEmpty();
        boolean hasAssets = assetIds != null && !assetIds.isEmpty();
        int counter = 0;
        if (hasIds) {
            ++counter;
        }
        if (hasTypes) {
            ++counter;
        }
        if (hasRealms) {
            ++counter;
        }
        if (hasUsers) {
            ++counter;
        }
        if (hasAssets) {
            ++counter;
        }
        if (isRemove && fromTimestamp == null && toTimestamp == null && counter == 0) {
            LOG.fine("No filters set for remove notifications request so not allowed");
            throw new IllegalArgumentException("No criteria specified");
        }
        if (hasIds) {
            builder.append(" AND n.id IN ?").append(parameters.size() + 1);
            parameters.add(ids);
            return;
        }
        if (hasTypes) {
            builder.append(" AND n.type IN ?").append(parameters.size() + 1);
            parameters.add(types);
        }
        if (fromTimestamp != null) {
            builder.append(" AND n.sentOn >= ?").append(parameters.size() + 1);
            parameters.add(new Date(fromTimestamp));
        }
        if (toTimestamp != null) {
            builder.append(" AND n.sentOn <= ?").append(parameters.size() + 1);
            parameters.add(new Date(toTimestamp));
        }
        if (hasAssets) {
            builder.append(" AND n.target = ?").append(parameters.size() + 1).append(" AND n.targetId IN ?").append(parameters.size() + 2);
            parameters.add(Notification.TargetType.ASSET);
            parameters.add(assetIds);
        } else if (hasUsers) {
            builder.append(" AND n.target = ?").append(parameters.size() + 1).append(" AND n.targetId IN ?").append(parameters.size() + 2);
            parameters.add(Notification.TargetType.USER);
            parameters.add(userIds);
        } else if (hasRealms) {
            builder.append(" AND n.target = ?").append(parameters.size() + 1).append(" AND n.targetId IN ?").append(parameters.size() + 2);
            parameters.add(Notification.TargetType.REALM);
            parameters.add(realmIds);
        }
    }

    protected Instant getRepeatAfterTimestamp(Notification notification, Instant lastSend) {
        Instant timestamp = null;
        if (TextUtil.isNullOrEmpty((String)notification.getName())) {
            return null;
        }
        if (notification.getRepeatFrequency() != null) {
            switch (notification.getRepeatFrequency()) {
                case HOURLY: {
                    timestamp = lastSend.truncatedTo(ChronoUnit.HOURS).plus(1L, ChronoUnit.HOURS);
                    break;
                }
                case DAILY: {
                    timestamp = lastSend.truncatedTo(ChronoUnit.DAYS).plus(1L, ChronoUnit.DAYS);
                    break;
                }
                case WEEKLY: {
                    timestamp = lastSend.truncatedTo(ChronoUnit.WEEKS).plus(1L, ChronoUnit.WEEKS);
                    break;
                }
                case MONTHLY: {
                    timestamp = lastSend.truncatedTo(ChronoUnit.MONTHS).plus(1L, ChronoUnit.MONTHS);
                    break;
                }
                case ANNUALLY: {
                    timestamp = lastSend.truncatedTo(ChronoUnit.YEARS).plus(1L, ChronoUnit.YEARS);
                }
            }
        } else if (!TextUtil.isNullOrEmpty((String)notification.getRepeatInterval())) {
            timestamp = lastSend.plus(TimeUtil.parseTimeDuration((String)notification.getRepeatInterval()), ChronoUnit.MILLIS);
        }
        return timestamp;
    }

    protected void checkAccess(Notification.Source source, String sourceId, List<Notification.Target> targets, String realm, String userId, boolean isSuperUser, boolean isRestrictedUser, String assetId) throws NotificationProcessingException {
        if (isSuperUser) {
            return;
        }
        if (targets == null || targets.isEmpty()) {
            return;
        }
        targets.forEach(target -> {
            switch (target.getType()) {
                case REALM: {
                    if (source == Notification.Source.CLIENT || source == Notification.Source.ASSET_RULESET) {
                        throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS);
                    }
                }
                case USER: {
                    if (TextUtil.isNullOrEmpty((String)realm) || isRestrictedUser) {
                        throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS);
                    }
                    boolean realmMatch = target.getType() == Notification.TargetType.USER ? Arrays.stream(this.identityService.getIdentityProvider().queryUsers(new UserQuery().ids(new String[]{target.getId()}).serviceUsers(Boolean.valueOf(false)).attributes(new UserQuery.AttributeValuePredicate[]{new UserQuery.AttributeValuePredicate(true, new StringPredicate("systemAccount"))}))).allMatch(user -> realm.equals(user.getRealm())) : realm.equals(target.getId());
                    if (realmMatch) break;
                    throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS, "Targets must all be in the same realm as the requester");
                }
                case ASSET: {
                    if (TextUtil.isNullOrEmpty((String)realm)) {
                        throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS);
                    }
                    if (isRestrictedUser && !this.assetStorageService.isUserAssets(userId, Collections.singletonList(target.getId()))) {
                        throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS, "Targets must all be linked to the requesting restricted user");
                    }
                    if (!this.assetStorageService.isRealmAssets(realm, Collections.singletonList(target.getId()))) {
                        throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS, "Targets must all be in the same realm as the requestor");
                    }
                    if (TextUtil.isNullOrEmpty((String)assetId) || this.assetStorageService.isDescendantAssets(assetId, Collections.singletonList(target.getId()))) break;
                    throw new NotificationProcessingException(NotificationProcessingException.Reason.INSUFFICIENT_ACCESS, "Targets must all be descendants of the requesting asset");
                }
            }
        });
    }

    protected boolean okToSendNotification(Notification.Source source, String sourceId, Notification.Target target, Notification notification) {
        if (notification.getRepeatFrequency() == RepeatFrequency.ALWAYS) {
            return true;
        }
        Date lastSend = ((List)this.persistenceService.doReturningTransaction(entityManager -> entityManager.createQuery("SELECT n.sentOn FROM SentNotification n WHERE n.source =:source AND n.sourceId =:sourceId AND n.target =:target AND n.targetId =:targetId AND n.name =:name ORDER BY n.sentOn DESC", Date.class).setParameter("source", (Object)source).setParameter("sourceId", (Object)sourceId).setParameter("target", (Object)target.getType()).setParameter("targetId", (Object)target.getId()).setParameter("name", (Object)notification.getName()).setMaxResults(1).getResultList())).stream().findFirst().orElse(null);
        return lastSend == null || notification.getRepeatFrequency() != RepeatFrequency.ONCE && this.timerService.getNow().plusSeconds(1L).isAfter(this.getRepeatAfterTimestamp(notification, lastSend.toInstant()));
    }
}

