/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.manager.energy;

import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.openremote.manager.energy.EnergyOptimisationService;
import org.openremote.model.util.Pair;

public class EnergyOptimiser {
    protected double intervalSize;
    protected double financialWeighting;

    public EnergyOptimiser(double intervalSize, double financialWeighting) throws IllegalArgumentException {
        if (24.0 / intervalSize != (double)((int)(24.0 / intervalSize))) {
            throw new IllegalArgumentException("24 divided by intervalSizeHours must be whole number");
        }
        this.intervalSize = intervalSize;
        this.financialWeighting = Math.max(0.0, Math.min(1.0, financialWeighting));
    }

    public double getIntervalSize() {
        return this.intervalSize;
    }

    public double getFinancialWeighting() {
        return this.financialWeighting;
    }

    public int get24HourIntervalCount() {
        return (int)(24.0 / this.intervalSize);
    }

    public void applyEnergySchedule(double[] energyLevelMins, double[] energyLevelMaxs, double energyCapacity, int[][] energyLevelSchedule, LocalDateTime currentTime) {
        int i;
        if (energyLevelSchedule == null) {
            return;
        }
        OffsetDateTime date = currentTime.plus(1L, ChronoUnit.HOURS).atOffset(ZoneOffset.UTC);
        int dayIndex = date.getDayOfWeek().getValue();
        int hourIndex = date.get(ChronoField.HOUR_OF_DAY);
        double[] schedule = new double[24];
        for (i = 0; i < 24; ++i) {
            schedule[i] = energyCapacity * (double)energyLevelSchedule[dayIndex][hourIndex] * 0.01;
            if (++hourIndex <= 23) continue;
            hourIndex = 0;
            dayIndex = (dayIndex + 1) % 7;
        }
        if (this.intervalSize <= 1.0) {
            int hourIntervals = (int)(1.0 / this.intervalSize);
            for (i = 0; i < schedule.length; ++i) {
                energyLevelMins[hourIntervals * i] = Math.min(energyLevelMaxs[hourIntervals * i], Math.max(energyLevelMins[hourIntervals * i], schedule[i]));
            }
        } else {
            int takeSize = (int)this.intervalSize;
            int hourIntervals = (int)(24.0 / this.intervalSize);
            for (i = 0; i < hourIntervals; ++i) {
                energyLevelMins[i] = Math.min(energyLevelMaxs[i], Math.max(energyLevelMins[i], Arrays.stream(schedule, i * takeSize, i * takeSize + takeSize).max().orElse(0.0)));
            }
        }
    }

    public void normaliseEnergyMinRequirements(double[] energyLevelMins, Function<Integer, Double> powerImportMaxCalculator, Function<Integer, Double> powerExportMaxCalculator, double energyLevel) {
        int intervalCount = this.get24HourIntervalCount();
        Function<Integer, Double> previousEnergyLevelCalculator = i -> i == 0 ? energyLevel : energyLevelMins[i - 1];
        IntStream.range(0, intervalCount).forEach(i -> {
            block4: {
                double min;
                double energyDelta;
                block5: {
                    energyDelta = energyLevelMins[i] - (Double)previousEnergyLevelCalculator.apply(i);
                    if (!(energyDelta > 0.0)) break block5;
                    for (int j = i; j >= 0; --j) {
                        double previousMin = energyLevelMins[j] - (Double)powerImportMaxCalculator.apply(j) * this.intervalSize;
                        double previous = (Double)previousEnergyLevelCalculator.apply(j);
                        if (!(previous < previousMin)) break block4;
                        if (j == 0) {
                            double shift = previous - previousMin;
                            int k = 0;
                            while (k <= i) {
                                int n = k++;
                                energyLevelMins[n] = energyLevelMins[n] + shift;
                            }
                            continue;
                        }
                        energyLevelMins[j - 1] = previousMin;
                    }
                    break block4;
                }
                if (!(energyDelta < 0.0)) break block4;
                for (int j = i; j < intervalCount && (min = (Double)previousEnergyLevelCalculator.apply(j) + (Double)powerExportMaxCalculator.apply(j) * this.intervalSize) > energyLevelMins[j]; ++j) {
                    energyLevelMins[j] = min;
                }
            }
        });
    }

    public void applyEnergyMinImports(double[][] importCostAndPower, double[] energyLevelMins, double[] powerSetpoints, Function<Integer, Double> energyLevelCalculator, BiFunction<Integer, double[], double[]> importOptimiser, Function<Integer, Double> powerImportMaxCalculator) {
        AtomicInteger fromInterval = new AtomicInteger(0);
        IntStream.range(0, this.get24HourIntervalCount()).forEach(i -> {
            double intervalEnergyLevel = (Double)energyLevelCalculator.apply(i);
            double energyDeficit = energyLevelMins[i] - intervalEnergyLevel;
            if (energyDeficit > 0.0) {
                double energyAttainable = (Double)powerImportMaxCalculator.apply(i) * this.intervalSize;
                energyAttainable = Math.min(energyDeficit, energyAttainable);
                powerSetpoints[i] = energyAttainable / this.intervalSize;
                if ((energyDeficit -= energyAttainable) > 0.0) {
                    this.retrospectiveEnergyAllocator(importCostAndPower, energyLevelMins, powerSetpoints, importOptimiser, powerImportMaxCalculator, energyDeficit, fromInterval.getAndSet(i), i);
                }
            }
        });
    }

    public void retrospectiveEnergyAllocator(double[][] importCostAndPower, double[] energyLevelMins, double[] powerSetpoints, BiFunction<Integer, double[], double[]> importOptimiser, Function<Integer, Double> powerImportMaxCalculator, double energyLevel, int fromInterval, int toInterval) {
        double importPower;
        boolean morePowerAvailable;
        double energyDeficit = energyLevelMins[toInterval] - energyLevel;
        if (energyDeficit <= 0.0) {
            return;
        }
        boolean canMeetDeficit = IntStream.range(fromInterval, toInterval).mapToDouble(i -> Math.min((Double)powerImportMaxCalculator.apply(i), importCostAndPower[i][2])).sum() >= energyDeficit;
        boolean bl = morePowerAvailable = !canMeetDeficit && IntStream.range(fromInterval, toInterval).mapToObj(i -> importCostAndPower[i][2] < (Double)powerImportMaxCalculator.apply(i)).anyMatch(b -> b);
        if (!canMeetDeficit && morePowerAvailable) {
            IntStream.range(fromInterval, toInterval).forEach(i -> {
                double powerImportMax = (Double)powerImportMaxCalculator.apply(i);
                if (importCostAndPower[i][2] < powerImportMax) {
                    importCostAndPower[i] = (double[])importOptimiser.apply(i, new double[]{0.0, powerImportMax});
                }
            });
        }
        List sortedImportCostAndPower = IntStream.range(fromInterval, toInterval).mapToObj(i -> new Pair((Object)i, (Object)importCostAndPower[i])).sorted(Comparator.comparingDouble(pair -> ((double[])pair.value)[0])).collect(Collectors.toList());
        for (int i2 = 0; energyDeficit > 0.0 && i2 < sortedImportCostAndPower.size(); energyDeficit -= importPower, ++i2) {
            importPower = Math.min(powerImportMaxCalculator.apply(i2), importCostAndPower[i2][2]);
            double requiredPower = energyDeficit / this.intervalSize;
            powerSetpoints[i2] = importPower = importCostAndPower[i2][0] < 0.0 ? importPower : Math.min(importPower, requiredPower);
        }
    }

    public void applyEarningOpportunities(double[][] importCostAndPower, double[][] exportCostAndPower, double[] energyLevelMins, double[] energyLevelMaxs, double[] powerSetpoints, Function<Integer, Double> energyLevelCalculator, Function<Integer, Double> powerImportMaxCalculator, Function<Integer, Double> powerExportMaxCalculator) {
        EnergyOptimisationService.LOG.finest("Applying earning opportunities");
        double[][] primary = importCostAndPower != null ? importCostAndPower : exportCostAndPower;
        double[][] secondary = (double[][])(importCostAndPower != null ? exportCostAndPower : null);
        List<Pair> earningOpportunities = IntStream.range(0, primary.length).mapToObj(i -> {
            if (secondary == null) {
                return new Pair((Object)i, (Object)primary[i]);
            }
            if (primary[i][0] < secondary[i][0]) {
                return new Pair((Object)i, (Object)primary[i]);
            }
            return new Pair((Object)i, (Object)secondary[i]);
        }).filter(intervalCostAndPowerBand -> ((double[])intervalCostAndPowerBand.value)[0] < 0.0).sorted(Comparator.comparingDouble(optimisedInterval -> ((double[])optimisedInterval.value)[0])).collect(Collectors.toList());
        if (earningOpportunities.isEmpty()) {
            EnergyOptimisationService.LOG.finest("No earning opportunities found");
        }
        if (EnergyOptimisationService.LOG.isLoggable(Level.FINEST)) {
            earningOpportunities.forEach(op -> EnergyOptimisationService.LOG.finest("Earning opportunity: interval=" + String.valueOf(op.key) + ", cost=" + ((double[])op.value)[0] + ", powerMin=" + ((double[])op.value)[1] + ", powerMax=" + ((double[])op.value)[2]));
        }
        for (Pair earningOpportunity : earningOpportunities) {
            int interval = (Integer)earningOpportunity.key;
            double[] costAndPower = (double[])earningOpportunity.value;
            assert (importCostAndPower != null);
            assert (exportCostAndPower != null);
            if (this.isImportOpportunity(costAndPower, powerSetpoints[interval], interval, powerImportMaxCalculator)) {
                this.applyImportOpportunity(importCostAndPower, exportCostAndPower, energyLevelMins, energyLevelMaxs, powerSetpoints, energyLevelCalculator, powerImportMaxCalculator, powerExportMaxCalculator, interval);
                continue;
            }
            if (!this.isExportOpportunity(costAndPower, powerSetpoints[interval], interval, powerExportMaxCalculator)) continue;
            this.applyExportOpportunity(importCostAndPower, exportCostAndPower, energyLevelMins, energyLevelMaxs, powerSetpoints, energyLevelCalculator, powerImportMaxCalculator, powerExportMaxCalculator, interval);
        }
    }

    protected boolean isImportOpportunity(double[] costAndPower, double powerSetpoint, int interval, Function<Integer, Double> powerImportMaxCalculator) {
        return costAndPower[2] > 0.0 && powerSetpoint >= 0.0 && powerSetpoint < Math.min(powerImportMaxCalculator.apply(interval), costAndPower[2]);
    }

    protected boolean isExportOpportunity(double[] costAndPower, double powerSetpoint, int interval, Function<Integer, Double> powerExportMaxCalculator) {
        return costAndPower[1] < 0.0 && powerSetpoint <= 0.0 && powerSetpoint > Math.max(powerExportMaxCalculator.apply(interval), costAndPower[1]);
    }

    public void applyImportOpportunity(double[][] importCostAndPower, double[][] exportCostAndPower, double[] energyLevelMins, double[] energyLevelMaxs, double[] powerSetpoints, Function<Integer, Double> energyLevelCalculator, Function<Integer, Double> powerImportMaxCalculator, Function<Integer, Double> powerExportMaxCalculator, int interval) {
        int k;
        EnergyOptimisationService.LOG.finest("Applying import earning opportunity: interval=" + interval);
        double[] costAndPower = importCostAndPower[interval];
        double impPowerMin = costAndPower[1];
        double impPowerMax = Math.min(powerImportMaxCalculator.apply(interval), costAndPower[2]);
        double powerCapacity = impPowerMax - powerSetpoints[interval];
        if (impPowerMin > powerCapacity) {
            EnergyOptimisationService.LOG.finest("Can't attain min power level to make use of this opportunity");
            return;
        }
        double energySpace = energyLevelMaxs[interval] - energyLevelCalculator.apply(interval);
        double energySpaceMax = powerCapacity * this.intervalSize;
        double energySpaceMin = impPowerMin * this.intervalSize;
        ArrayList<Pair> pastIntervalPowerDeltas = new ArrayList<Pair>();
        for (k = interval; k < powerSetpoints.length && energySpace > 0.0 && energySpace >= energySpaceMin; ++k) {
            double futureEnergySpace = energyLevelMaxs[k] - energyLevelCalculator.apply(k);
            energySpace = Math.min(energySpace, futureEnergySpace);
        }
        if (energySpace < energySpaceMax && exportCostAndPower != null) {
            EnergyOptimisationService.LOG.finest("Looking for earlier export opportunities to maximise on this import opportunity: space=" + energySpace + ", max=" + energySpaceMax);
            ArrayList<Pair> pastOpportunities = new ArrayList<Pair>();
            for (int i = interval - 1; i >= 0; --i) {
                if (!(costAndPower[0] + exportCostAndPower[i][0] < 0.0) || !(powerSetpoints[i] <= 0.0)) continue;
                pastOpportunities.add(new Pair((Object)i, (Object)exportCostAndPower[i][0]));
            }
            pastOpportunities.sort(Comparator.comparingDouble(op -> (Double)op.value));
            int j = 0;
            if (pastOpportunities.isEmpty()) {
                EnergyOptimisationService.LOG.finest("No earlier export opportunities identified");
            }
            while (energySpace < energySpaceMax && j < pastOpportunities.size()) {
                Pair opportunity = (Pair)pastOpportunities.get(j);
                int pastInterval = (Integer)opportunity.key;
                double[] pastCostAndPower = exportCostAndPower[pastInterval];
                double expPowerMax = Math.max(powerExportMaxCalculator.apply(pastInterval), pastCostAndPower[1]);
                double expPowerCapacity = expPowerMax - powerSetpoints[pastInterval];
                if (expPowerCapacity >= 0.0 || expPowerCapacity > pastCostAndPower[2]) {
                    EnergyOptimisationService.LOG.finest("Power capacity is outside optimum power band so cannot use this opportunity");
                    ++j;
                    continue;
                }
                double energySurplusMin = pastCostAndPower[2] * this.intervalSize;
                double energySurplus = energyLevelMins[pastInterval] - energyLevelCalculator.apply(pastInterval);
                energySurplus = Math.max(energySurplus, energySpace - energySpaceMax);
                for (k = pastInterval; k < powerSetpoints.length && energySurplus < 0.0 && energySurplus <= energySurplusMin; ++k) {
                    double futureEnergySurplus = energyLevelCalculator.apply(k) - energyLevelMins[k];
                    if (!((energySurplus = Math.max(energySurplus, -futureEnergySurplus)) <= 0.0)) continue;
                    EnergyOptimisationService.LOG.finest("Earlier export opportunity would violate future energy min level: interval=" + j + ", futureInterval=" + k);
                }
                if ((expPowerCapacity = Math.max(expPowerCapacity, energySurplus / this.intervalSize)) < 0.0 && expPowerCapacity < pastCostAndPower[2]) {
                    energySpace += -1.0 * expPowerCapacity * this.intervalSize;
                    pastIntervalPowerDeltas.add(new Pair((Object)pastInterval, (Object)expPowerCapacity));
                    EnergyOptimisationService.LOG.finest("Earlier export opportunity identified: interval=" + pastInterval + ", power=" + expPowerCapacity);
                }
                ++j;
            }
        }
        if (energySpace > 0.0 && energySpace >= energySpaceMin) {
            pastIntervalPowerDeltas.forEach(intervalAndDelta -> {
                int n = (Integer)intervalAndDelta.key;
                powerSetpoints[n] = powerSetpoints[n] + (Double)intervalAndDelta.value;
            });
            energySpaceMax = Math.min(energySpaceMax, energySpace);
            powerCapacity = Math.min(impPowerMax - powerSetpoints[interval], energySpaceMax / this.intervalSize);
            powerSetpoints[interval] = powerSetpoints[interval] + powerCapacity;
            EnergyOptimisationService.LOG.finest("Applied import earning opportunity: set point=" + powerSetpoints[interval] + " (delta: " + powerCapacity + ")");
        }
    }

    public void applyExportOpportunity(double[][] importCostAndPower, double[][] exportCostAndPower, double[] energyLevelMins, double[] energyLevelMaxs, double[] powerSetpoints, Function<Integer, Double> energyLevelCalculator, Function<Integer, Double> powerImportMaxCalculator, Function<Integer, Double> powerExportMaxCalculator, int interval) {
        int k;
        EnergyOptimisationService.LOG.finest("Applying export earning opportunity: interval=" + interval);
        double[] costAndPower = exportCostAndPower[interval];
        double expPowerMin = costAndPower[2];
        double expPowerMax = Math.max(powerExportMaxCalculator.apply(interval), costAndPower[1]);
        double powerCapacity = expPowerMax - powerSetpoints[interval];
        if (expPowerMin < powerCapacity) {
            EnergyOptimisationService.LOG.finest("Can't attain min power level to make use of this opportunity");
            return;
        }
        double energySurplus = energyLevelCalculator.apply(interval) - energyLevelMins[interval];
        double energySurplusMin = -1.0 * expPowerMin * this.intervalSize;
        double energySurplusMax = -1.0 * powerCapacity * this.intervalSize;
        ArrayList<Pair> pastAndFutureIntervalPowerDeltas = new ArrayList<Pair>();
        for (k = interval; k < powerSetpoints.length && energySurplus > 0.0 && energySurplus >= energySurplusMin; ++k) {
            double futureEnergySurplus = energyLevelCalculator.apply(k) - energyLevelMins[k];
            energySurplus = Math.min(energySurplus, futureEnergySurplus);
        }
        if (energySurplus < energySurplusMax && importCostAndPower != null) {
            EnergyOptimisationService.LOG.finest("Looking for earlier import opportunities to maximise on this export opportunity: surplus=" + energySurplus + ", max=" + energySurplusMax);
            ArrayList<Pair> pastOpportunities = new ArrayList<Pair>();
            for (int i = interval - 1; i >= 0; --i) {
                if (!(costAndPower[0] + importCostAndPower[i][0] < 0.0) || !(powerSetpoints[i] >= 0.0)) continue;
                pastOpportunities.add(new Pair((Object)i, (Object)importCostAndPower[i][0]));
            }
            pastOpportunities.sort(Comparator.comparingDouble(op -> (Double)op.value));
            int j = 0;
            if (pastOpportunities.isEmpty()) {
                EnergyOptimisationService.LOG.finest("No earlier import opportunities identified");
            }
            while (energySurplus < energySurplusMax && j < pastOpportunities.size()) {
                Pair opportunity = (Pair)pastOpportunities.get(j);
                int pastInterval = (Integer)opportunity.key;
                double[] pastCostAndPower = importCostAndPower[pastInterval];
                double impPowerMax = Math.min(powerImportMaxCalculator.apply(interval), pastCostAndPower[2]);
                double impPowerCapacity = impPowerMax - powerSetpoints[pastInterval];
                if (impPowerCapacity <= 0.0 || impPowerCapacity < pastCostAndPower[1]) {
                    EnergyOptimisationService.LOG.finest("Power capacity is outside optimum power band so cannot use this opportunity");
                    ++j;
                    continue;
                }
                double energySpaceMin = pastCostAndPower[1] * this.intervalSize;
                double energySpace = energyLevelMaxs[interval] - energyLevelCalculator.apply(pastInterval);
                energySpace = Math.max(energySpace, energySpace - energySurplusMax);
                for (k = pastInterval; k < powerSetpoints.length && energySpace > 0.0 && energySpace >= energySpaceMin; ++k) {
                    double futureEnergySpace = energyLevelMaxs[k] - energyLevelCalculator.apply(k);
                    if (!((energySpace = Math.min(energySpace, futureEnergySpace)) <= 0.0)) continue;
                    EnergyOptimisationService.LOG.finest("Earlier import opportunity would violate future energy max level: interval=" + j + ", futureInterval=" + k);
                }
                if ((impPowerCapacity = Math.min(impPowerCapacity, energySpace / this.intervalSize)) > 0.0 && impPowerCapacity > pastCostAndPower[1]) {
                    energySurplus += impPowerCapacity * this.intervalSize;
                    pastAndFutureIntervalPowerDeltas.add(new Pair((Object)pastInterval, (Object)impPowerCapacity));
                    EnergyOptimisationService.LOG.finest("Earlier import opportunity identified: interval=" + pastInterval + ", power=" + impPowerCapacity);
                }
                ++j;
            }
        }
        if (energySurplus > 0.0 && energySurplus >= energySurplusMin) {
            pastAndFutureIntervalPowerDeltas.forEach(intervalAndDelta -> {
                int n = (Integer)intervalAndDelta.key;
                powerSetpoints[n] = powerSetpoints[n] + (Double)intervalAndDelta.value;
            });
            energySurplusMax = Math.min(energySurplusMax, energySurplus);
            powerCapacity = Math.max(expPowerMax - powerSetpoints[interval], -1.0 * (energySurplusMax / this.intervalSize));
            powerSetpoints[interval] = powerSetpoints[interval] + powerCapacity;
            EnergyOptimisationService.LOG.finest("Applied export earning opportunity: interval=" + interval + ", set point=" + powerSetpoints[interval] + " (delta: " + powerCapacity + ")");
        }
    }

    public BiFunction<Integer, Double, double[]> getExportOptimiser(double[] powerNets, double[] powerNetLimits, double[] tariffImports, double[] tariffExports, double assetExportCost) {
        return (interval, powerMax) -> {
            double powerNet = powerNets[interval];
            double powerNetLimit = powerNetLimits[interval];
            double tariffImport = tariffImports[interval];
            double tariffExport = tariffExports[interval];
            if ((powerMax = Double.valueOf(Math.max(powerMax, powerNetLimit - powerNet))) >= 0.0) {
                return new double[]{Double.MAX_VALUE, 0.0, 0.0};
            }
            if (powerNet <= 0.0) {
                return new double[]{tariffExport + assetExportCost, powerMax, 0.0};
            }
            if (powerNet + powerMax > 0.0) {
                return new double[]{-1.0 * tariffImport + assetExportCost, powerMax, 0.0};
            }
            double powerStart = 0.0;
            double powerEnd = 0.0 - powerNet;
            double cost = powerEnd * (tariffImport - assetExportCost);
            if (tariffExport + assetExportCost < 0.0 || tariffExport <= -1.0 * tariffImport) {
                if (Math.abs(-1.0 * tariffImport - tariffExport) > Double.MIN_VALUE) {
                    powerStart = powerMax;
                }
                cost += -1.0 * (powerMax - powerEnd) * (tariffExport + assetExportCost);
                powerEnd = powerMax;
            }
            return new double[]{cost /= -1.0 * powerEnd, powerEnd, powerStart};
        };
    }

    public BiFunction<Integer, double[], double[]> getImportOptimiser(double[] powerNets, double[] powerNetLimits, double[] tariffImports, double[] tariffExports, double assetImportCost) {
        return (interval, powerRequiredMinMax) -> {
            double powerNet = powerNets[interval];
            double powerNetLimit = powerNetLimits[interval];
            double tariffImport = tariffImports[interval];
            double tariffExport = tariffExports[interval];
            double powerMin = powerRequiredMinMax[0];
            double powerMax = Math.min(powerRequiredMinMax[1], powerNetLimit - powerNet);
            if (powerMax <= 0.0) {
                return new double[]{Double.MAX_VALUE, 0.0, 0.0};
            }
            if (powerNet >= 0.0) {
                return new double[]{tariffImport + assetImportCost, powerMin, powerMax};
            }
            if (powerNet + powerMax < 0.0) {
                return new double[]{-1.0 * tariffExport + assetImportCost, powerMin, powerMax};
            }
            double powerStart = powerMin;
            double powerEnd = 0.0 - powerNet;
            double cost = -1.0 * powerEnd * (tariffExport - assetImportCost);
            if (powerMin > powerEnd) {
                cost += (powerMin - powerEnd) * (tariffImport + assetImportCost);
                powerEnd = powerMin;
            }
            if (powerEnd < powerMax && (tariffImport + assetImportCost < 0.0 || tariffImport <= -1.0 * tariffExport)) {
                if (Math.abs(-1.0 * tariffExport - tariffImport) > Double.MIN_VALUE) {
                    powerStart = powerMax;
                }
                cost += (powerMax - powerEnd) * (tariffImport + assetImportCost);
                powerEnd = powerMax;
            }
            return new double[]{cost /= powerEnd, powerStart, powerEnd};
        };
    }
}

