/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.manager.energy;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.openremote.container.message.MessageBrokerService;
import org.openremote.container.persistence.PersistenceService;
import org.openremote.container.timer.TimerService;
import org.openremote.manager.asset.AssetProcessingService;
import org.openremote.manager.asset.AssetStorageService;
import org.openremote.manager.datapoint.AssetPredictedDatapointService;
import org.openremote.manager.energy.EnergyOptimiser;
import org.openremote.manager.event.ClientEventService;
import org.openremote.manager.gateway.GatewayService;
import org.openremote.model.Container;
import org.openremote.model.ContainerService;
import org.openremote.model.PersistenceEvent;
import org.openremote.model.asset.Asset;
import org.openremote.model.asset.AssetDescriptor;
import org.openremote.model.asset.impl.ElectricVehicleAsset;
import org.openremote.model.asset.impl.ElectricityAsset;
import org.openremote.model.asset.impl.ElectricityChargerAsset;
import org.openremote.model.asset.impl.ElectricityConsumerAsset;
import org.openremote.model.asset.impl.ElectricityProducerAsset;
import org.openremote.model.asset.impl.ElectricityStorageAsset;
import org.openremote.model.asset.impl.ElectricitySupplierAsset;
import org.openremote.model.asset.impl.EnergyOptimisationAsset;
import org.openremote.model.asset.impl.GroupAsset;
import org.openremote.model.attribute.Attribute;
import org.openremote.model.attribute.AttributeEvent;
import org.openremote.model.attribute.AttributeExecuteStatus;
import org.openremote.model.attribute.AttributeRef;
import org.openremote.model.datapoint.ValueDatapoint;
import org.openremote.model.datapoint.query.AssetDatapointIntervalQuery;
import org.openremote.model.datapoint.query.AssetDatapointQuery;
import org.openremote.model.query.AssetQuery;
import org.openremote.model.query.LogicGroup;
import org.openremote.model.query.filter.AttributePredicate;
import org.openremote.model.query.filter.BooleanPredicate;
import org.openremote.model.query.filter.StringPredicate;
import org.openremote.model.query.filter.ValuePredicate;
import org.openremote.model.util.ValueUtil;
import org.openremote.model.value.MetaItemType;

public class EnergyOptimisationService
extends RouteBuilder
implements ContainerService {
    protected static final Logger LOG = Logger.getLogger(EnergyOptimisationService.class.getName());
    protected static final int OPTIMISATION_TIMEOUT_MILLIS = 600000;
    protected DateTimeFormatter formatter = DateTimeFormatter.ISO_LOCAL_DATE_TIME.withZone(ZoneId.from(ZoneOffset.UTC));
    protected TimerService timerService;
    protected AssetProcessingService assetProcessingService;
    protected AssetStorageService assetStorageService;
    protected AssetPredictedDatapointService assetPredictedDatapointService;
    protected MessageBrokerService messageBrokerService;
    protected ClientEventService clientEventService;
    protected GatewayService gatewayService;
    protected ExecutorService executorService;
    protected ScheduledExecutorService scheduledExecutorService;
    protected final Map<String, OptimisationInstance> assetOptimisationInstanceMap = new HashMap<String, OptimisationInstance>();
    protected List<String> forceChargeAssetIds = new ArrayList<String>();

    public void init(Container container) throws Exception {
        this.timerService = (TimerService)container.getService(TimerService.class);
        this.assetPredictedDatapointService = (AssetPredictedDatapointService)container.getService(AssetPredictedDatapointService.class);
        this.assetProcessingService = (AssetProcessingService)container.getService(AssetProcessingService.class);
        this.assetStorageService = (AssetStorageService)container.getService(AssetStorageService.class);
        this.messageBrokerService = (MessageBrokerService)container.getService(MessageBrokerService.class);
        this.clientEventService = (ClientEventService)container.getService(ClientEventService.class);
        this.gatewayService = (GatewayService)container.getService(GatewayService.class);
        this.executorService = container.getExecutor();
        this.scheduledExecutorService = container.getScheduledExecutor();
    }

    public void start(Container container) throws Exception {
        ((MessageBrokerService)container.getService(MessageBrokerService.class)).getContext().addRoutes((RoutesBuilder)this);
        LOG.fine("Loading optimisation assets...");
        List<EnergyOptimisationAsset> energyOptimisationAssets = this.assetStorageService.findAll(new AssetQuery().types(EnergyOptimisationAsset.class)).stream().map(asset -> (EnergyOptimisationAsset)asset).filter(optimisationAsset -> optimisationAsset.isOptimisationDisabled().orElse(false) == false).toList();
        LOG.fine("Found enabled optimisation asset count = " + energyOptimisationAssets.size());
        energyOptimisationAssets.forEach(this::startOptimisation);
        this.clientEventService.addSubscription(AttributeEvent.class, null, this::processAttributeEvent);
    }

    public void configure() throws Exception {
        this.from("seda://PersistenceTopic?multipleConsumers=true&concurrentConsumers=1&waitForTaskToComplete=NEVER&purgeWhenStopping=true&discardIfNoConsumers=true&size=25000").routeId("Persistence-EnergyOptimisation").filter(PersistenceService.isPersistenceEventForEntityType(EnergyOptimisationAsset.class)).filter(GatewayService.isNotForGateway(this.gatewayService)).process(exchange -> this.processAssetChange((PersistenceEvent<EnergyOptimisationAsset>)((PersistenceEvent)exchange.getIn().getBody(PersistenceEvent.class))));
    }

    public void stop(Container container) throws Exception {
        new ArrayList<String>(this.assetOptimisationInstanceMap.keySet()).forEach(this::stopOptimisation);
    }

    protected void processAssetChange(PersistenceEvent<EnergyOptimisationAsset> persistenceEvent) {
        LOG.fine("Processing optimisation asset change: " + String.valueOf(persistenceEvent));
        this.stopOptimisation(((EnergyOptimisationAsset)persistenceEvent.getEntity()).getId());
        if (persistenceEvent.getCause() != PersistenceEvent.Cause.DELETE && !((EnergyOptimisationAsset)persistenceEvent.getEntity()).isOptimisationDisabled().orElse(false).booleanValue()) {
            this.startOptimisation((EnergyOptimisationAsset)persistenceEvent.getEntity());
        }
    }

    protected void processAttributeEvent(AttributeEvent attributeEvent) {
        OptimisationInstance optimisationInstance = this.assetOptimisationInstanceMap.get(attributeEvent.getId());
        if (optimisationInstance != null) {
            this.processOptimisationAssetAttributeEvent(optimisationInstance, attributeEvent);
            return;
        }
        String attributeName = attributeEvent.getName();
        if ((attributeName.equals(ElectricityChargerAsset.VEHICLE_CONNECTED.getName()) || attributeName.equals(ElectricVehicleAsset.CHARGER_CONNECTED.getName())) && attributeEvent.getValue().orElse(false).booleanValue()) {
            if (this.forceChargeAssetIds.remove(attributeEvent.getId())) {
                LOG.fine("Previously force charged asset has now been disconnected so clearing force charge flag: " + attributeEvent.getId());
            }
            return;
        }
        if (attributeName.equals(ElectricityStorageAsset.FORCE_CHARGE.getName())) {
            Asset<?> asset = this.assetStorageService.find(attributeEvent.getId());
            if (!(asset instanceof ElectricityStorageAsset)) {
                LOG.fine("Request to force charge asset will be ignored as asset not found or is not of type '" + ElectricityStorageAsset.class.getSimpleName() + "': " + attributeEvent.getId());
                return;
            }
            ElectricityStorageAsset storageAsset = (ElectricityStorageAsset)asset;
            if (attributeEvent.getValue().orElse(null) == AttributeExecuteStatus.REQUEST_START) {
                double powerImportMax = storageAsset.getPowerImportMax().orElse(Double.MAX_VALUE);
                double maxEnergyLevel = this.getElectricityStorageAssetEnergyLevelMax(storageAsset);
                double currentEnergyLevel = storageAsset.getEnergyLevel().orElse(0.0);
                LOG.fine("Request to force charge asset '" + attributeEvent.getId() + "': attempting to set powerSetpoint=" + powerImportMax);
                if (this.forceChargeAssetIds.contains(attributeEvent.getId())) {
                    LOG.fine("Request to force charge asset will be ignored as force charge already requested for asset: " + String.valueOf(storageAsset));
                    return;
                }
                if (currentEnergyLevel >= maxEnergyLevel) {
                    LOG.fine("Request to force charge asset will be ignored as asset is already at or above maxEnergyLevel: " + String.valueOf(storageAsset));
                    return;
                }
                this.forceChargeAssetIds.add(attributeEvent.getId());
                this.assetProcessingService.sendAttributeEvent(new AttributeEvent(storageAsset.getId(), ElectricityAsset.POWER_SETPOINT, (Object)powerImportMax), ((Object)((Object)this)).getClass().getSimpleName());
                this.assetProcessingService.sendAttributeEvent(new AttributeEvent(storageAsset.getId(), ElectricityStorageAsset.FORCE_CHARGE, (Object)AttributeExecuteStatus.RUNNING), ((Object)((Object)this)).getClass().getSimpleName());
            } else if (attributeEvent.getValue().orElse(null) == AttributeExecuteStatus.REQUEST_CANCEL && this.forceChargeAssetIds.remove(attributeEvent.getId())) {
                LOG.info("Request to cancel force charge asset: " + storageAsset.getId());
                this.assetProcessingService.sendAttributeEvent(new AttributeEvent(storageAsset.getId(), ElectricityAsset.POWER_SETPOINT, (Object)0.0), ((Object)((Object)this)).getClass().getSimpleName());
                this.assetProcessingService.sendAttributeEvent(new AttributeEvent(storageAsset.getId(), ElectricityStorageAsset.FORCE_CHARGE, (Object)AttributeExecuteStatus.CANCELLED), ((Object)((Object)this)).getClass().getSimpleName());
            }
        }
    }

    protected double getElectricityStorageAssetEnergyLevelMax(ElectricityStorageAsset asset) {
        double energyCapacity = asset.getEnergyCapacity().orElse(0.0);
        int maxEnergyLevelPercentage = asset.getEnergyLevelPercentageMax().orElse(100);
        return energyCapacity * (1.0 * (double)maxEnergyLevelPercentage / 100.0);
    }

    protected synchronized void processOptimisationAssetAttributeEvent(OptimisationInstance optimisationInstance, AttributeEvent attributeEvent) {
        if (EnergyOptimisationAsset.FINANCIAL_SAVING.getName().equals(attributeEvent.getName()) || EnergyOptimisationAsset.CARBON_SAVING.getName().equals(attributeEvent.getName())) {
            return;
        }
        if (attributeEvent.getName().equals(EnergyOptimisationAsset.OPTIMISATION_DISABLED.getName())) {
            boolean disabled = attributeEvent.getValue().orElse(false);
            if (!disabled && this.assetOptimisationInstanceMap.containsKey(optimisationInstance.optimisationAsset.getId())) {
                return;
            }
            if (disabled && !this.assetOptimisationInstanceMap.containsKey(optimisationInstance.optimisationAsset.getId())) {
                return;
            }
        }
        LOG.info("Processing optimisation asset attribute event: " + String.valueOf(attributeEvent));
        this.stopOptimisation(attributeEvent.getId());
        EnergyOptimisationAsset asset = (EnergyOptimisationAsset)this.assetStorageService.find(attributeEvent.getId());
        if (asset != null && !asset.isOptimisationDisabled().orElse(false).booleanValue()) {
            this.startOptimisation(asset);
        }
    }

    protected synchronized void startOptimisation(EnergyOptimisationAsset optimisationAsset) {
        LOG.fine("Initialising optimiser for optimisation asset: " + String.valueOf(optimisationAsset));
        double intervalSize = optimisationAsset.getIntervalSize().orElse(0.25);
        int financialWeighting = optimisationAsset.getFinancialWeighting().orElse(100);
        try {
            EnergyOptimiser optimiser = new EnergyOptimiser(intervalSize, (double)financialWeighting / 100.0);
            long periodSeconds = (long)(optimiser.intervalSize * 60.0 * 60.0);
            if (periodSeconds < 300L) {
                throw new IllegalStateException("Optimiser interval size is too small (minimum is 5 mins) for asset: " + optimisationAsset.getId());
            }
            long currentMillis = this.timerService.getCurrentTimeMillis();
            Instant optimisationStartTime = EnergyOptimisationService.getOptimisationStartTime(currentMillis, periodSeconds);
            long offsetSeconds = (long)(Math.random() * 30.0) + periodSeconds;
            Duration startDuration = Duration.between(Instant.ofEpochMilli(currentMillis), optimisationStartTime.plus(offsetSeconds, ChronoUnit.SECONDS));
            ScheduledFuture<?> optimisationFuture = this.scheduleOptimisation(optimisationAsset.getId(), optimiser, startDuration, periodSeconds);
            this.assetOptimisationInstanceMap.put(optimisationAsset.getId(), new OptimisationInstance(optimisationAsset, optimiser, optimisationFuture));
            LOG.finest(this.getLogPrefix(optimisationAsset.getId()) + "Running first optimisation for time '" + this.formatter.format(optimisationStartTime));
            this.executorService.execute(() -> {
                try {
                    this.runOptimisation(optimisationAsset.getId(), optimisationStartTime);
                }
                catch (Exception e) {
                    LOG.log(Level.SEVERE, "Failed to run energy optimiser for asset: " + optimisationAsset.getId(), e);
                }
            });
        }
        catch (Exception e) {
            LOG.log(Level.SEVERE, "Failed to start energy optimiser for asset: " + String.valueOf(optimisationAsset), e);
        }
    }

    protected synchronized void stopOptimisation(String optimisationAssetId) {
        OptimisationInstance optimisationInstance = this.assetOptimisationInstanceMap.remove(optimisationAssetId);
        if (optimisationInstance == null || optimisationInstance.optimiserFuture == null) {
            return;
        }
        LOG.fine("Removing optimiser for optimisation asset: " + optimisationAssetId);
        optimisationInstance.optimiserFuture.cancel(false);
    }

    protected ScheduledFuture<?> scheduleOptimisation(String optimisationAssetId, EnergyOptimiser optimiser, Duration startDuration, long periodSeconds) throws IllegalStateException {
        if (optimiser == null) {
            throw new IllegalStateException("Optimiser instance not found for asset: " + optimisationAssetId);
        }
        return this.scheduledExecutorService.scheduleAtFixedRate(() -> {
            try {
                this.runOptimisation(optimisationAssetId, Instant.ofEpochMilli(this.timerService.getCurrentTimeMillis()).truncatedTo(ChronoUnit.MINUTES));
            }
            catch (Exception e) {
                LOG.log(Level.SEVERE, "Failed to run energy optimiser for asset: " + optimisationAssetId, e);
            }
        }, startDuration.getSeconds(), periodSeconds, TimeUnit.SECONDS);
    }

    protected static Instant getOptimisationStartTime(long currentMillis, long periodSeconds) {
        Instant now = Instant.ofEpochMilli(currentMillis);
        Instant optimisationStartTime = now.truncatedTo(ChronoUnit.DAYS);
        while (optimisationStartTime.isBefore(now)) {
            optimisationStartTime = optimisationStartTime.plus(periodSeconds, ChronoUnit.SECONDS);
        }
        return optimisationStartTime.minus(periodSeconds, ChronoUnit.SECONDS);
    }

    protected String getLogPrefix(String optimisationAssetId) {
        return "Optimisation '" + optimisationAssetId + "': ";
    }

    protected void checkTimeoutAndThrow(String optimisationAssetId, long startTimeMillis) throws TimeoutException {
        long runtime = this.timerService.getCurrentTimeMillis() - startTimeMillis;
        if (runtime > 600000L) {
            String logMsg = this.getLogPrefix(optimisationAssetId) + "Optimisation has been running for " + runtime + "ms, timeout is at 600000ms";
            LOG.warning(logMsg);
            throw new TimeoutException(logMsg);
        }
    }

    protected void runOptimisation(String optimisationAssetId, Instant optimisationTime) throws Exception {
        OptimisationInstance optimisationInstance = this.assetOptimisationInstanceMap.get(optimisationAssetId);
        if (optimisationInstance == null) {
            return;
        }
        LOG.finest(this.getLogPrefix(optimisationAssetId) + "Running for time '" + this.formatter.format(optimisationTime));
        long startTimeMillis = this.timerService.getCurrentTimeMillis();
        EnergyOptimiser optimiser = optimisationInstance.energyOptimiser;
        int intervalCount = optimiser.get24HourIntervalCount();
        double intervalSize = optimiser.getIntervalSize();
        LOG.finest(this.getLogPrefix(optimisationAssetId) + "Fetching child assets of type '" + ElectricitySupplierAsset.class.getSimpleName() + "'");
        List<ElectricitySupplierAsset> supplierAssets = this.assetStorageService.findAll(new AssetQuery().types(ElectricitySupplierAsset.class).recursive(true).parents(new String[]{optimisationAssetId})).stream().filter(asset -> asset.hasAttribute(ElectricitySupplierAsset.TARIFF_IMPORT)).map(asset -> (ElectricitySupplierAsset)asset).toList();
        if (supplierAssets.size() != 1) {
            LOG.warning(this.getLogPrefix(optimisationAssetId) + "Expected exactly one " + ElectricitySupplierAsset.class.getSimpleName() + " asset with a '" + ElectricitySupplierAsset.TARIFF_IMPORT.getName() + "' attribute but found: " + supplierAssets.size());
            return;
        }
        double[] powerNets = new double[intervalCount];
        ElectricitySupplierAsset supplierAsset = supplierAssets.get(0);
        if (LOG.isLoggable(Level.FINEST)) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Found child asset of type '" + ElectricitySupplierAsset.class.getSimpleName() + "': " + String.valueOf(supplierAsset));
        }
        if (supplierAsset.getTariffImport().isPresent()) {
            LOG.warning(this.getLogPrefix(optimisationAssetId) + ElectricitySupplierAsset.class.getSimpleName() + " asset '" + ElectricitySupplierAsset.TARIFF_IMPORT.getName() + "' attribute has no value");
        }
        LOG.finest(this.getLogPrefix(optimisationAssetId) + "Fetching optimisable child assets of type '" + ElectricityStorageAsset.class.getSimpleName() + "'");
        List optimisableStorageAssets = this.assetStorageService.findAll(new AssetQuery().recursive(true).parents(new String[]{optimisationAssetId}).types(ElectricityStorageAsset.class).attributes(new LogicGroup(LogicGroup.Operator.AND, Collections.singletonList(new LogicGroup(LogicGroup.Operator.OR, (Object[])new AttributePredicate[]{new AttributePredicate(ElectricityStorageAsset.SUPPORTS_IMPORT.getName(), (ValuePredicate)new BooleanPredicate(true)), new AttributePredicate(ElectricityStorageAsset.SUPPORTS_EXPORT.getName(), (ValuePredicate)new BooleanPredicate(true))})), (Object[])new AttributePredicate[]{new AttributePredicate().name(new StringPredicate(ElectricityAsset.POWER_SETPOINT.getName()))}))).stream().map(asset -> (ElectricityStorageAsset)asset).collect(Collectors.toList());
        this.checkTimeoutAndThrow(optimisationAssetId, startTimeMillis);
        List finalOptimisableStorageAssets = optimisableStorageAssets;
        optimisableStorageAssets = optimisableStorageAssets.stream().filter(asset -> {
            if (this.forceChargeAssetIds.contains(asset.getId())) {
                LOG.finest("Optimisable asset was requested to force charge so it won't be optimised: " + asset.getId());
                Attribute powerAttribute = (Attribute)asset.getAttribute(ElectricityAsset.POWER).get();
                double[] powerLevels = this.get24HAttributeValues(asset.getId(), (Attribute<Double>)powerAttribute, optimiser.getIntervalSize(), intervalCount, optimisationTime);
                IntStream.range(0, intervalCount).forEach(i -> {
                    int n = i;
                    powerNets[n] = powerNets[n] + powerLevels[i];
                });
                double currentEnergyLevel = asset.getEnergyLevel().orElse(0.0);
                double maxEnergyLevel = this.getElectricityStorageAssetEnergyLevelMax((ElectricityStorageAsset)asset);
                if (currentEnergyLevel >= maxEnergyLevel) {
                    LOG.info("Force charged asset has reached maxEnergyLevelPercentage so stopping charging: " + asset.getId());
                    this.forceChargeAssetIds.remove(asset.getId());
                    this.assetProcessingService.sendAttributeEvent(new AttributeEvent(asset.getId(), ElectricityStorageAsset.POWER_SETPOINT, (Object)0.0), ((Object)((Object)this)).getClass().getSimpleName());
                    this.assetProcessingService.sendAttributeEvent(new AttributeEvent(asset.getId(), ElectricityStorageAsset.FORCE_CHARGE, (Object)AttributeExecuteStatus.COMPLETED), ((Object)((Object)this)).getClass().getSimpleName());
                }
                return false;
            }
            if (asset instanceof ElectricityChargerAsset) {
                return finalOptimisableStorageAssets.stream().noneMatch(a -> {
                    if (a instanceof ElectricVehicleAsset && a.getParentId().equals(asset.getId())) {
                        double vehiclePowerImportMax = a.getPowerImportMax().orElse(Double.MAX_VALUE);
                        double vehiclePowerExportMax = a.getPowerExportMax().orElse(Double.MAX_VALUE);
                        double chargerPowerImportMax = asset.getPowerImportMax().orElse(Double.MAX_VALUE);
                        double chargerPowerExportMax = asset.getPowerExportMax().orElse(Double.MAX_VALUE);
                        double smallestPowerImportMax = Math.min(vehiclePowerImportMax, chargerPowerImportMax);
                        double smallestPowerExportMax = Math.min(vehiclePowerExportMax, chargerPowerExportMax);
                        if (smallestPowerImportMax < vehiclePowerImportMax) {
                            LOG.fine("Reducing vehicle power import max due to connected charger limit: vehicle=" + a.getId() + ", oldPowerImportMax=" + vehiclePowerImportMax + ", newPowerImportMax=" + smallestPowerImportMax);
                            a.setPowerImportMax(Double.valueOf(smallestPowerImportMax));
                        }
                        if (smallestPowerExportMax < vehiclePowerExportMax) {
                            LOG.fine("Reducing vehicle power Export max due to connected charger limit: vehicle=" + a.getId() + ", oldPowerExportMax=" + vehiclePowerExportMax + ", newPowerExportMax=" + smallestPowerExportMax);
                            a.setPowerExportMax(Double.valueOf(smallestPowerExportMax));
                        }
                        LOG.finest("Excluding charger from optimisable assets and child vehicle will be used instead: " + asset.getId());
                        return true;
                    }
                    return false;
                });
            }
            return true;
        }).sorted(Comparator.comparingInt(asset -> asset.getEnergyLevelSchedule().map(schedule -> 0).orElse(1))).collect(Collectors.toList());
        this.checkTimeoutAndThrow(optimisationAssetId, startTimeMillis);
        if (optimisableStorageAssets.isEmpty()) {
            LOG.warning(this.getLogPrefix(optimisationAssetId) + "Expected at least one optimisable '" + ElectricityStorageAsset.class.getSimpleName() + " asset with a '" + ElectricityAsset.POWER_SETPOINT.getName() + "' attribute but found none");
            return;
        }
        if (LOG.isLoggable(Level.FINEST)) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Found optimisable child assets of type '" + ElectricityStorageAsset.class.getSimpleName() + "': " + optimisableStorageAssets.stream().map(Asset::getId).collect(Collectors.joining(", ")));
        }
        LOG.finest(this.getLogPrefix(optimisationAssetId) + "Fetching plain consumer and producer child assets of type '" + ElectricityProducerAsset.class.getSimpleName() + "', '" + ElectricityConsumerAsset.class.getSimpleName() + "', '" + ElectricityStorageAsset.class.getSimpleName() + "'");
        AtomicInteger count = new AtomicInteger(0);
        this.assetStorageService.findAll(new AssetQuery().recursive(true).parents(new String[]{optimisationAssetId}).types(new Class[]{ElectricityConsumerAsset.class, ElectricityProducerAsset.class}).attributes(new AttributePredicate[]{new AttributePredicate().name(new StringPredicate(ElectricityAsset.POWER.getName()))})).forEach(asset -> {
            Attribute powerAttribute = (Attribute)asset.getAttribute(ElectricityAsset.POWER).get();
            double[] powerLevels = this.get24HAttributeValues(asset.getId(), (Attribute<Double>)powerAttribute, optimiser.getIntervalSize(), intervalCount, optimisationTime);
            IntStream.range(0, intervalCount).forEach(i -> {
                int n = i;
                powerNets[n] = powerNets[n] + powerLevels[i];
            });
            count.incrementAndGet();
        });
        this.checkTimeoutAndThrow(optimisationAssetId, startTimeMillis);
        List<ElectricityStorageAsset> plainStorageAssets = this.assetStorageService.findAll(new AssetQuery().recursive(true).parents(new String[]{optimisationAssetId}).types(ElectricityStorageAsset.class).attributes(new AttributePredicate[]{new AttributePredicate().name(new StringPredicate(ElectricityAsset.POWER.getName())), new AttributePredicate(ElectricityStorageAsset.SUPPORTS_IMPORT.getName(), (ValuePredicate)new BooleanPredicate(true), true, null), new AttributePredicate(ElectricityStorageAsset.SUPPORTS_EXPORT.getName(), (ValuePredicate)new BooleanPredicate(true), true, null)})).stream().map(asset -> (ElectricityStorageAsset)asset).toList();
        this.checkTimeoutAndThrow(optimisationAssetId, startTimeMillis);
        plainStorageAssets.stream().filter(asset -> {
            if (asset instanceof ElectricityChargerAsset) {
                return plainStorageAssets.stream().noneMatch(a -> {
                    if (a instanceof ElectricVehicleAsset && a.getParentId().equals(asset.getId())) {
                        LOG.finest("Excluding charger from plain consumer/producer calculations to avoid double counting power: " + asset.getId());
                        return true;
                    }
                    return false;
                }) && finalOptimisableStorageAssets.stream().noneMatch(a -> {
                    if (a instanceof ElectricVehicleAsset && a.getParentId().equals(asset.getId())) {
                        LOG.finest("Excluding charger from plain consumer/producer calculations to avoid double counting power: " + asset.getId());
                        return true;
                    }
                    return false;
                });
            }
            return true;
        }).forEach(asset -> {
            Attribute powerAttribute = (Attribute)asset.getAttribute(ElectricityAsset.POWER).get();
            double[] powerLevels = this.get24HAttributeValues(asset.getId(), (Attribute<Double>)powerAttribute, optimiser.getIntervalSize(), intervalCount, optimisationTime);
            IntStream.range(0, intervalCount).forEach(i -> {
                int n = i;
                powerNets[n] = powerNets[n] + powerLevels[i];
            });
            count.incrementAndGet();
        });
        this.checkTimeoutAndThrow(optimisationAssetId, startTimeMillis);
        if (LOG.isLoggable(Level.FINEST)) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Found plain consumer and producer child assets count=" + count.get());
            LOG.finest("Calculated net power of consumers and producers: " + Arrays.toString(powerNets));
        }
        double financialWeightingImport = optimiser.getFinancialWeighting();
        double financialWeightingExport = optimiser.getFinancialWeighting();
        if (financialWeightingImport < 1.0 && !supplierAsset.getCarbonImport().isPresent()) {
            financialWeightingImport = 1.0;
        }
        if (financialWeightingExport < 1.0 && !supplierAsset.getCarbonExport().isPresent()) {
            financialWeightingExport = 1.0;
        }
        double[] costsImport = this.get24HAttributeValues(supplierAsset.getId(), (Attribute<Double>)((Attribute)supplierAsset.getAttribute(ElectricitySupplierAsset.TARIFF_IMPORT).orElse(null)), optimiser.getIntervalSize(), intervalCount, optimisationTime);
        double[] costsExport = this.get24HAttributeValues(supplierAsset.getId(), (Attribute<Double>)((Attribute)supplierAsset.getAttribute(ElectricitySupplierAsset.TARIFF_EXPORT).orElse(null)), optimiser.getIntervalSize(), intervalCount, optimisationTime);
        if (financialWeightingImport < 1.0 || financialWeightingExport < 1.0) {
            double[] carbonImport = this.get24HAttributeValues(supplierAsset.getId(), (Attribute<Double>)((Attribute)supplierAsset.getAttribute(ElectricitySupplierAsset.CARBON_IMPORT).orElse(null)), optimiser.getIntervalSize(), intervalCount, optimisationTime);
            double[] carbonExport = this.get24HAttributeValues(supplierAsset.getId(), (Attribute<Double>)((Attribute)supplierAsset.getAttribute(ElectricitySupplierAsset.CARBON_EXPORT).orElse(null)), optimiser.getIntervalSize(), intervalCount, optimisationTime);
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Adjusting costs to include some carbon weighting, financialWeightingImport=" + financialWeightingImport + ", financialWeightingExport=" + financialWeightingExport);
            for (int i2 = 0; i2 < costsImport.length; ++i2) {
                costsImport[i2] = financialWeightingImport * costsImport[i2] + (1.0 - financialWeightingImport) * carbonImport[i2];
                costsExport[i2] = financialWeightingExport * costsExport[i2] + (1.0 - financialWeightingExport) * carbonExport[i2];
            }
        }
        if (LOG.isLoggable(Level.FINEST)) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Import costs: " + Arrays.toString(costsImport));
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Export costs: " + Arrays.toString(costsExport));
        }
        ArrayList<String> obsoleteUnoptimisedAssetIds = new ArrayList<String>(optimisationInstance.unoptimisedStorageAssetEnergyLevels.keySet());
        double unoptimisedPower = powerNets[0];
        double financialCost = 0.0;
        double carbonCost = 0.0;
        double unoptimisedFinancialCost = 0.0;
        double unoptimisedCarbonCost = 0.0;
        double importPowerMax = supplierAsset.getPowerImportMax().orElse(Double.MAX_VALUE);
        double exportPowerMax = -1.0 * supplierAsset.getPowerExportMax().orElse(Double.MAX_VALUE);
        double[] importPowerMaxes = new double[intervalCount];
        double[] exportPowerMaxes = new double[intervalCount];
        Arrays.fill(importPowerMaxes, importPowerMax);
        Arrays.fill(exportPowerMaxes, exportPowerMax);
        long periodSeconds = (long)(optimiser.getIntervalSize() * 60.0 * 60.0);
        for (ElectricityStorageAsset storageAsset : optimisableStorageAssets) {
            double[] setpoints;
            boolean hasSetpoint = storageAsset.hasAttribute(ElectricityStorageAsset.POWER_SETPOINT);
            boolean supportsExport = storageAsset.isSupportsExport().orElse(false);
            boolean supportsImport = storageAsset.isSupportsImport().orElse(false);
            this.checkTimeoutAndThrow(optimisationAssetId, startTimeMillis);
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Optimising power set points for storage asset: " + String.valueOf(storageAsset));
            if (!supportsExport && !supportsImport) {
                LOG.finest(this.getLogPrefix(optimisationAssetId) + "Storage asset doesn't support import or export: " + storageAsset.getId());
                continue;
            }
            if (!hasSetpoint) {
                LOG.info(this.getLogPrefix(optimisationAssetId) + "Storage asset has no '" + ElectricityStorageAsset.POWER_SETPOINT.getName() + "' attribute so cannot be controlled: " + storageAsset.getId());
                continue;
            }
            double energyCapacity = storageAsset.getEnergyCapacity().orElse(0.0);
            double energyLevel = Math.min(energyCapacity, storageAsset.getEnergyLevel().orElse(-1.0));
            if (energyCapacity <= 0.0 || energyLevel < 0.0) {
                LOG.info(this.getLogPrefix(optimisationAssetId) + "Storage asset has no capacity or energy level so cannot import or export energy: " + storageAsset.getId());
                continue;
            }
            double energyLevelMax = Math.min(energyCapacity, (double)storageAsset.getEnergyLevelPercentageMax().orElse(100).intValue() / 100.0 * energyCapacity);
            double energyLevelMin = Math.min(energyCapacity, (double)storageAsset.getEnergyLevelPercentageMin().orElse(0).intValue() / 100.0 * energyCapacity);
            double[] energyLevelMins = new double[intervalCount];
            double[] energyLevelMaxs = new double[intervalCount];
            Arrays.fill(energyLevelMins, energyLevelMin);
            Arrays.fill(energyLevelMaxs, energyLevelMax);
            Optional energyLevelScheduleOptional = storageAsset.getEnergyLevelSchedule();
            boolean hasEnergyMinRequirement = energyLevelMin > 0.0 || energyLevelScheduleOptional.isPresent();
            double powerExportMax = storageAsset.getPowerExportMax().map(power -> -1.0 * power).orElse((Double)Double.MIN_VALUE);
            double powerImportMax = storageAsset.getPowerImportMax().orElse(Double.MAX_VALUE);
            int[][] energySchedule = energyLevelScheduleOptional.map(dayArr -> (int[][])Arrays.stream(dayArr).map(hourArr -> Arrays.stream(hourArr).mapToInt(i -> i != null ? i : 0).toArray()).toArray(x$0 -> new int[x$0][])).orElse(null);
            if (energySchedule != null) {
                LOG.finest(this.getLogPrefix(optimisationAssetId) + "Applying energy schedule for storage asset: " + storageAsset.getId());
                optimiser.applyEnergySchedule(energyLevelMins, energyLevelMaxs, energyCapacity, energySchedule, Instant.ofEpochMilli(this.timerService.getCurrentTimeMillis()).atZone(ZoneId.systemDefault()).toLocalDateTime());
            }
            double maxEnergyLevelMin = Arrays.stream(energyLevelMins).max().orElse(0.0);
            boolean isConnected = this.storageAssetConnected(storageAsset);
            Function<Integer, Double> powerImportMaxCalculator = interval -> interval == 0 && !isConnected ? 0.0 : powerImportMax;
            Function<Integer, Double> powerExportMaxCalculator = interval -> interval == 0 && !isConnected ? 0.0 : powerExportMax;
            if (hasEnergyMinRequirement) {
                LOG.finest(this.getLogPrefix(optimisationAssetId) + "Normalising min energy requirements for storage asset: " + storageAsset.getId());
                optimiser.normaliseEnergyMinRequirements(energyLevelMins, powerImportMaxCalculator, powerExportMaxCalculator, energyLevel);
                if (LOG.isLoggable(Level.FINEST)) {
                    LOG.finest(this.getLogPrefix(optimisationAssetId) + "Min energy requirements for storage asset '" + storageAsset.getId() + "': " + Arrays.toString(energyLevelMins));
                }
            }
            if ((setpoints = this.getStoragePowerSetpoints(optimisationInstance, storageAsset, energyLevelMins, energyLevelMaxs, powerNets, importPowerMaxes, exportPowerMaxes, costsImport, costsExport)) != null) {
                for (int i3 = 0; i3 < powerNets.length; ++i3) {
                    if (i3 == 0) {
                        if (!this.storageAssetConnected(storageAsset)) {
                            LOG.finest("Optimised storage asset not connected so interval 0 will not be counted or actioned: " + storageAsset.getId());
                            setpoints[i3] = 0.0;
                            continue;
                        }
                        financialCost = setpoints[i3] > 0.0 ? (financialCost += storageAsset.getTariffImport().orElse(0.0) * setpoints[i3] * intervalSize) : (financialCost += storageAsset.getTariffExport().orElse(0.0) * -1.0 * setpoints[i3] * intervalSize);
                    }
                    int n = i3;
                    powerNets[n] = powerNets[n] + setpoints[i3];
                }
                List<ValueDatapoint<?>> valuesAndTimestamps = IntStream.range(1, setpoints.length).mapToObj(i -> new ValueDatapoint(optimisationTime.plus(periodSeconds * (long)i, ChronoUnit.SECONDS).toEpochMilli(), (Object)setpoints[i])).collect(Collectors.toList());
                this.assetPredictedDatapointService.updateValues(storageAsset.getId(), ElectricityAsset.POWER_SETPOINT.getName(), valuesAndTimestamps);
            }
            this.assetProcessingService.sendAttributeEvent(new AttributeEvent(storageAsset.getId(), ElectricityAsset.POWER_SETPOINT, setpoints != null ? Double.valueOf(setpoints[0]) : null), ((Object)((Object)this)).getClass().getSimpleName());
            obsoleteUnoptimisedAssetIds.remove(storageAsset.getId());
            double assetUnoptimisedPower = this.getStorageUnoptimisedImportPower(optimisationInstance, optimisationAssetId, storageAsset, maxEnergyLevelMin, Math.max(0.0, powerImportMax - unoptimisedPower));
            unoptimisedPower += assetUnoptimisedPower;
            unoptimisedFinancialCost += storageAsset.getTariffImport().orElse(0.0) * assetUnoptimisedPower * intervalSize;
        }
        obsoleteUnoptimisedAssetIds.forEach(optimisationInstance.unoptimisedStorageAssetEnergyLevels.keySet()::remove);
        carbonCost = (powerNets[0] >= 0.0 ? supplierAsset.getCarbonImport().orElse(0.0) : -1.0 * supplierAsset.getCarbonExport().orElse(0.0)) * powerNets[0] * intervalSize;
        unoptimisedCarbonCost = (unoptimisedPower >= 0.0 ? supplierAsset.getCarbonImport().orElse(0.0) : -1.0 * supplierAsset.getCarbonExport().orElse(0.0)) * unoptimisedPower * intervalSize;
        double financialSaving = (unoptimisedFinancialCost += (unoptimisedPower >= 0.0 ? supplierAsset.getTariffImport().orElse(0.0) : -1.0 * supplierAsset.getTariffExport().orElse(0.0)) * unoptimisedPower * intervalSize) - (financialCost += (powerNets[0] >= 0.0 ? supplierAsset.getTariffImport().orElse(0.0) : -1.0 * supplierAsset.getTariffExport().orElse(0.0)) * powerNets[0] * intervalSize);
        double carbonSaving = unoptimisedCarbonCost - carbonCost;
        LOG.info(this.getLogPrefix(optimisationAssetId) + "Current interval financial saving = " + financialSaving);
        LOG.info(this.getLogPrefix(optimisationAssetId) + "Current interval carbon saving = " + carbonSaving);
        optimisationInstance.optimisationAsset.setFinancialSaving(Double.valueOf(financialSaving += optimisationInstance.optimisationAsset.getFinancialSaving().orElse(0.0).doubleValue()));
        optimisationInstance.optimisationAsset.setCarbonSaving(Double.valueOf(carbonSaving += optimisationInstance.optimisationAsset.getCarbonSaving().orElse(0.0).doubleValue()));
        this.assetProcessingService.sendAttributeEvent(new AttributeEvent(optimisationAssetId, EnergyOptimisationAsset.FINANCIAL_SAVING, (Object)financialSaving), ((Object)((Object)this)).getClass().getSimpleName());
        this.assetProcessingService.sendAttributeEvent(new AttributeEvent(optimisationAssetId, EnergyOptimisationAsset.CARBON_SAVING, (Object)carbonSaving), ((Object)((Object)this)).getClass().getSimpleName());
    }

    protected boolean isElectricityGroupAsset(Asset<?> asset) {
        if (!(asset instanceof GroupAsset)) {
            return false;
        }
        Class assetClass = ValueUtil.getAssetDescriptor((String)((GroupAsset)asset).getChildAssetType().orElse(null)).map(AssetDescriptor::getType).orElse(null);
        return assetClass != null && ElectricityAsset.class.isAssignableFrom(assetClass);
    }

    protected double[] get24HAttributeValues(String assetId, Attribute<Double> attribute, double intervalSize, int intervalCount, Instant optimisationTime) {
        double[] values = new double[intervalCount];
        if (attribute == null) {
            return values;
        }
        AttributeRef ref = new AttributeRef(assetId, attribute.getName());
        if (attribute.hasMeta(MetaItemType.HAS_PREDICTED_DATA_POINTS)) {
            LocalDateTime timestamp = optimisationTime.atZone(ZoneId.systemDefault()).toLocalDateTime();
            List<ValueDatapoint<?>> predictedData = this.assetPredictedDatapointService.queryDatapoints(ref.getId(), ref.getName(), (AssetDatapointQuery)new AssetDatapointIntervalQuery(timestamp, timestamp.plus(24L, ChronoUnit.HOURS).minus((long)(intervalSize * 60.0), ChronoUnit.MINUTES), intervalSize * 60.0 + " minutes", AssetDatapointIntervalQuery.Formula.AVG, true));
            if (predictedData.size() != values.length) {
                LOG.warning("Returned predicted data point count does not match interval count: Ref=" + String.valueOf(ref) + ", expected=" + values.length + ", actual=" + predictedData.size());
            } else {
                IntStream.range(0, predictedData.size()).forEach(i -> {
                    if (((ValueDatapoint)predictedData.get(i)).getValue() != null) {
                        values[i] = (Double)((ValueDatapoint)predictedData.get(i)).getValue();
                    } else {
                        int j;
                        Double previous = null;
                        Double next = null;
                        for (j = i - 1; previous == null && j >= 0; --j) {
                            previous = (Double)((ValueDatapoint)predictedData.get(j)).getValue();
                        }
                        for (j = i + 1; next == null && j < predictedData.size(); ++j) {
                            next = (Double)((ValueDatapoint)predictedData.get(j)).getValue();
                        }
                        if (next == null) {
                            next = previous;
                        }
                        if (previous == null) {
                            previous = next;
                        }
                        if (next != null) {
                            values[i] = (previous + next) / 2.0;
                        }
                    }
                });
            }
        }
        values[0] = attribute.getValue().orElse(0.0);
        return values;
    }

    protected double[] getStoragePowerSetpoints(OptimisationInstance optimisationInstance, ElectricityStorageAsset storageAsset, double[] normalisedEnergyLevelMins, double[] energyLevelMaxs, double[] powerNets, double[] importPowerLimits, double[] exportPowerLimits, double[] costImports, double[] costExports) {
        EnergyOptimiser optimiser = optimisationInstance.energyOptimiser;
        String optimisationAssetId = optimisationInstance.optimisationAsset.getId();
        int intervalCount = optimiser.get24HourIntervalCount();
        boolean supportsExport = storageAsset.isSupportsExport().orElse(false);
        boolean supportsImport = storageAsset.isSupportsImport().orElse(false);
        LOG.finest(this.getLogPrefix(optimisationAssetId) + "Optimising storage asset: " + String.valueOf(storageAsset));
        double energyCapacity = storageAsset.getEnergyCapacity().orElse(0.0);
        double energyLevel = Math.min(energyCapacity, storageAsset.getEnergyLevel().orElse(-1.0));
        double powerExportMax = storageAsset.getPowerExportMax().map(power -> -1.0 * power).orElse((Double)Double.MIN_VALUE);
        double powerImportMax = storageAsset.getPowerImportMax().orElse(Double.MAX_VALUE);
        boolean isConnected = this.storageAssetConnected(storageAsset);
        Function<Integer, Double> powerImportMaxCalculator = interval -> interval == 0 && !isConnected ? 0.0 : powerImportMax;
        Function<Integer, Double> powerExportMaxCalculator = interval -> interval == 0 && !isConnected ? 0.0 : powerExportMax;
        double[][] exportCostAndPower = null;
        double[][] importCostAndPower = null;
        double[] powerSetpoints = new double[intervalCount];
        Function<Integer, Double> energyLevelCalculator = interval -> energyLevel + IntStream.range(0, interval).mapToDouble(j -> powerSetpoints[j] * optimiser.getIntervalSize()).sum();
        if (supportsExport) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Storage asset supports export so calculating export cost and power levels for each interval: " + storageAsset.getId());
            BiFunction<Integer, Double, double[]> exportOptimiser = optimiser.getExportOptimiser(powerNets, exportPowerLimits, costImports, costExports, storageAsset.getTariffExport().orElse(0.0));
            exportCostAndPower = (double[][])IntStream.range(0, intervalCount).mapToObj(it -> (double[])exportOptimiser.apply(it, powerExportMax)).toArray(x$0 -> new double[x$0][]);
        }
        if (supportsImport) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Storage asset supports import so calculating export cost and power levels for each interval: " + storageAsset.getId());
            BiFunction<Integer, double[], double[]> importOptimiser = optimiser.getImportOptimiser(powerNets, importPowerLimits, costImports, costExports, storageAsset.getTariffImport().orElse(0.0));
            importCostAndPower = (double[][])IntStream.range(0, intervalCount).mapToObj(it -> (double[])importOptimiser.apply(it, new double[]{0.0, powerImportMax})).toArray(x$0 -> new double[x$0][]);
            boolean hasEnergyMinRequirement = Arrays.stream(normalisedEnergyLevelMins).anyMatch(el -> el > 0.0);
            if (hasEnergyMinRequirement) {
                LOG.finest(this.getLogPrefix(optimisationAssetId) + "Applying imports to achieve min energy level requirements for storage asset: " + storageAsset.getId());
                optimiser.applyEnergyMinImports(importCostAndPower, normalisedEnergyLevelMins, powerSetpoints, energyLevelCalculator, importOptimiser, powerImportMaxCalculator);
                if (LOG.isLoggable(Level.FINEST)) {
                    LOG.finest(this.getLogPrefix(optimisationAssetId) + "Setpoints to achieve min energy level requirements for storage asset '" + storageAsset.getId() + "': " + Arrays.toString(powerSetpoints));
                }
            }
        }
        optimiser.applyEarningOpportunities(importCostAndPower, exportCostAndPower, normalisedEnergyLevelMins, energyLevelMaxs, powerSetpoints, energyLevelCalculator, powerImportMaxCalculator, powerExportMaxCalculator);
        if (LOG.isLoggable(Level.FINEST)) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Calculated earning opportunity power set points for storage asset '" + storageAsset.getId() + "': " + Arrays.toString(powerSetpoints));
        }
        return powerSetpoints;
    }

    protected boolean storageAssetConnected(ElectricityStorageAsset storageAsset) {
        if (storageAsset instanceof ElectricVehicleAsset) {
            return ((ElectricVehicleAsset)storageAsset).getChargerConnected().orElse(false);
        }
        if (storageAsset instanceof ElectricityChargerAsset) {
            return ((ElectricityChargerAsset)storageAsset).getVehicleConnected().orElse(false);
        }
        return true;
    }

    protected double getStorageUnoptimisedImportPower(OptimisationInstance optimisationInstance, String optimisationAssetId, ElectricityStorageAsset storageAsset, double energyLevelTarget, double remainingPowerCapacity) {
        double intervalSize = optimisationInstance.energyOptimiser.getIntervalSize();
        boolean isConnected = this.storageAssetConnected(storageAsset);
        if (!isConnected) {
            optimisationInstance.unoptimisedStorageAssetEnergyLevels.remove(storageAsset.getId());
            return 0.0;
        }
        double energyLevel = optimisationInstance.unoptimisedStorageAssetEnergyLevels.get(storageAsset.getId()) != null ? optimisationInstance.unoptimisedStorageAssetEnergyLevels.get(storageAsset.getId()) : storageAsset.getEnergyLevel().orElse(-1.0);
        if (energyLevel < 0.0) {
            LOG.finest(this.getLogPrefix(optimisationAssetId) + "Storage asset has no energy level so cannot calculate un-optimised power demand: " + storageAsset.getId());
            return 0.0;
        }
        double powerImportMax = storageAsset.getPowerImportMax().orElse(Double.MAX_VALUE);
        double remainingEnergy = Math.max(0.0, energyLevelTarget - energyLevel);
        double toFillPower = remainingEnergy / intervalSize;
        double power = Math.min(Math.min(toFillPower, powerImportMax), remainingPowerCapacity);
        optimisationInstance.unoptimisedStorageAssetEnergyLevels.put(storageAsset.getId(), energyLevel += power * intervalSize);
        return power;
    }

    protected static class OptimisationInstance {
        EnergyOptimisationAsset optimisationAsset;
        EnergyOptimiser energyOptimiser;
        ScheduledFuture<?> optimiserFuture;
        Map<String, Double> unoptimisedStorageAssetEnergyLevels = new HashMap<String, Double>();

        public OptimisationInstance(EnergyOptimisationAsset optimisationAsset, EnergyOptimiser energyOptimiser, ScheduledFuture<?> optimiserFuture) {
            this.optimisationAsset = optimisationAsset;
            this.energyOptimiser = energyOptimiser;
            this.optimiserFuture = optimiserFuture;
        }
    }
}

