/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.manager.asset;

import jakarta.persistence.TypedQuery;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.openremote.container.message.MessageBrokerService;
import org.openremote.container.persistence.PersistenceService;
import org.openremote.container.timer.TimerService;
import org.openremote.manager.asset.AssetStorageService;
import org.openremote.manager.datapoint.AssetDatapointService;
import org.openremote.manager.datapoint.AssetPredictedDatapointService;
import org.openremote.manager.gateway.GatewayService;
import org.openremote.model.Container;
import org.openremote.model.ContainerService;
import org.openremote.model.PersistenceEvent;
import org.openremote.model.asset.Asset;
import org.openremote.model.attribute.Attribute;
import org.openremote.model.attribute.AttributeMap;
import org.openremote.model.attribute.AttributeRef;
import org.openremote.model.datapoint.AssetDatapoint;
import org.openremote.model.datapoint.Datapoint;
import org.openremote.model.datapoint.ValueDatapoint;
import org.openremote.model.query.AssetQuery;
import org.openremote.model.query.filter.AttributePredicate;
import org.openremote.model.query.filter.NameValuePredicate;
import org.openremote.model.query.filter.StringPredicate;
import org.openremote.model.query.filter.ValuePredicate;
import org.openremote.model.util.TextUtil;
import org.openremote.model.value.ForecastConfiguration;
import org.openremote.model.value.ForecastConfigurationWeightedExponentialAverage;
import org.openremote.model.value.MetaItemType;
import org.openremote.model.value.NameHolder;

public class ForecastService
extends RouteBuilder
implements ContainerService {
    private static final Logger LOG = Logger.getLogger(ForecastService.class.getName());
    private static long STOP_TIMEOUT = Duration.ofSeconds(5L).toMillis();
    protected TimerService timerService;
    protected GatewayService gatewayService;
    protected AssetStorageService assetStorageService;
    protected AssetDatapointService assetDatapointService;
    protected PersistenceService persistenceService;
    protected AssetPredictedDatapointService assetPredictedDatapointService;
    protected ScheduledExecutorService scheduledExecutorService;
    protected ForecastTaskManager forecastTaskManager = new ForecastTaskManager();

    public void init(Container container) throws Exception {
        this.timerService = (TimerService)container.getService(TimerService.class);
        this.gatewayService = (GatewayService)container.getService(GatewayService.class);
        this.assetStorageService = (AssetStorageService)container.getService(AssetStorageService.class);
        this.assetDatapointService = (AssetDatapointService)container.getService(AssetDatapointService.class);
        this.persistenceService = (PersistenceService)container.getService(PersistenceService.class);
        this.assetPredictedDatapointService = (AssetPredictedDatapointService)container.getService(AssetPredictedDatapointService.class);
        this.scheduledExecutorService = container.getScheduledExecutor();
    }

    public void start(Container container) throws Exception {
        ((MessageBrokerService)container.getService(MessageBrokerService.class)).getContext().addRoutes((RoutesBuilder)this);
        LOG.fine("Loading forecast asset attributes...");
        List<Asset<?>> assets = this.getForecastAssets();
        Set<ForecastAttribute> forecastAttributes = assets.stream().flatMap(asset -> asset.getAttributes().stream().filter(attr -> {
            if (attr.hasMeta(MetaItemType.FORECAST)) {
                Optional forecastConfig = attr.getMetaValue(MetaItemType.FORECAST);
                return forecastConfig.isPresent() && "wea".equals(((ForecastConfiguration)forecastConfig.get()).getType());
            }
            return false;
        }).map(attr -> new ForecastAttribute((Asset<?>)asset, (Attribute<?>)attr))).collect(Collectors.toSet());
        LOG.fine("Found forecast asset attributes count  = " + forecastAttributes.size());
        this.forecastTaskManager.init(forecastAttributes);
    }

    public void stop(Container container) throws Exception {
        this.forecastTaskManager.stop(STOP_TIMEOUT);
    }

    public void configure() throws Exception {
        this.from("seda://PersistenceTopic?multipleConsumers=true&concurrentConsumers=1&waitForTaskToComplete=NEVER&purgeWhenStopping=true&discardIfNoConsumers=true&size=25000").routeId("Persistence-ForecastConfiguration").filter(PersistenceService.isPersistenceEventForEntityType(Asset.class)).filter(GatewayService.isNotForGateway(this.gatewayService)).process(exchange -> {
            PersistenceEvent persistenceEvent = (PersistenceEvent)exchange.getIn().getBody(PersistenceEvent.class);
            this.processAssetChange(persistenceEvent);
        });
    }

    protected List<Asset<?>> getForecastAssets() {
        return this.assetStorageService.findAll(new AssetQuery().attributes(new AttributePredicate[]{new AttributePredicate().meta(new NameValuePredicate[]{new NameValuePredicate((NameHolder)MetaItemType.FORECAST, (ValuePredicate)new StringPredicate(AssetQuery.Match.CONTAINS, true, "type"))})}));
    }

    protected void processAssetChange(PersistenceEvent<Asset<?>> persistenceEvent) {
        Asset asset = (Asset)persistenceEvent.getEntity();
        Set<ForecastAttribute> forecastAttributes = null;
        switch (persistenceEvent.getCause()) {
            case CREATE: {
                forecastAttributes = asset.getAttributes().stream().filter(attr -> {
                    if (attr.hasMeta(MetaItemType.FORECAST)) {
                        Optional forecastConfig = attr.getMetaValue(MetaItemType.FORECAST);
                        return forecastConfig.isPresent() && "wea".equals(((ForecastConfiguration)forecastConfig.get()).getType());
                    }
                    return false;
                }).map(attr -> new ForecastAttribute((Asset<?>)asset, (Attribute<?>)attr)).collect(Collectors.toSet());
                this.forecastTaskManager.add(forecastAttributes);
                break;
            }
            case UPDATE: {
                if (persistenceEvent.getPropertyNames() == null || persistenceEvent.getPropertyNames().indexOf("attributes") < 0) {
                    return;
                }
                List oldAttributes = ((AttributeMap)persistenceEvent.getPreviousState("attributes")).stream().filter(attr -> attr.hasMeta(MetaItemType.FORECAST)).collect(Collectors.toList());
                List newAttributes = ((AttributeMap)persistenceEvent.getCurrentState("attributes")).stream().filter(attr -> attr.hasMeta(MetaItemType.FORECAST)).collect(Collectors.toList());
                List newOrModifiedAttributes = Attribute.getAddedOrModifiedAttributes(oldAttributes, newAttributes).collect(Collectors.toList());
                Set<ForecastAttribute> attributesToDelete = newOrModifiedAttributes.stream().map(attr -> new ForecastAttribute((Asset<?>)asset, (Attribute<?>)attr)).filter(attr -> this.forecastTaskManager.containsAttribute((ForecastAttribute)attr)).collect(Collectors.toSet());
                attributesToDelete.addAll(oldAttributes.stream().filter(oldAttr -> newAttributes.stream().filter(newAttr -> newAttr.getName().equals(oldAttr.getName())).count() == 0L).map(attr -> new ForecastAttribute((Asset<?>)asset, (Attribute<?>)attr)).toList());
                this.forecastTaskManager.delete(attributesToDelete);
                forecastAttributes = newOrModifiedAttributes.stream().filter(attr -> {
                    if (attr.hasMeta(MetaItemType.FORECAST)) {
                        Optional forecastConfig = attr.getMetaValue(MetaItemType.FORECAST);
                        return forecastConfig.isPresent() && "wea".equals(((ForecastConfiguration)forecastConfig.get()).getType());
                    }
                    return false;
                }).map(attr -> new ForecastAttribute((Asset<?>)asset, (Attribute<?>)attr)).collect(Collectors.toSet());
                this.forecastTaskManager.add(forecastAttributes);
                break;
            }
            case DELETE: {
                forecastAttributes = asset.getAttributes().stream().filter(attr -> attr.hasMeta(MetaItemType.FORECAST)).map(attr -> new ForecastAttribute((Asset<?>)asset, (Attribute<?>)attr)).collect(Collectors.toSet());
                this.forecastTaskManager.delete(forecastAttributes);
            }
        }
    }

    private class ForecastTaskManager {
        private static long DELAY_MIN_TO_CANCEL_SAFELY = Duration.ofSeconds(2L).toMillis();
        private static long DEFAULT_SCHEDULE_DELAY = Duration.ofMinutes(15L).toMillis();
        protected ScheduledFuture<?> scheduledFuture;
        protected Map<ForecastAttribute, Long> nextForecastCalculationMap = new HashMap<ForecastAttribute, Long>();
        protected Set<ForecastAttribute> forecastAttributes = new HashSet<ForecastAttribute>();

        private ForecastTaskManager() {
        }

        public synchronized void init(Set<ForecastAttribute> attributes) {
            if (attributes == null || attributes.size() == 0) {
                return;
            }
            long now = ForecastService.this.timerService.getCurrentTimeMillis();
            attributes.forEach(attr -> {
                if (attr.isValidConfig()) {
                    attr.setForecastTimestamps(this.loadForecastTimestampsFromDb(attr.getAttributeRef(), now));
                    this.forecastAttributes.add((ForecastAttribute)attr);
                }
            });
            this.start(now, true);
        }

        public synchronized void add(Set<ForecastAttribute> attributes) {
            if (attributes == null || attributes.size() == 0) {
                return;
            }
            attributes.forEach(attr -> {
                if (attr.isValidConfig()) {
                    LOG.fine("Adding asset attribute to forecast calculation service: " + String.valueOf(attr.getAttributeRef()));
                    this.forecastAttributes.add((ForecastAttribute)attr);
                }
            });
            long now = ForecastService.this.timerService.getCurrentTimeMillis();
            if (this.scheduledFuture != null) {
                if (this.scheduledFuture.getDelay(TimeUnit.MILLISECONDS) > DELAY_MIN_TO_CANCEL_SAFELY) {
                    this.scheduledFuture.cancel(false);
                    this.scheduledFuture = null;
                    this.start(now);
                }
            } else {
                this.start(now);
            }
        }

        public synchronized void delete(Set<ForecastAttribute> attributes) {
            attributes.forEach(attr -> this.delete((ForecastAttribute)attr));
        }

        public synchronized void delete(ForecastAttribute attribute) {
            LOG.fine("Removing asset attribute from forecast calculation service: " + String.valueOf(attribute.getAttributeRef()));
            this.nextForecastCalculationMap.remove(attribute);
            this.forecastAttributes.remove(attribute);
            ForecastService.this.assetPredictedDatapointService.purgeValues(attribute.getAttributeRef().getId(), attribute.getAttributeRef().getName());
        }

        public synchronized boolean containsAttribute(ForecastAttribute attribute) {
            return this.forecastAttributes.contains(attribute);
        }

        public synchronized boolean containsAttribute(AttributeRef attributeRef) {
            return this.forecastAttributes.stream().filter(attr -> attr.getAttributeRef().equals((Object)attributeRef)).findFirst().isPresent();
        }

        public synchronized ForecastAttribute getAttribute(AttributeRef attributeRef) {
            return this.forecastAttributes.stream().filter(attr -> attr.getAttributeRef().equals((Object)attributeRef)).findFirst().orElse(null);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean stop(long timeout) {
            long start = ForecastService.this.timerService.getCurrentTimeMillis();
            while (true) {
                ForecastTaskManager forecastTaskManager = this;
                synchronized (forecastTaskManager) {
                    if (this.scheduledFuture == null) {
                        return true;
                    }
                    if (this.scheduledFuture != null && this.scheduledFuture.getDelay(TimeUnit.MILLISECONDS) > DELAY_MIN_TO_CANCEL_SAFELY) {
                        this.scheduledFuture.cancel(false);
                        this.scheduledFuture = null;
                        return true;
                    }
                    if (ForecastService.this.timerService.getCurrentTimeMillis() - start > timeout) {
                        this.scheduledFuture.cancel(true);
                        this.scheduledFuture = null;
                        return false;
                    }
                }
                try {
                    Thread.currentThread();
                    Thread.sleep(300L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return false;
                }
            }
        }

        private synchronized void start(long now) {
            this.start(now, false);
        }

        private synchronized void start(long now, boolean isServerRestart) {
            if (this.scheduledFuture == null) {
                this.addForecastTimestamps(now, isServerRestart);
                this.updateNextForecastCalculationMap();
                this.scheduleForecastCalculation(now, Optional.empty());
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void calculateForecasts() {
            long now = ForecastService.this.timerService.getCurrentTimeMillis();
            ArrayList attributesToCalculate = new ArrayList();
            try {
                ForecastTaskManager forecastTaskManager = this;
                synchronized (forecastTaskManager) {
                    if (Thread.currentThread().isInterrupted()) {
                        return;
                    }
                    this.purgeForecastTimestamps(now);
                    this.addForecastTimestamps(now, false);
                    this.purgeForecastTimestamps(now);
                    this.nextForecastCalculationMap.forEach((attribute, nextForecastCalculationTimestamp) -> {
                        if (nextForecastCalculationTimestamp <= now) {
                            attributesToCalculate.add(attribute);
                        }
                    });
                }
                attributesToCalculate.forEach(attr -> {
                    if (Thread.currentThread().isInterrupted()) {
                        return;
                    }
                    List<Long> forecastTimestamps = attr.getForecastTimestamps();
                    if (forecastTimestamps == null || forecastTimestamps.size() == 0) {
                        return;
                    }
                    if (!(attr.getConfig() instanceof ForecastConfigurationWeightedExponentialAverage)) {
                        return;
                    }
                    ForecastConfigurationWeightedExponentialAverage weaConfig = (ForecastConfigurationWeightedExponentialAverage)attr.getConfig();
                    LOG.fine("Calculating forecast values for attribute: " + String.valueOf(attr.getAttributeRef()));
                    Long offset = forecastTimestamps.get(0) - (now + weaConfig.getForecastPeriod().toMillis());
                    List<List<Long>> allSampleTimestamps = this.calculateSampleTimestamps(weaConfig, offset);
                    List<DatapointBucket> historyDatapointBuckets = this.getHistoryDataFromDb(attr.getAttributeRef(), weaConfig, offset);
                    List<Optional> forecastValues = allSampleTimestamps.stream().map(sampleTimestamps -> {
                        List<AssetDatapoint> sampleDatapoints = this.findSampleDatapoints(historyDatapointBuckets, (List<Long>)sampleTimestamps);
                        if (sampleDatapoints.size() == weaConfig.getPastCount().intValue()) {
                            return this.calculateWeightedExponentialAverage(attr.getAttribute(), sampleDatapoints);
                        }
                        return Optional.empty();
                    }).toList();
                    if (forecastTimestamps.size() >= forecastValues.size()) {
                        List<ValueDatapoint<?>> datapoints = IntStream.range(0, forecastValues.size()).filter(i -> ((Optional)forecastValues.get(i)).isPresent()).mapToObj(i -> new ValueDatapoint(((Long)forecastTimestamps.get(i)).longValue(), (Object)((Number)((Optional)forecastValues.get(i)).get()))).collect(Collectors.toList());
                        ForecastService.this.assetPredictedDatapointService.purgeValues(attr.getId(), attr.getName());
                        if (datapoints.size() > 0) {
                            LOG.fine("Updating forecast values for attribute: " + String.valueOf(attr.getAttributeRef()));
                            ForecastService.this.assetPredictedDatapointService.updateValues(attr.getId(), attr.getName(), datapoints);
                        }
                    }
                });
                forecastTaskManager = this;
                synchronized (forecastTaskManager) {
                    if (Thread.currentThread().isInterrupted()) {
                        return;
                    }
                    this.updateNextForecastCalculationMap();
                    this.scheduleForecastCalculation(ForecastService.this.timerService.getCurrentTimeMillis(), Optional.empty());
                }
            }
            catch (Exception e) {
                LOG.log(Level.SEVERE, "Exception while calculating and updating forecast values", e);
                this.scheduleForecastCalculation(ForecastService.this.timerService.getCurrentTimeMillis(), Optional.of(DEFAULT_SCHEDULE_DELAY));
            }
        }

        private synchronized void scheduleForecastCalculation(long now, Optional<Long> fixedDelay) {
            Optional<Long> delay = fixedDelay;
            if (delay.isEmpty()) {
                delay = this.calculateScheduleDelay(now);
            }
            if (delay.isPresent()) {
                LOG.fine("Scheduling next forecast calculation in '" + String.valueOf(delay.get()) + " [ms]'.");
                this.scheduledFuture = ForecastService.this.scheduledExecutorService.schedule(() -> this.calculateForecasts(), (long)delay.get(), TimeUnit.MILLISECONDS);
            } else {
                this.scheduledFuture = null;
                if (!this.forecastAttributes.isEmpty()) {
                    LOG.fine("Scheduling next forecast calculation in '" + DEFAULT_SCHEDULE_DELAY + " [ms]'.");
                    this.scheduleForecastCalculation(now, Optional.of(DEFAULT_SCHEDULE_DELAY));
                }
            }
        }

        private List<List<Long>> calculateSampleTimestamps(ForecastConfigurationWeightedExponentialAverage config, Long offset) {
            ArrayList<List<Long>> sampleTimestamps = new ArrayList<List<Long>>(config.getForecastCount());
            long now = ForecastService.this.timerService.getCurrentTimeMillis();
            long pastPeriod = config.getPastPeriod().toMillis();
            long forecastPeriod = config.getForecastPeriod().toMillis();
            for (int forecastIndex = 1; forecastIndex <= config.getForecastCount(); ++forecastIndex) {
                ArrayList<Long> timestamps = new ArrayList<Long>(config.getPastCount());
                for (int pastPeriodIndex = config.getPastCount().intValue(); pastPeriodIndex > 0; --pastPeriodIndex) {
                    timestamps.add(now - pastPeriod * (long)pastPeriodIndex + forecastPeriod * (long)forecastIndex + offset);
                }
                sampleTimestamps.add(timestamps);
            }
            return sampleTimestamps;
        }

        private List<Long> calculateForecastTimestamps(long now, ForecastConfigurationWeightedExponentialAverage config) {
            ArrayList<Long> forecastTimestamps = new ArrayList<Long>(config.getForecastCount());
            long forecastPeriod = config.getForecastPeriod().toMillis();
            for (int forecastIndex = 1; forecastIndex <= config.getForecastCount(); ++forecastIndex) {
                forecastTimestamps.add(now + forecastPeriod * (long)forecastIndex);
            }
            return forecastTimestamps;
        }

        private List<AssetDatapoint> findSampleDatapoints(List<DatapointBucket> datapointBuckets, List<Long> sampleTimestamps) {
            ArrayList<AssetDatapoint> sampleDatapoints = new ArrayList<AssetDatapoint>(sampleTimestamps.size());
            for (Long timestamp : sampleTimestamps) {
                AssetDatapoint foundDatapoint = null;
                List datapoints = datapointBuckets.stream().filter(bucket -> bucket.isInTimeRange(timestamp)).findFirst().map(bucket -> bucket.getDatapoints()).orElse(null);
                if (datapoints == null) continue;
                for (AssetDatapoint assetDatapoint : datapoints) {
                    if (assetDatapoint.getTimestamp() <= timestamp) {
                        foundDatapoint = assetDatapoint;
                        continue;
                    }
                    if (assetDatapoint.getTimestamp() <= timestamp) continue;
                    break;
                }
                if (foundDatapoint == null) continue;
                sampleDatapoints.add(foundDatapoint);
            }
            return sampleDatapoints;
        }

        private Optional<Number> calculateWeightedExponentialAverage(Attribute<?> attribute, List<AssetDatapoint> datapoints) {
            List values = datapoints.stream().map(Datapoint::getValue).collect(Collectors.toList());
            double R = datapoints.size();
            double a = 2.0 / (R + 1.0);
            Class clazz = attribute.getTypeClass();
            if (Long.class == clazz || Integer.class == clazz || Short.class == clazz || Byte.class == clazz || Double.class == clazz || Float.class == clazz) {
                Optional<Number> value;
                if (values.size() == 1) {
                    values.add(0, 0.0);
                }
                if ((value = values.stream().map(v -> (Number)v).reduce((olderValue, oldValue) -> oldValue.doubleValue() * a + olderValue.doubleValue() * (1.0 - a))).isPresent()) {
                    value = clazz == Long.class ? Optional.of(value.get().longValue()) : (clazz == Integer.class ? Optional.of(value.get().intValue()) : (clazz == Short.class ? Optional.of(value.get().shortValue()) : (clazz == Byte.class ? Optional.of(value.get().byteValue()) : (clazz == Double.class ? Optional.of(value.get().doubleValue()) : (clazz == Float.class ? Optional.of(Float.valueOf(value.get().floatValue())) : Optional.empty())))));
                }
                return value;
            }
            if (attribute.getTypeClass() == BigDecimal.class) {
                if (values.size() == 1) {
                    values.add(0, BigDecimal.valueOf(0L));
                }
                return values.stream().map(v -> (Number)v).reduce((olderValue, oldValue) -> ((BigDecimal)oldValue).multiply(BigDecimal.valueOf(a)).add(((BigDecimal)olderValue).multiply(BigDecimal.valueOf(1.0 - a))));
            }
            if (attribute.getTypeClass() == BigInteger.class) {
                if (values.size() == 1) {
                    values.add(0, BigInteger.valueOf(0L));
                }
                return values.stream().map(v -> (Number)v).reduce((olderValue, oldValue) -> ((BigInteger)oldValue).multiply(BigInteger.valueOf(2L)).add(((BigInteger)olderValue).multiply(BigInteger.valueOf((long)R - 1L))).divide(BigInteger.valueOf((long)R + 1L)));
            }
            return Optional.empty();
        }

        private void updateNextForecastCalculationMap() {
            this.nextForecastCalculationMap.clear();
            this.forecastAttributes.forEach(attr -> {
                List<Long> timestamps = attr.getForecastTimestamps();
                if (timestamps != null && timestamps.size() > 0) {
                    this.nextForecastCalculationMap.put((ForecastAttribute)attr, timestamps.get(0));
                }
            });
        }

        private Optional<Long> calculateScheduleDelay(long now) {
            OptionalLong calculateForecastTimestamp = this.nextForecastCalculationMap.values().stream().mapToLong(v -> v).min();
            if (calculateForecastTimestamp.isPresent()) {
                long delay = calculateForecastTimestamp.getAsLong() - now;
                return Optional.of(delay < 0L ? 0L : delay);
            }
            return Optional.empty();
        }

        private void addForecastTimestamps(long now, boolean isServerRestart) {
            this.forecastAttributes.forEach(attr -> {
                ForecastConfiguration config = attr.getConfig();
                ForecastConfigurationWeightedExponentialAverage weaConfig = null;
                if (!(config instanceof ForecastConfigurationWeightedExponentialAverage)) {
                    return;
                }
                weaConfig = (ForecastConfigurationWeightedExponentialAverage)config;
                List<Long> newTimestamps = this.calculateForecastTimestamps(now, weaConfig);
                List<Long> oldTimestamps = attr.getForecastTimestamps();
                if (oldTimestamps == null || oldTimestamps.size() == 0) {
                    if (newTimestamps.size() > 0) {
                        newTimestamps.add(0, now);
                    }
                    attr.setForecastTimestamps(newTimestamps);
                } else if (newTimestamps.size() > 0 && oldTimestamps.size() > 0) {
                    long offset = oldTimestamps.get(0) - newTimestamps.get(0);
                    List<Long> newShiftedTimestamps = newTimestamps.stream().map(timestamp -> timestamp + offset).collect(Collectors.toList());
                    while ((Long)newShiftedTimestamps.get(0) < now) {
                        newShiftedTimestamps = newShiftedTimestamps.stream().map(timestamp -> timestamp + ((ForecastConfigurationWeightedExponentialAverage)config).getForecastPeriod().toMillis()).collect(Collectors.toList());
                    }
                    if (isServerRestart && (oldTimestamps.get(0) < now || newTimestamps.size() > oldTimestamps.size())) {
                        newShiftedTimestamps.add(0, now);
                    }
                    attr.setForecastTimestamps(newShiftedTimestamps);
                }
            });
        }

        public void purgeForecastTimestamps(long now) {
            this.forecastAttributes.forEach(attr -> {
                List<Long> timestamps = attr.getForecastTimestamps();
                if (timestamps == null) {
                    return;
                }
                int clearCount = 0;
                int index = Collections.binarySearch(timestamps, now);
                clearCount = index >= 0 ? index + 1 : index * -1 - 1;
                if (clearCount > 0) {
                    if (clearCount == timestamps.size()) {
                        timestamps.clear();
                    } else {
                        try {
                            timestamps.subList(0, clearCount).clear();
                        }
                        catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    }
                }
            });
        }

        private List<DatapointBucket> getHistoryDataFromDb(AttributeRef attributeRef, ForecastConfigurationWeightedExponentialAverage config, long offset) {
            ArrayList<DatapointBucket> datapointBuckets = new ArrayList<DatapointBucket>(config.getPastCount());
            StringBuilder sb = new StringBuilder();
            sb.append("select dp from " + AssetDatapoint.class.getSimpleName() + " dp where dp.assetId = :assetId and dp.attributeName = :attributeName ");
            for (int i = 1; i <= config.getPastCount(); ++i) {
                sb.append(i == 1 ? "and (" : " or ");
                sb.append("(dp.timestamp >= :timestampMin" + i + " and dp.timestamp <= :timestampMax" + i + ")");
                if (i != config.getPastCount()) continue;
                sb.append(") ");
            }
            sb.append("order by dp.timestamp asc");
            List datapoints = (List)ForecastService.this.persistenceService.doReturningTransaction(entityManager -> {
                TypedQuery query = entityManager.createQuery(sb.toString(), AssetDatapoint.class).setParameter("assetId", (Object)attributeRef.getId()).setParameter("attributeName", (Object)attributeRef.getName());
                long now = ForecastService.this.timerService.getCurrentTimeMillis();
                long pastPeriod = config.getPastPeriod().toMillis();
                long forecastPeriod = config.getForecastPeriod().toMillis();
                long totalForecastPeriod = Math.min(forecastPeriod * (long)config.getForecastCount().intValue(), pastPeriod);
                for (int i = config.getPastCount().intValue(); i >= 1; --i) {
                    long timestampMin = now - pastPeriod * (long)i + offset;
                    long timestampMax = now - pastPeriod * (long)i + totalForecastPeriod + offset;
                    datapointBuckets.add(new DatapointBucket(timestampMin, timestampMax));
                    query.setParameter("timestampMin" + i, (Object)new Date(timestampMin));
                    query.setParameter("timestampMax" + i, (Object)new Date(timestampMax));
                }
                return query.getResultList();
            });
            datapoints.forEach(datapoint -> datapointBuckets.stream().filter(bucket -> bucket.isInTimeRange(datapoint.getTimestamp())).findFirst().ifPresent(bucket -> bucket.add((AssetDatapoint)datapoint)));
            return datapointBuckets;
        }

        private List<Long> loadForecastTimestampsFromDb(AttributeRef attributeRef, long now) {
            List<ValueDatapoint> datapoints = ForecastService.this.assetPredictedDatapointService.getDatapoints(attributeRef);
            List<Long> timestamps = datapoints.stream().map(ValueDatapoint::getTimestamp).filter(timestamp -> timestamp >= now).sorted().collect(Collectors.toList());
            return timestamps;
        }
    }

    public static class ForecastAttribute {
        private String assetId;
        private AttributeRef attributeRef;
        private Attribute<?> attribute;
        private ForecastConfiguration config;
        private List<Long> forecastTimestamps = new ArrayList<Long>();

        public ForecastAttribute(Asset<?> asset, Attribute<?> attribute) {
            this(asset.getId(), attribute);
        }

        public ForecastAttribute(String assetId, Attribute<?> attribute) {
            TextUtil.requireNonNullAndNonEmpty((String)assetId);
            if (attribute == null) {
                throw new IllegalArgumentException("Attribute cannot be null");
            }
            this.assetId = assetId;
            this.attribute = attribute;
            this.attributeRef = new AttributeRef(assetId, attribute.getName());
            this.config = attribute.getMetaValue(MetaItemType.FORECAST).orElse(null);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ForecastAttribute that = (ForecastAttribute)o;
            return this.assetId.equals(that.assetId) && this.attribute.getName().equals(that.attribute.getName());
        }

        public int hashCode() {
            int result = this.assetId.hashCode();
            result = 31 * result + this.attribute.getName().hashCode();
            return result;
        }

        public String getId() {
            return this.assetId;
        }

        public String getName() {
            return this.attribute.getName();
        }

        public AttributeRef getAttributeRef() {
            return this.attributeRef;
        }

        public Attribute<?> getAttribute() {
            return this.attribute;
        }

        public ForecastConfiguration getConfig() {
            return this.config;
        }

        public boolean isValidConfig() {
            return this.config != null && this.config.isValid();
        }

        public void setForecastTimestamps(List<Long> timestamps) {
            this.forecastTimestamps = timestamps;
        }

        public List<Long> getForecastTimestamps() {
            return this.forecastTimestamps;
        }
    }

    private static class DatapointBucket {
        private long begin;
        private long end;
        private List<AssetDatapoint> datapoints = new ArrayList<AssetDatapoint>();

        public DatapointBucket(long begin, long end) {
            this.begin = begin;
            this.end = end;
        }

        public long getBegin() {
            return this.begin;
        }

        public long getEnd() {
            return this.end;
        }

        public List<AssetDatapoint> getDatapoints() {
            return this.datapoints;
        }

        public boolean isInTimeRange(long timestamp) {
            return timestamp >= this.begin && timestamp <= this.end;
        }

        public void add(AssetDatapoint datapoint) {
            this.datapoints.add(datapoint);
        }
    }
}

