/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.manager.alarm;

import jakarta.persistence.EntityNotFoundException;
import jakarta.persistence.PersistenceException;
import jakarta.persistence.TypedQuery;
import java.sql.PreparedStatement;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.hibernate.Session;
import org.openremote.container.message.MessageBrokerService;
import org.openremote.container.persistence.PersistenceService;
import org.openremote.container.timer.TimerService;
import org.openremote.container.util.MapAccess;
import org.openremote.manager.alarm.AlarmResourceImpl;
import org.openremote.manager.event.ClientEventService;
import org.openremote.manager.notification.NotificationService;
import org.openremote.manager.security.ManagerIdentityProvider;
import org.openremote.manager.security.ManagerIdentityService;
import org.openremote.manager.web.ManagerWebService;
import org.openremote.model.Container;
import org.openremote.model.ContainerService;
import org.openremote.model.PersistenceEvent;
import org.openremote.model.alarm.Alarm;
import org.openremote.model.alarm.AlarmAssetLink;
import org.openremote.model.alarm.AlarmEvent;
import org.openremote.model.alarm.SentAlarm;
import org.openremote.model.event.shared.EventFilter;
import org.openremote.model.event.shared.EventSubscription;
import org.openremote.model.event.shared.RealmFilter;
import org.openremote.model.notification.AbstractNotificationMessage;
import org.openremote.model.notification.EmailNotificationMessage;
import org.openremote.model.notification.Notification;
import org.openremote.model.notification.PushNotificationAction;
import org.openremote.model.notification.PushNotificationMessage;
import org.openremote.model.query.UserQuery;
import org.openremote.model.query.filter.RealmPredicate;
import org.openremote.model.query.filter.StringPredicate;
import org.openremote.model.security.User;
import org.openremote.model.util.TextUtil;

public class AlarmService
extends RouteBuilder
implements ContainerService {
    public static final Logger LOG = Logger.getLogger(AlarmService.class.getName());
    private Container container;
    private ClientEventService clientEventService;
    private ManagerIdentityService identityService;
    private NotificationService notificationService;
    private PersistenceService persistenceService;
    private TimerService timerService;

    public int getPriority() {
        return 1000;
    }

    public void init(Container container) throws Exception {
        this.container = container;
        this.clientEventService = (ClientEventService)container.getService(ClientEventService.class);
        this.identityService = (ManagerIdentityService)container.getService(ManagerIdentityService.class);
        this.notificationService = (NotificationService)container.getService(NotificationService.class);
        this.persistenceService = (PersistenceService)container.getService(PersistenceService.class);
        this.timerService = (TimerService)container.getService(TimerService.class);
        ((MessageBrokerService)container.getService(MessageBrokerService.class)).getContext().addRoutes((RoutesBuilder)this);
        ((ManagerWebService)container.getService(ManagerWebService.class)).addApiSingleton((Object)new AlarmResourceImpl(this.timerService, this.identityService, this));
        this.clientEventService.addSubscriptionAuthorizer((realm, authContext, eventSubscription) -> {
            if (!eventSubscription.isEventType(AlarmEvent.class) || authContext == null) {
                return false;
            }
            if (!authContext.isSuperUser()) {
                EventSubscription subscription = eventSubscription;
                subscription.setFilter((EventFilter)new RealmFilter(authContext.getAuthenticatedRealmName()));
            }
            return true;
        });
    }

    public void start(Container container) throws Exception {
    }

    public void stop(Container container) throws Exception {
    }

    public void configure() throws Exception {
    }

    protected Set<String> getAlarmRealms(List<SentAlarm> alarms) {
        return alarms == null ? Set.of() : alarms.stream().map(SentAlarm::getRealm).collect(Collectors.toSet());
    }

    protected void validateAlarmId(Long alarmId) {
        if (alarmId == null) {
            throw new IllegalArgumentException("Missing alarm ID");
        }
        if (alarmId < 0L) {
            throw new IllegalArgumentException("Alarm ID cannot be negative");
        }
    }

    protected void validateAlarmIds(Collection<Long> alarmIds) {
        if (alarmIds == null || alarmIds.isEmpty()) {
            throw new IllegalArgumentException("Missing alarm IDs");
        }
        alarmIds.forEach(this::validateAlarmId);
    }

    protected void validateAssetIds(Collection<String> assetIds) {
        if (assetIds == null || assetIds.isEmpty()) {
            throw new IllegalArgumentException("Missing asset IDs");
        }
        assetIds.forEach(assetId -> {
            if (TextUtil.isNullOrEmpty((String)assetId)) {
                throw new IllegalArgumentException("Missing asset ID");
            }
        });
    }

    public SentAlarm sendAlarm(Alarm alarm, List<String> assetIds) {
        Objects.requireNonNull(alarm, "Alarm cannot be null");
        Objects.requireNonNull(alarm.getRealm(), "Alarm realm cannot be null");
        Objects.requireNonNull(alarm.getTitle(), "Alarm title cannot be null");
        Objects.requireNonNull(alarm.getSeverity(), "Alarm severity cannot be null");
        Objects.requireNonNull(alarm.getSource(), "Source cannot be null");
        Objects.requireNonNull(alarm.getSourceId(), "Source ID cannot be null");
        Date timestamp = new Date(this.timerService.getCurrentTimeMillis());
        SentAlarm sentAlarm = (SentAlarm)this.persistenceService.doReturningTransaction(entityManager -> (SentAlarm)entityManager.merge((Object)new SentAlarm().setAssigneeId(alarm.getAssigneeId()).setRealm(alarm.getRealm()).setTitle(alarm.getTitle()).setContent(alarm.getContent()).setSeverity(alarm.getSeverity()).setStatus(alarm.getStatus()).setSource(alarm.getSource()).setSourceId(alarm.getSourceId()).setCreatedOn(timestamp).setLastModified(timestamp)));
        if (assetIds != null && !assetIds.isEmpty()) {
            this.linkAssets(assetIds, sentAlarm.getRealm(), sentAlarm.getId());
        }
        this.clientEventService.publishEvent(new AlarmEvent(alarm.getRealm(), PersistenceEvent.Cause.CREATE));
        if (alarm.getSeverity() == Alarm.Severity.HIGH) {
            Set<String> excludeUserIds = alarm.getSource() == Alarm.Source.MANUAL ? Set.of(alarm.getSourceId()) : Set.of();
            this.sendAssigneeNotification(sentAlarm, excludeUserIds);
        }
        return sentAlarm;
    }

    protected void sendAssigneeNotification(SentAlarm alarm, Set<String> excludeUserIds) {
        List<User> users = this.getAlarmNotificationUsers(alarm);
        users.removeIf(user -> excludeUserIds.contains(user.getId()));
        if (users.isEmpty()) {
            LOG.fine("No matching users to send alarm notification");
            return;
        }
        LOG.fine("Sending alarm notification to " + users.size() + " matching user(s)");
        String title = String.format("Alarm: %s - %s", alarm.getSeverity(), alarm.getTitle());
        String url = this.getAlarmNotificationUrl(alarm);
        Map<String, String> content = this.getAlarmNotificationContent(alarm, url);
        Notification email = new Notification().setName("New Alarm").setMessage((AbstractNotificationMessage)new EmailNotificationMessage().setHtml(this.getAlarmNotificationHtml(content)).setSubject(title).setTo(users.stream().filter(user -> user.getEmail() != null && !user.getEmail().isBlank()).map(user -> new EmailNotificationMessage.Recipient(user.getFullName(), user.getEmail())).toList()));
        Notification push = new Notification();
        push.setName("New Alarm").setMessage((AbstractNotificationMessage)new PushNotificationMessage().setTitle(title).setBody(this.getAlarmNotificationText(content)).setAction(url == null ? null : new PushNotificationAction(url))).setTargets(users.stream().map(user -> new Notification.Target(Notification.TargetType.USER, user.getId())).toList());
        this.notificationService.sendNotificationAsync(push, Notification.Source.INTERNAL, "alarms");
        this.notificationService.sendNotificationAsync(email, Notification.Source.INTERNAL, "alarms");
    }

    private List<User> getAlarmNotificationUsers(SentAlarm alarm) {
        ManagerIdentityProvider identityProvider = this.identityService.getIdentityProvider();
        ArrayList<User> users = new ArrayList<User>();
        if (alarm.getAssigneeId() == null) {
            UserQuery userQuery = new UserQuery().realm(new RealmPredicate(alarm.getRealm())).clientRoles(new StringPredicate[]{new StringPredicate("write:alarms")}).realmRoles(new StringPredicate[]{new StringPredicate("admin")}).serviceUsers(Boolean.valueOf(false));
            users.addAll(Arrays.asList(identityProvider.queryUsers(userQuery)));
        } else {
            users.add(identityProvider.getUser(alarm.getAssigneeId()));
        }
        return users;
    }

    private String getAlarmNotificationUrl(SentAlarm alarm) {
        String defaultHostname = MapAccess.getString((Map)this.container.getConfig(), (String)"OR_HOSTNAME", null);
        return defaultHostname == null ? null : String.format("https://%s/manager/#/alarms/%s?realm=%s", defaultHostname, alarm.getId(), alarm.getRealm());
    }

    private Map<String, String> getAlarmNotificationContent(SentAlarm alarm, String url) {
        LinkedHashMap<String, String> result = new LinkedHashMap<String, String>();
        result.put("Title", alarm.getTitle());
        result.put("Content", alarm.getContent());
        result.put("Created", DateFormat.getDateTimeInstance().format(alarm.getCreatedOn()));
        result.put("Source", alarm.getSource().name());
        result.put("Severity", alarm.getSeverity().name());
        result.put("Status", alarm.getStatus().name());
        List<String> assetLinks = this.getAssetLinks(alarm.getId(), alarm.getRealm()).stream().map(AlarmAssetLink::getAssetName).toList();
        result.put("Linked assets", assetLinks.isEmpty() ? "None" : String.join((CharSequence)", ", assetLinks));
        result.put("Assignee", TextUtil.isNullOrEmpty((String)alarm.getAssigneeUsername()) ? "None" : alarm.getAssigneeUsername());
        if (url != null) {
            result.put("URL", url);
        }
        return result;
    }

    private String getAlarmNotificationHtml(Map<String, String> content) {
        StringBuilder sb = new StringBuilder("<html><head><style>td {vertical-align: top;}</style></head><body><table>");
        content.forEach((key, value) -> sb.append(String.format("<tr><td><b>%s</b></td><td>%s</td></tr>", key, value.replaceAll("\n", "<br>"))));
        sb.append("</table><body></html>");
        return sb.toString();
    }

    private String getAlarmNotificationText(Map<String, String> content) {
        StringBuilder sb = new StringBuilder();
        content.forEach((key, value) -> sb.append(String.format("%s: %s\n", key, value)));
        return sb.toString();
    }

    public void updateAlarm(SentAlarm oldAlarm, SentAlarm newAlarm) {
        String oldAssigneeId = oldAlarm.getAssigneeId();
        String newAssigneeId = newAlarm.getAssigneeId();
        this.persistenceService.doTransaction(entityManager -> entityManager.createQuery("update SentAlarm set title=:title, content=:content, severity=:severity, status=:status, lastModified=:lastModified, assigneeId=:assigneeId\nwhere id =:id\n").setParameter("id", (Object)oldAlarm.getId()).setParameter("title", (Object)newAlarm.getTitle()).setParameter("content", (Object)newAlarm.getContent()).setParameter("severity", (Object)newAlarm.getSeverity()).setParameter("status", (Object)newAlarm.getStatus()).setParameter("lastModified", (Object)new Timestamp(this.timerService.getCurrentTimeMillis())).setParameter("assigneeId", (Object)newAssigneeId).executeUpdate());
        this.clientEventService.publishEvent(new AlarmEvent(newAlarm.getRealm(), PersistenceEvent.Cause.UPDATE));
        if (newAlarm.getSeverity() == Alarm.Severity.HIGH) {
            Set<String> excludeUserIds = Stream.of(oldAssigneeId).filter(Objects::nonNull).collect(Collectors.toSet());
            this.sendAssigneeNotification(this.getAlarm(oldAlarm.getId()), excludeUserIds);
        }
    }

    public void linkAssets(List<String> assetIds, String realm, Long alarmId) {
        this.linkAssets(assetIds.stream().map(assetId -> new AlarmAssetLink(realm, alarmId, assetId)).toList());
    }

    public void linkAssets(List<AlarmAssetLink> links) {
        Set<Long> alarmIds = links.stream().map(link -> link.getId().getAlarmId()).collect(Collectors.toSet());
        this.validateAlarmIds(alarmIds);
        Set<String> assetIds = links.stream().map(link -> link.getId().getAssetId()).collect(Collectors.toSet());
        this.validateAssetIds(assetIds);
        this.persistenceService.doTransaction(entityManager -> ((Session)entityManager.unwrap(Session.class)).doWork(connection -> {
            PreparedStatement st = connection.prepareStatement("insert into ALARM_ASSET_LINK (sentalarm_id, realm, asset_id, created_on) values (?, ?, ?, ?)\non conflict (sentalarm_id, realm, asset_id) do nothing\n");
            for (AlarmAssetLink link : links) {
                st.setLong(1, link.getId().getAlarmId());
                st.setString(2, link.getId().getRealm());
                st.setString(3, link.getId().getAssetId());
                st.setTimestamp(4, new Timestamp(this.timerService.getCurrentTimeMillis()));
                st.addBatch();
            }
            st.executeBatch();
        }));
    }

    public List<AlarmAssetLink> getAssetLinks(Long alarmId, String realm) throws IllegalArgumentException {
        return (List)this.persistenceService.doReturningTransaction(entityManager -> entityManager.createQuery("select aal from AlarmAssetLink aal\nwhere aal.id.realm = :realm and aal.id.sentalarmId = :alarmId\norder by aal.createdOn desc\n", AlarmAssetLink.class).setParameter("realm", (Object)realm).setParameter("alarmId", (Object)alarmId).getResultList());
    }

    public SentAlarm getAlarm(Long alarmId) throws IllegalArgumentException {
        SentAlarm alarm;
        this.validateAlarmId(alarmId);
        try {
            alarm = (SentAlarm)this.persistenceService.doReturningTransaction(entityManager -> (SentAlarm)entityManager.createQuery("select sa from SentAlarm sa where sa.id = :id", SentAlarm.class).setParameter("id", (Object)alarmId).getSingleResult());
        }
        catch (PersistenceException e) {
            alarm = null;
        }
        if (alarm == null) {
            throw new EntityNotFoundException("Alarm does not exist");
        }
        return alarm;
    }

    public List<SentAlarm> getAlarms(List<Long> alarmIds) throws IllegalArgumentException {
        List alarms;
        this.validateAlarmIds(alarmIds);
        try {
            alarms = (List)this.persistenceService.doReturningTransaction(entityManager -> entityManager.createQuery("select sa from SentAlarm sa where sa.id in :ids", SentAlarm.class).setParameter("ids", (Object)alarmIds).getResultList());
        }
        catch (PersistenceException e) {
            alarms = null;
        }
        if (alarms == null || alarmIds.size() != alarms.size()) {
            throw new EntityNotFoundException("One or more alarms do not exist");
        }
        return alarms;
    }

    public List<SentAlarm> getAlarms(String realm, Alarm.Status status, String assetId, String assigneeId) throws IllegalArgumentException {
        HashMap<String, String> parameters = new HashMap<String, String>();
        StringBuilder sb = new StringBuilder("select sa from SentAlarm sa ");
        if (assetId != null) {
            sb.append("join AlarmAssetLink aal on sa.id = aal.id.sentalarmId where sa.realm = :realm and aal.id.assetId = :assetId ");
            parameters.put("assetId", assetId);
        } else {
            sb.append("where sa.realm = :realm ");
        }
        parameters.put("realm", realm);
        if (status != null) {
            sb.append("and sa.status = :status ");
            parameters.put("status", (String)status);
        }
        if (assigneeId != null) {
            sb.append("and sa.assigneeId = :assigneeId ");
            parameters.put("assigneeId", assigneeId);
        }
        sb.append("order by sa.createdOn desc");
        return (List)this.persistenceService.doReturningTransaction(entityManager -> {
            TypedQuery query = entityManager.createQuery(sb.toString(), SentAlarm.class);
            parameters.forEach((arg_0, arg_1) -> ((TypedQuery)query).setParameter(arg_0, arg_1));
            return query.getResultList();
        });
    }

    public void removeAlarm(SentAlarm alarm) {
        this.persistenceService.doTransaction(entityManager -> entityManager.createQuery("delete SentAlarm where id = :id").setParameter("id", (Object)alarm.getId()).executeUpdate());
        this.clientEventService.publishEvent(new AlarmEvent(alarm.getRealm(), PersistenceEvent.Cause.DELETE));
    }

    public void removeAlarms(List<SentAlarm> alarms, List<Long> alarmIds) throws IllegalArgumentException {
        this.persistenceService.doTransaction(entityManager -> entityManager.createQuery("delete from SentAlarm sa where sa.id in :ids").setParameter("ids", (Object)alarmIds).executeUpdate());
        this.getAlarmRealms(alarms).forEach(realm -> this.clientEventService.publishEvent(new AlarmEvent(realm, PersistenceEvent.Cause.DELETE)));
    }
}

