/*
 * Decompiled with CFR 0.152.
 */
package io.openraven.magpie.core.dmap.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.openraven.magpie.core.dmap.Util;
import io.openraven.magpie.core.dmap.client.DMapMLClient;
import io.openraven.magpie.core.dmap.client.dto.AppProbability;
import io.openraven.magpie.core.dmap.client.dto.DMapLambdaResponse;
import io.openraven.magpie.core.dmap.client.dto.DmapLambdaRequest;
import io.openraven.magpie.core.dmap.dto.DMapFingerprints;
import io.openraven.magpie.core.dmap.dto.DMapScanResult;
import io.openraven.magpie.core.dmap.dto.FingerprintAnalysis;
import io.openraven.magpie.core.dmap.dto.LambdaDetails;
import io.openraven.magpie.core.dmap.exception.DMapProcessingException;
import io.openraven.magpie.core.dmap.model.EC2Target;
import io.openraven.magpie.core.dmap.model.VpcConfig;
import io.openraven.magpie.core.dmap.service.DMapLambdaService;
import java.nio.charset.Charset;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.internal.http.loader.DefaultSdkHttpClientBuilder;
import software.amazon.awssdk.http.SdkHttpClient;
import software.amazon.awssdk.http.SdkHttpConfigurationOption;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.iam.IamClient;
import software.amazon.awssdk.services.iam.IamClientBuilder;
import software.amazon.awssdk.services.iam.model.AttachedPolicy;
import software.amazon.awssdk.services.iam.model.CreatePolicyResponse;
import software.amazon.awssdk.services.iam.model.CreateRoleRequest;
import software.amazon.awssdk.services.iam.model.CreateRoleResponse;
import software.amazon.awssdk.services.iam.model.EntityAlreadyExistsException;
import software.amazon.awssdk.services.iam.model.GetRoleResponse;
import software.amazon.awssdk.services.iam.model.NoSuchEntityException;
import software.amazon.awssdk.services.iam.model.Role;
import software.amazon.awssdk.services.lambda.LambdaClient;
import software.amazon.awssdk.services.lambda.LambdaClientBuilder;
import software.amazon.awssdk.services.lambda.model.CreateFunctionRequest;
import software.amazon.awssdk.services.lambda.model.CreateFunctionResponse;
import software.amazon.awssdk.services.lambda.model.DeleteFunctionRequest;
import software.amazon.awssdk.services.lambda.model.InvocationType;
import software.amazon.awssdk.services.lambda.model.InvokeRequest;
import software.amazon.awssdk.services.lambda.model.InvokeResponse;
import software.amazon.awssdk.services.lambda.model.Runtime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.NamedThreadFactory;
import software.amazon.awssdk.utils.StringUtils;

public class DMapLambdaServiceImpl
implements DMapLambdaService {
    private static final Logger LOGGER = LoggerFactory.getLogger(DMapLambdaServiceImpl.class);
    private static final Duration FIFTEEN_MINUTES = Duration.ofMinutes(15L);
    private static final String DMAP_LAMBDA_JAR_PATH = "lambda/source/dmap-lambda-0.0.0.jar";
    private static final String ROLE_NAME = "openraven-dmap-scan-role";
    private static final String POLICY_NAME = "openraven-dmap-scan-policy";
    private static final String DMAP_LAMBDA_HANDLER = "io.openraven.dmap.lambda.Scan::handleRequest";
    private static final Set<String> SQUEEZED_FIELDS = Set.of("tds_prelogin_response", "tns_ping_response", "mongo_db", "db2_response", "oracle_nosql", "vertica", "etcd");
    private static final int TIMEOUT = 900;
    private static final int MEMORY_SIZE = 512;
    private static final int SYNC_TIMEOUT = 9000;
    private static final int THREADS_PER_LAMBDA = 5;
    private final ObjectMapper mapper;
    private final DMapMLClient dMapMLClient;
    private final ExecutorService lambdaExecutorService;
    private final ExecutorService ec2targetExecutorService;
    private final List<LambdaDetails> registeredLambdas = new CopyOnWriteArrayList<LambdaDetails>();

    public DMapLambdaServiceImpl(DMapMLClient dMapMLClient, ObjectMapper mapper, int workers) {
        this.mapper = mapper;
        this.dMapMLClient = dMapMLClient;
        this.lambdaExecutorService = Executors.newFixedThreadPool(workers, (ThreadFactory)new NamedThreadFactory(Executors.defaultThreadFactory(), "lamb-pool-thread"));
        this.ec2targetExecutorService = Executors.newFixedThreadPool(workers * 5, (ThreadFactory)new NamedThreadFactory(Executors.defaultThreadFactory(), "serv-pool-thread"));
    }

    @Override
    public DMapScanResult startDMapScan(Map<VpcConfig, List<EC2Target>> vpcGroups) {
        Instant startTime = Instant.now();
        String roleArn = this.createRequiredRole();
        List<FingerprintAnalysis> fingerprintAnalysis = this.analyzeTargetSerivces(vpcGroups, roleArn);
        LOGGER.debug("DMap predictions: {}", fingerprintAnalysis);
        Duration duration = Duration.between(startTime, Instant.now());
        return new DMapScanResult(fingerprintAnalysis, Date.from(startTime), duration);
    }

    @Override
    public void cleanupCreatedResources() {
        LOGGER.info("Cleanup created resources after DMap Lambda execution");
        try (IamClient iam = (IamClient)((IamClientBuilder)IamClient.builder().region(Region.AWS_GLOBAL)).build();){
            List attachedPolicies = iam.listAttachedRolePolicies(builder -> builder.roleName(ROLE_NAME)).attachedPolicies();
            attachedPolicies.forEach(attachedPolicy -> {
                iam.detachRolePolicy(builder -> builder.roleName(ROLE_NAME).policyArn(attachedPolicy.policyArn()));
                iam.deletePolicy(builder -> builder.policyArn(attachedPolicy.policyArn()));
                LOGGER.info("Policy: {} has been removed", (Object)attachedPolicy.policyArn());
            });
            iam.deleteRole(builder -> builder.roleName(ROLE_NAME).build());
            LOGGER.info("Role: {} has been removed", (Object)ROLE_NAME);
        }
        catch (NoSuchEntityException e) {
            LOGGER.info("DMap Lambda related resources not found. Assume stack clean");
            LOGGER.debug("Exception: ", (Throwable)e);
        }
        this.registeredLambdas.stream().collect(Collectors.groupingBy(LambdaDetails::getRegion, Collectors.mapping(LambdaDetails::getFunctionName, Collectors.toSet()))).forEach((region, lambdas) -> {
            try (LambdaClient lambdaClient = (LambdaClient)((LambdaClientBuilder)LambdaClient.builder().region(Region.of((String)region))).build();){
                lambdas.forEach(lambda -> this.deleteLambda(lambdaClient, (String)lambda));
            }
            catch (Exception e) {
                LOGGER.warn("Unable to delete lambdas in region: {} lambdas: {}", region, lambdas);
            }
        });
    }

    private List<FingerprintAnalysis> analyzeTargetSerivces(Map<VpcConfig, List<EC2Target>> vpcGroups, String lambdaRoleArn) {
        ArrayList<FingerprintAnalysis> dmapAnalysis = new ArrayList<FingerprintAnalysis>();
        CountDownLatch lambdaCountDownLatch = new CountDownLatch(vpcGroups.size());
        vpcGroups.forEach((vpcConfig, ec2Targets) -> this.lambdaExecutorService.submit(() -> {
            try (SdkHttpClient httpClient = this.getSdkHttpClient();
                 LambdaClient lambdaClient = this.getLambdaClient(httpClient, vpcConfig.getRegion());){
                String lambdaName = this.createLambda(lambdaClient, (VpcConfig)vpcConfig, lambdaRoleArn);
                Semaphore bulkheadSemaphore = new Semaphore(5);
                CountDownLatch ec2TargetCountDownLatch = new CountDownLatch(ec2Targets.size());
                ec2Targets.forEach(ec2Target -> {
                    this.acquire(bulkheadSemaphore, (EC2Target)ec2Target);
                    this.ec2targetExecutorService.submit(() -> {
                        LOGGER.info("Starting lambda: {} for : {}", (Object)lambdaName, ec2Target);
                        InvokeRequest request = this.getLambdaRequest(lambdaName, (EC2Target)ec2Target);
                        InvokeResponse response = lambdaClient.invoke(request);
                        DMapLambdaResponse dMapLambdaResponse = this.processResponse((VpcConfig)vpcConfig, response);
                        Map<String, DMapFingerprints> hosts = dMapLambdaResponse.getHosts();
                        LOGGER.debug("Response from lambda {} is {}", (Object)lambdaName, hosts);
                        hosts.values().forEach(fingerprint -> this.registerFingerprint((List<FingerprintAnalysis>)dmapAnalysis, (VpcConfig)vpcConfig, (DMapFingerprints)fingerprint));
                        this.release(bulkheadSemaphore, (EC2Target)ec2Target);
                        ec2TargetCountDownLatch.countDown();
                    });
                });
                this.waitForCompletion(ec2TargetCountDownLatch);
            }
            lambdaCountDownLatch.countDown();
        }));
        this.waitForCompletion(lambdaCountDownLatch);
        this.ec2targetExecutorService.shutdown();
        this.lambdaExecutorService.shutdown();
        return dmapAnalysis;
    }

    private Map<String, String> squeezeFeatures(Map<String, String> signature) {
        String response = signature.getOrDefault("banner_1", "");
        SQUEEZED_FIELDS.stream().filter(f -> response.equals(signature.get(f))).forEach(signature::remove);
        return signature;
    }

    private void registerFingerprint(List<FingerprintAnalysis> dmapAnalysis, VpcConfig vpcConfig, DMapFingerprints fingerprint) {
        FingerprintAnalysis fingerprintAnalysis = new FingerprintAnalysis();
        fingerprintAnalysis.setResourceId(fingerprint.getId());
        fingerprintAnalysis.setRegion(vpcConfig.getRegion());
        fingerprintAnalysis.setAddress(fingerprint.getAddress());
        fingerprint.getSignatures().forEach((port, signature) -> {
            signature = this.squeezeFeatures((Map<String, String>)signature);
            LOGGER.debug("Sending request to OpenRaven::DmapML service to analyze fingerprints by port: {}", port);
            List<AppProbability> predictions = this.dMapMLClient.predict((Map<String, String>)signature);
            fingerprintAnalysis.getPredictionsByPort().put((Integer)port, predictions);
        });
        dmapAnalysis.add(fingerprintAnalysis);
    }

    private String createRequiredRole() {
        try (IamClient iam = (IamClient)((IamClientBuilder)IamClient.builder().region(Region.AWS_GLOBAL)).build();){
            String lambdaRoleArn = this.createLambdaRole(iam);
            String policyArn = this.createPolicyAndAttachRole(iam);
            this.waitForRoleFinalization();
            LOGGER.info("Role: {} with attached Policy: {} for Dmap-Lambda has been created\n", (Object)policyArn, (Object)lambdaRoleArn);
            String string = lambdaRoleArn;
            return string;
        }
    }

    private String createLambdaRole(IamClient iam) {
        try {
            LOGGER.info("Creating {}", (Object)ROLE_NAME);
            CreateRoleRequest request = (CreateRoleRequest)CreateRoleRequest.builder().roleName(ROLE_NAME).assumeRolePolicyDocument(Util.getResourceAsString("/lambda/policy/assume-role-policy.json")).description("Role lambda invoking for DMAP port scanning over EC2 services").build();
            CreateRoleResponse response = iam.createRole(request);
            iam.waiter().waitUntilRoleExists(builder -> builder.roleName(ROLE_NAME).build());
            return response.role().arn();
        }
        catch (EntityAlreadyExistsException e) {
            GetRoleResponse response = iam.getRole(builder -> builder.roleName(ROLE_NAME).build());
            LOGGER.info("Required role: {} was found reusing its ARN: {}", (Object)ROLE_NAME, (Object)response.role().arn());
            return response.role().arn();
        }
    }

    private String createPolicyAndAttachRole(IamClient iam) {
        List attachedPolicies = iam.listAttachedRolePolicies(builder -> builder.roleName(ROLE_NAME)).attachedPolicies();
        if (attachedPolicies.isEmpty()) {
            CreatePolicyResponse policyResponse = iam.createPolicy(builder -> builder.policyName(POLICY_NAME).policyDocument(Util.getResourceAsString("/lambda/policy/role-policy.json")).build());
            String policyArn = policyResponse.policy().arn();
            iam.waiter().waitUntilPolicyExists(builder -> builder.policyArn(policyArn).build());
            iam.attachRolePolicy(builder -> builder.policyArn(policyArn).roleName(ROLE_NAME).build());
            return policyArn;
        }
        LOGGER.info("Skipped policy creation. Target role found with attached policies: {}", (Object)attachedPolicies);
        return attachedPolicies.stream().filter(attachedPolicy -> attachedPolicy.policyName().equals(POLICY_NAME)).findFirst().map(AttachedPolicy::policyArn).orElseThrow(() -> new DMapProcessingException("Incomplete state of Role: openraven-dmap-scan-role Policy: openraven-dmap-scan-policy not found"));
    }

    private String createLambda(LambdaClient lambdaClient, VpcConfig vpcConfig, String roleArn) {
        LOGGER.info("Creating lambda function in VPC: {} ", (Object)vpcConfig);
        SdkBytes source = Optional.ofNullable(this.getClass().getClassLoader().getResourceAsStream(DMAP_LAMBDA_JAR_PATH)).map(SdkBytes::fromInputStream).orElseThrow(() -> new RuntimeException("Unable to find sources under lambda/source/dmap-lambda-0.0.0.jar"));
        CreateFunctionRequest functionRequest = (CreateFunctionRequest)CreateFunctionRequest.builder().functionName("dmap-" + UUID.randomUUID()).description("DMAP Lambda function for port scanning distributed by OpenRaven").code(builder -> builder.zipFile(source).build()).handler(DMAP_LAMBDA_HANDLER).runtime(Runtime.JAVA11).timeout(Integer.valueOf(900)).memorySize(Integer.valueOf(512)).role(roleArn).vpcConfig(builder -> builder.securityGroupIds(vpcConfig.getSecurityGroupIds()).subnetIds(new String[]{vpcConfig.getSubnetId()}).build()).build();
        CreateFunctionResponse functionResponse = lambdaClient.createFunction(functionRequest);
        String functionName = functionResponse.functionName();
        LOGGER.debug("Waiting for creation: {}", (Object)functionName);
        lambdaClient.waiter().waitUntilFunctionActive(builder -> builder.functionName(functionName).build());
        this.registeredLambdas.add(new LambdaDetails(vpcConfig.getRegion(), functionName));
        LOGGER.debug("Lambda function for DMap port scan has been created: {}", (Object)functionName);
        return functionName;
    }

    private void deleteLambda(LambdaClient lambdaClient, String lambdaName) {
        lambdaClient.deleteFunction((DeleteFunctionRequest)DeleteFunctionRequest.builder().functionName(lambdaName).build());
        LOGGER.info("Lambda function: {} has been removed", (Object)lambdaName);
    }

    private InvokeRequest getLambdaRequest(String lambdaName, EC2Target ec2Target) {
        try {
            String payload = this.mapper.writeValueAsString((Object)new DmapLambdaRequest(Map.of(ec2Target.getResourceId(), ec2Target.getIpAddress())));
            return (InvokeRequest)InvokeRequest.builder().functionName(lambdaName).payload(SdkBytes.fromUtf8String((String)payload)).invocationType(InvocationType.REQUEST_RESPONSE).build();
        }
        catch (JsonProcessingException e) {
            LOGGER.error("Unable to serialize host data", (Throwable)e);
            throw new DMapProcessingException(e);
        }
    }

    private DMapLambdaResponse processResponse(VpcConfig vpcConfig, InvokeResponse response) {
        String functionError = response.functionError();
        if (StringUtils.isNotBlank((CharSequence)functionError)) {
            String id = String.format("accountId=%s, subnet=%s, securityGroups=%s", "account", vpcConfig.getSubnetId(), String.join((CharSequence)",", vpcConfig.getSecurityGroupIds()));
            String errorMessage = String.format("Non-null function error for %s: %s", id, functionError);
            throw new DMapProcessingException(errorMessage);
        }
        String payload = response.payload().asString(Charset.defaultCharset());
        try {
            return (DMapLambdaResponse)this.mapper.readValue(payload, DMapLambdaResponse.class);
        }
        catch (JsonProcessingException e) {
            LOGGER.info("Unable to parse response data from lambda: {}", (Object)payload, (Object)e);
            throw new DMapProcessingException(e);
        }
    }

    private LambdaClient getLambdaClient(SdkHttpClient httpClient, String region) {
        return (LambdaClient)((LambdaClientBuilder)((LambdaClientBuilder)((LambdaClientBuilder)LambdaClient.builder().region(Region.of((String)region))).httpClient(httpClient)).overrideConfiguration((ClientOverrideConfiguration)ClientOverrideConfiguration.builder().apiCallAttemptTimeout(FIFTEEN_MINUTES).apiCallTimeout(FIFTEEN_MINUTES).build())).build();
    }

    private SdkHttpClient getSdkHttpClient() {
        return new DefaultSdkHttpClientBuilder().buildWithDefaults(AttributeMap.builder().put((AttributeMap.Key)SdkHttpConfigurationOption.READ_TIMEOUT, (Object)FIFTEEN_MINUTES).build());
    }

    private void waitForRoleFinalization() {
        try {
            LOGGER.info("Waiting for role state finalization");
            Thread.sleep(9000L);
        }
        catch (InterruptedException e) {
            LOGGER.warn("Finalization has been interrupted");
        }
    }

    private void release(Semaphore bulkheadSemaphore, EC2Target ec2Target) {
        LOGGER.debug("Thread released of DMap scan for: {}", (Object)ec2Target);
        bulkheadSemaphore.release();
    }

    private void acquire(Semaphore bulkheadSemaphore, EC2Target ec2Target) {
        try {
            LOGGER.debug("Waiting for thread for DMap scan of: {}", (Object)ec2Target);
            bulkheadSemaphore.acquire();
            LOGGER.debug("Acquired thread for DMap scan of: {}", (Object)ec2Target);
        }
        catch (InterruptedException e) {
            LOGGER.error("Interrupted thread of DMap Scan for following service: " + ec2Target);
            throw new DMapProcessingException(e);
        }
    }

    private void waitForCompletion(CountDownLatch countDownLatch) {
        try {
            countDownLatch.await();
        }
        catch (InterruptedException e) {
            LOGGER.warn("DMap scan process was interrupted");
        }
    }

    @Override
    public void validateEnvironment(boolean reuseResourcesAllowed) {
        try (IamClient iam = (IamClient)((IamClientBuilder)IamClient.builder().region(Region.AWS_GLOBAL)).build();){
            Role role = iam.getRole(builder -> builder.roleName(ROLE_NAME).build()).role();
            if (!reuseResourcesAllowed) {
                LOGGER.error("Role: {} his alredy exist on environment. Another DMap scan probably running. \nTo skip this error and reuse existing roles and policies start magpie-dmap with -r=true argument, WARN existing role could be removed by another DMap execution and have impact on other simulteneous executions", (Object)role.roleName());
                throw new DMapProcessingException("Simultaneous DMap execution is disabled by default, due to contention for AWS Global resources");
            }
        }
        catch (NoSuchEntityException e) {
            LOGGER.debug("Required role not found, assume env is ready for DMap scan");
        }
    }
}

