/*
 * Decompiled with CFR 0.152.
 */
package io.micronaut.oraclecloud.certificates.services;

import com.oracle.bmc.certificates.Certificates;
import com.oracle.bmc.certificates.model.CertificateBundleWithPrivateKey;
import com.oracle.bmc.certificates.requests.GetCertificateBundleRequest;
import com.oracle.bmc.certificates.responses.GetCertificateBundleResponse;
import io.micronaut.context.annotation.Requirements;
import io.micronaut.context.annotation.Requires;
import io.micronaut.context.event.ApplicationEventPublisher;
import io.micronaut.core.annotation.NonNull;
import io.micronaut.oraclecloud.certificates.OracleCloudCertificationsConfiguration;
import io.micronaut.oraclecloud.certificates.events.CertificateEvent;
import io.micronaut.retry.annotation.Retryable;
import jakarta.inject.Singleton;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.security.PrivateKey;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.openssl.PEMException;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
@Requirements(value={@Requires(classes={Certificates.class}), @Requires(beans={Certificates.class}), @Requires(property="oci.certificates.enabled", value="true")})
public class OracleCloudCertificateService {
    private static final Logger LOG = LoggerFactory.getLogger(OracleCloudCertificateService.class);
    private static final String X509_CERT = "X.509";
    private final OracleCloudCertificationsConfiguration oracleCloudCertificationsConfiguration;
    private final Certificates certificates;
    private final ApplicationEventPublisher<CertificateEvent> eventPublisher;

    public OracleCloudCertificateService(OracleCloudCertificationsConfiguration oracleCloudCertificationsConfiguration, Certificates certificates, ApplicationEventPublisher<CertificateEvent> eventPublisher) {
        this.oracleCloudCertificationsConfiguration = oracleCloudCertificationsConfiguration;
        this.certificates = certificates;
        this.eventPublisher = eventPublisher;
    }

    @NonNull
    protected Optional<CertificateEvent> getCertificateEvent() {
        try {
            CertificateFactory cf = CertificateFactory.getInstance(X509_CERT);
            GetCertificateBundleResponse certificateBundle = this.certificates.getCertificateBundle(GetCertificateBundleRequest.builder().certificateId(this.oracleCloudCertificationsConfiguration.certificateId()).versionNumber(this.oracleCloudCertificationsConfiguration.versionNumber()).certificateVersionName(this.oracleCloudCertificationsConfiguration.certificateVersionName()).certificateBundleType(GetCertificateBundleRequest.CertificateBundleType.CertificateContentWithPrivateKey).build());
            List<X509Certificate> intermediate = Collections.emptyList();
            if (certificateBundle.getCertificateBundle().getCertChainPem() != null) {
                intermediate = cf.generateCertificates(new ByteArrayInputStream(certificateBundle.getCertificateBundle().getCertChainPem().getBytes())).stream().map(cert -> (X509Certificate)cert).collect(Collectors.toList());
            }
            CertificateEvent certificateEvent = new CertificateEvent(this.getPrivateKey(certificateBundle), (X509Certificate)cf.generateCertificate(new ByteArrayInputStream(certificateBundle.getCertificateBundle().getCertificatePem().getBytes())), intermediate);
            return Optional.of(certificateEvent);
        }
        catch (CertificateException e) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("Could not create certificate from file", (Throwable)e);
            }
            return Optional.empty();
        }
    }

    private PrivateKey getPrivateKey(GetCertificateBundleResponse getCertificateBundleResponse) {
        try {
            CertificateBundleWithPrivateKey certificateBundleWithPrivateKey = (CertificateBundleWithPrivateKey)getCertificateBundleResponse.getCertificateBundle();
            return this.parsePrivateKey(certificateBundleWithPrivateKey.getPrivateKeyPem());
        }
        catch (IOException ioException) {
            return null;
        }
    }

    @Retryable(attempts="${oci.certificates.refresh.retry.attempts:3}", delay="${oci.certificates.refresh.retry.delay:1s}")
    public void refreshCertificate() {
        Optional<CertificateEvent> certificateEvent = this.getCertificateEvent();
        if (certificateEvent.isPresent()) {
            this.eventPublisher.publishEvent((Object)certificateEvent.get());
        } else if (LOG.isErrorEnabled()) {
            LOG.error("Oracle Cloud certificate could not be loaded from service.");
        }
    }

    private PrivateKey parsePrivateKey(String privateKeyPem) throws IOException {
        PrivateKey privateKey;
        PEMParser parser = new PEMParser((Reader)new StringReader(privateKeyPem));
        try {
            PrivateKeyInfo privateKeyInfo;
            Object parsedObject = parser.readObject();
            if (parsedObject instanceof PEMKeyPair) {
                PEMKeyPair pemkeypair = (PEMKeyPair)parsedObject;
                privateKeyInfo = pemkeypair.getPrivateKeyInfo();
            } else if (parsedObject instanceof PrivateKeyInfo) {
                PrivateKeyInfo privateKeyInfoParsed;
                privateKeyInfo = privateKeyInfoParsed = (PrivateKeyInfo)parsedObject;
            } else {
                throw new IllegalStateException("Unexpected value: " + parser.readObject());
            }
            privateKey = new JcaPEMKeyConverter().getPrivateKey(privateKeyInfo);
        }
        catch (Throwable throwable) {
            try {
                try {
                    parser.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (PEMException ex) {
                throw new IOException("Invalid PEM file", ex);
            }
        }
        parser.close();
        return privateKey;
    }
}

