package io.manbang.ebatis.core.request;

import io.manbang.ebatis.core.annotation.MultiSearch;
import io.manbang.ebatis.core.domain.ContextHolder;
import io.manbang.ebatis.core.domain.Pageable;
import io.manbang.ebatis.core.meta.MethodMeta;
import io.manbang.ebatis.core.meta.ParameterMeta;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.MultiSearchRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.search.builder.SearchSourceBuilder;

import java.util.Collection;
import java.util.Optional;

/**
 * @author 章多亮
 * @since 2020/1/14 15:58
 */
class MultiSearchRequestFactory extends AbstractRequestFactory<MultiSearch, MultiSearchRequest> {
    static final MultiSearchRequestFactory INSTANCE = new MultiSearchRequestFactory();

    private MultiSearchRequestFactory() {
    }

    @Override
    protected void setAnnotationMeta(MultiSearchRequest request, MultiSearch multiSearch) {
        int maxConcurrentSearchRequests = multiSearch.maxConcurrentSearchRequests();
        if (maxConcurrentSearchRequests >= 1) {
            request.maxConcurrentSearchRequests(maxConcurrentSearchRequests);
        }

        for (SearchRequest searchRequest : request.requests()) {
            SearchSourceBuilder source = searchRequest.source();
            if (multiSearch.countOnly()) {
                source.fetchSource(false).size(0);
            }

            searchRequest
                    .searchType(multiSearch.searchType())
                    .preference(StringUtils.trimToNull(multiSearch.preference()));

            searchRequest.setPreFilterShardSize(multiSearch.preFilterShardSize());
            searchRequest.setBatchedReduceSize(multiSearch.batchedReduceSize());
        }
    }

    @Override
    protected MultiSearchRequest doCreate(MethodMeta meta, Object[] args) {
        ParameterMeta parameterMeta = meta.getConditionParameter();
        Object arg = parameterMeta.getValue(args);


        Object[] conditions;
        if (parameterMeta.isCollection()) {
            Collection<?> collection = (Collection<?>) arg;
            conditions = collection.toArray();
        } else if (parameterMeta.isArray()) {
            conditions = (Object[]) arg;
        } else {
            conditions = new Object[]{arg};
        }

        Optional<ParameterMeta> pageableParameter = meta.getPageableParameter();

        Pageable[] pageables = pageableParameter.map(p -> {
            Object value = p.getValue(args);
            if (p.isCollection()) {
                @SuppressWarnings("unchecked")
                Collection<Pageable> collection = (Collection<Pageable>) value;
                return collection.toArray(new Pageable[0]);
            } else if (p.isArray()) {
                return (Pageable[]) value;
            } else {
                return new Pageable[]{(Pageable) value};
            }
        }).orElse(new Pageable[conditions.length]);

        ContextHolder.setPageables(pageables);

        int parameterLength = Math.max(parameterMeta.getIndex(), pageableParameter.map(ParameterMeta::getIndex).orElse(-1)) + 1;

        MultiSearchRequest request = new MultiSearchRequest();
        for (int i = 0; i < conditions.length; i++) {
            Pageable pageable = pageables[i];
            ContextHolder.setPageable(pageable);

            Object[] parameters = new Object[parameterLength];
            parameters[parameterMeta.getIndex()] = conditions[i];
            pageableParameter.ifPresent(p -> parameters[p.getIndex()] = pageable);
            request.add(RequestFactory.search().create(meta, parameters));
        }

        return request;
    }
}
