package io.lsn.spring.result.cache;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.lsn.logger.factory.LoggerFactory;
import io.lsn.logger.factory.logger.Logger;
import io.lsn.spring.result.cache.configuration.domain.CacheableResponse;
import io.lsn.spring.result.cache.configuration.domain.ResultCacheProperties;
import io.lsn.spring.result.cache.domain.CacheElement;
import org.redisson.api.RMapCache;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
@SuppressWarnings("unchecked")
public class ResultCacheService {

    private Logger logger = LoggerFactory.getLogger(ResultCacheService.class);
    private ResultCacheProperties properties;
    private ObjectMapper objectMapper;

    private RMapCache<String, CacheElement> cache;

    public ResultCacheService(ResultCacheProperties properties,
                              @Qualifier("io.lsn.spring.result.cache.object.mapper") ObjectMapper objectMapper,
                              @Qualifier("io.lsn.spring.result.cache.redisson") RedissonClient redissonClient) {
        this.properties = properties;
        this.objectMapper = objectMapper;
        this.cache = redissonClient.getMapCache("io.lsn.spring.result.cache");
    }

    /*
        setters
     */
    public <T> String cacheList(List<T> elements) {
        return cacheList(elements, (String) null, Duration.of(properties.getCache().getTime(), ChronoUnit.SECONDS));
    }

    public <T> String cacheList(List<T> elements, Duration duration) {
        return this.cacheList(elements, (String) null, duration);
    }

    public <T> String cacheList(List<T> elements, String additionalSecurityKey) {
        return this.cacheList(elements, additionalSecurityKey, Duration.of(properties.getCache().getTime(), ChronoUnit.SECONDS));
    }

    public <T> String cacheList(List<T> elements, Long additionalSecurityKey) {
        return this.cacheList(elements, toString(additionalSecurityKey), Duration.of(properties.getCache().getTime(), ChronoUnit.SECONDS));
    }

    public <T> String cacheList(List<T> elements, Long additionalSecurityKey, Duration duration) {
        return cacheList(elements, toString(additionalSecurityKey), duration);
    }

    /**
     * @param elements              list of element to be stored in the cache
     * @param additionalSecurityKey element prefix
     * @param duration              cache duration
     */
    @SuppressWarnings("squid:S1168")
    public <T> String cacheList(List<T> elements, String additionalSecurityKey, Duration duration) {
        String uuid = UUID.randomUUID().toString();
        try {
            CacheElement<T> element = CacheElement.of(toString(additionalSecurityKey), objectMapper.writeValueAsString(elements), duration.getSeconds());
            cache.fastPut(uuid, element, duration.getSeconds(), TimeUnit.SECONDS);
            return uuid;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    /*
        getters
     */
    public <T> CacheableResponse<T> getElements(Class<T[]> resultClass, String hash, int limit, int offset) {
        return getElements(resultClass, hash, (String) null, limit, offset);
    }

    public <T> CacheableResponse<T> getElements(Class<T[]> resultClass, String hash, Long additionalSecurityKey, int limit, int offset) {
        return getElements(resultClass, hash, toString(additionalSecurityKey), limit, offset);
    }

    /**
     * @param hash                  has of cache
     * @param additionalSecurityKey additional security key
     * @param limit                 list limitation
     * @param offset                offset
     * @return
     */
    @SuppressWarnings("squid:S1168")
    public <T> CacheableResponse<T> getElements(Class<T[]> resultClass, String hash, String additionalSecurityKey, int limit, int offset) {
        CacheElement<T> element = cache.get(hash);

        if (element == null || !toString(additionalSecurityKey).equals(element.getKey()) || element.getJson() == null) {
            return null;
        }

        // extend TTL
        cache.fastPut(hash, element, element.getDuration(), TimeUnit.SECONDS);

        try {
            List<T> list = Arrays.asList(this.objectMapper.readValue(element.getJson(), resultClass));
            CacheableResponse<T> response = new CacheableResponse<>();
            response.setTotalCount(Long.valueOf(list.size()));
            response.setCacheUid(hash);
            response.setResultList(list.stream()
                    .skip(offset)
                    .limit(limit)
                    .collect(Collectors.toList()));
            return response;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    /*
        helpers
     */

    private String toString(String key) {
        return key != null ? key : "";
    }

    private String toString(Long key) {
        return key != null ? key.toString() : "";
    }
}
