/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.oauthbearer.internals.secured;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.security.oauthbearer.internals.secured.AccessTokenRetriever;
import org.apache.kafka.common.security.oauthbearer.internals.secured.Retry;
import org.apache.kafka.common.security.oauthbearer.internals.secured.UnretryableException;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpAccessTokenRetriever
implements AccessTokenRetriever {
    private static final Logger log = LoggerFactory.getLogger(HttpAccessTokenRetriever.class);
    private static final Set<Integer> UNRETRYABLE_HTTP_CODES = new HashSet<Integer>();
    private static final int MAX_RESPONSE_BODY_LENGTH = 1000;
    public static final String AUTHORIZATION_HEADER = "Authorization";
    private final String clientId;
    private final String clientSecret;
    private final String scope;
    private final SSLSocketFactory sslSocketFactory;
    private final String tokenEndpointUrl;
    private final long loginRetryBackoffMs;
    private final long loginRetryBackoffMaxMs;
    private final Integer loginConnectTimeoutMs;
    private final Integer loginReadTimeoutMs;

    public HttpAccessTokenRetriever(String clientId, String clientSecret, String scope, SSLSocketFactory sslSocketFactory, String tokenEndpointUrl, long loginRetryBackoffMs, long loginRetryBackoffMaxMs, Integer loginConnectTimeoutMs, Integer loginReadTimeoutMs) {
        this.clientId = Objects.requireNonNull(clientId);
        this.clientSecret = Objects.requireNonNull(clientSecret);
        this.scope = scope;
        this.sslSocketFactory = sslSocketFactory;
        this.tokenEndpointUrl = Objects.requireNonNull(tokenEndpointUrl);
        this.loginRetryBackoffMs = loginRetryBackoffMs;
        this.loginRetryBackoffMaxMs = loginRetryBackoffMaxMs;
        this.loginConnectTimeoutMs = loginConnectTimeoutMs;
        this.loginReadTimeoutMs = loginReadTimeoutMs;
    }

    @Override
    public String retrieve() throws IOException {
        String responseBody;
        String authorizationHeader = HttpAccessTokenRetriever.formatAuthorizationHeader(this.clientId, this.clientSecret);
        String requestBody = HttpAccessTokenRetriever.formatRequestBody(this.scope);
        Retry<String> retry = new Retry<String>(this.loginRetryBackoffMs, this.loginRetryBackoffMaxMs);
        Map<String, String> headers = Collections.singletonMap(AUTHORIZATION_HEADER, authorizationHeader);
        try {
            responseBody = retry.execute(() -> {
                HttpURLConnection con = null;
                try {
                    con = (HttpURLConnection)new URL(this.tokenEndpointUrl).openConnection();
                    if (this.sslSocketFactory != null && con instanceof HttpsURLConnection) {
                        ((HttpsURLConnection)con).setSSLSocketFactory(this.sslSocketFactory);
                    }
                    String string = HttpAccessTokenRetriever.post(con, headers, requestBody, this.loginConnectTimeoutMs, this.loginReadTimeoutMs);
                    return string;
                }
                catch (IOException e) {
                    throw new ExecutionException(e);
                }
                finally {
                    if (con != null) {
                        con.disconnect();
                    }
                }
            });
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof IOException) {
                throw (IOException)e.getCause();
            }
            throw new KafkaException(e.getCause());
        }
        return HttpAccessTokenRetriever.parseAccessToken(responseBody);
    }

    public static String post(HttpURLConnection con, Map<String, String> headers, String requestBody, Integer connectTimeoutMs, Integer readTimeoutMs) throws IOException, UnretryableException {
        HttpAccessTokenRetriever.handleInput(con, headers, requestBody, connectTimeoutMs, readTimeoutMs);
        return HttpAccessTokenRetriever.handleOutput(con);
    }

    private static void handleInput(HttpURLConnection con, Map<String, String> headers, String requestBody, Integer connectTimeoutMs, Integer readTimeoutMs) throws IOException, UnretryableException {
        log.debug("handleInput - starting post for {}", (Object)con.getURL());
        con.setRequestMethod("POST");
        con.setRequestProperty("Accept", "application/json");
        if (headers != null) {
            for (Map.Entry<String, String> header : headers.entrySet()) {
                con.setRequestProperty(header.getKey(), header.getValue());
            }
        }
        con.setRequestProperty("Cache-Control", "no-cache");
        if (requestBody != null) {
            con.setRequestProperty("Content-Length", String.valueOf(requestBody.length()));
            con.setDoOutput(true);
        }
        con.setUseCaches(false);
        if (connectTimeoutMs != null) {
            con.setConnectTimeout(connectTimeoutMs);
        }
        if (readTimeoutMs != null) {
            con.setReadTimeout(readTimeoutMs);
        }
        log.debug("handleInput - preparing to connect to {}", (Object)con.getURL());
        con.connect();
        if (requestBody != null) {
            try (OutputStream os = con.getOutputStream();){
                ByteArrayInputStream is = new ByteArrayInputStream(requestBody.getBytes(StandardCharsets.UTF_8));
                log.debug("handleInput - preparing to write request body to {}", (Object)con.getURL());
                HttpAccessTokenRetriever.copy(is, os);
            }
        }
    }

    static String handleOutput(HttpURLConnection con) throws IOException {
        int responseCode = con.getResponseCode();
        log.debug("handleOutput - responseCode: {}", (Object)responseCode);
        String responseBody = null;
        String errorResponseBody = null;
        try (InputStream is = con.getInputStream();){
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            log.debug("handleOutput - preparing to read response body from {}", (Object)con.getURL());
            HttpAccessTokenRetriever.copy(is, os);
            responseBody = os.toString(StandardCharsets.UTF_8.name());
        }
        catch (Exception e) {
            try (InputStream is2 = con.getErrorStream();){
                ByteArrayOutputStream os = new ByteArrayOutputStream();
                log.debug("handleOutput - preparing to read error response body from {}", (Object)con.getURL());
                HttpAccessTokenRetriever.copy(is2, os);
                errorResponseBody = os.toString(StandardCharsets.UTF_8.name());
            }
            catch (Exception e2) {
                log.warn("handleOutput - error retrieving error information", e2);
            }
            log.warn("handleOutput - error retrieving data", e);
        }
        if (responseCode == 200 || responseCode == 201) {
            log.debug("handleOutput - responseCode: {}, error response: {}", (Object)responseCode, (Object)errorResponseBody);
            if (responseBody == null || responseBody.isEmpty()) {
                throw new IOException(String.format("The token endpoint response was unexpectedly empty despite response code %d from %s and error message %s", responseCode, con.getURL(), HttpAccessTokenRetriever.formatErrorMessage(errorResponseBody)));
            }
            return responseBody;
        }
        log.warn("handleOutput - error response code: {}, error response body: {}", (Object)responseCode, (Object)HttpAccessTokenRetriever.formatErrorMessage(errorResponseBody));
        if (UNRETRYABLE_HTTP_CODES.contains(responseCode)) {
            throw new UnretryableException(new IOException(String.format("The response code %s and error response %s was encountered reading the token endpoint response; will not attempt further retries", responseCode, HttpAccessTokenRetriever.formatErrorMessage(errorResponseBody))));
        }
        throw new IOException(String.format("The unexpected response code %s and error message %s was encountered reading the token endpoint response", responseCode, HttpAccessTokenRetriever.formatErrorMessage(errorResponseBody)));
    }

    static void copy(InputStream is, OutputStream os) throws IOException {
        int b;
        byte[] buf = new byte[4096];
        while ((b = is.read(buf)) != -1) {
            os.write(buf, 0, b);
        }
    }

    static String formatErrorMessage(String errorResponseBody) {
        if (errorResponseBody == null || errorResponseBody.trim().equals("")) {
            return "{}";
        }
        ObjectMapper mapper = new ObjectMapper();
        try {
            JsonNode rootNode = mapper.readTree(errorResponseBody);
            if (!rootNode.at("/error").isMissingNode()) {
                return String.format("{%s - %s}", rootNode.at("/error"), rootNode.at("/error_description"));
            }
            if (!rootNode.at("/errorCode").isMissingNode()) {
                return String.format("{%s - %s}", rootNode.at("/errorCode"), rootNode.at("/errorSummary"));
            }
            return errorResponseBody;
        }
        catch (Exception e) {
            log.warn("Error parsing error response", e);
            return String.format("{%s}", errorResponseBody);
        }
    }

    static String parseAccessToken(String responseBody) throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        JsonNode rootNode = mapper.readTree(responseBody);
        JsonNode accessTokenNode = rootNode.at("/access_token");
        if (accessTokenNode == null) {
            String snippet = responseBody;
            if (snippet.length() > 1000) {
                int actualLength = responseBody.length();
                String s = responseBody.substring(0, 1000);
                snippet = String.format("%s (trimmed to first %d characters out of %d total)", s, 1000, actualLength);
            }
            throw new IOException(String.format("The token endpoint response did not contain an access_token value. Response: (%s)", snippet));
        }
        return HttpAccessTokenRetriever.sanitizeString("the token endpoint response's access_token JSON attribute", accessTokenNode.textValue());
    }

    static String formatAuthorizationHeader(String clientId, String clientSecret) {
        clientId = HttpAccessTokenRetriever.sanitizeString("the token endpoint request client ID parameter", clientId);
        clientSecret = HttpAccessTokenRetriever.sanitizeString("the token endpoint request client secret parameter", clientSecret);
        String s = String.format("%s:%s", clientId, clientSecret);
        String encoded = Base64.getEncoder().encodeToString(Utils.utf8(s));
        return String.format("Basic %s", encoded);
    }

    static String formatRequestBody(String scope) throws IOException {
        try {
            StringBuilder requestParameters = new StringBuilder();
            requestParameters.append("grant_type=client_credentials");
            if (scope != null && !scope.trim().isEmpty()) {
                scope = scope.trim();
                String encodedScope = URLEncoder.encode(scope, StandardCharsets.UTF_8.name());
                requestParameters.append("&scope=").append(encodedScope);
            }
            return requestParameters.toString();
        }
        catch (UnsupportedEncodingException e) {
            throw new IOException(String.format("Encoding %s not supported", StandardCharsets.UTF_8.name()));
        }
    }

    private static String sanitizeString(String name, String value) {
        if (value == null) {
            throw new IllegalArgumentException(String.format("The value for %s must be non-null", name));
        }
        if (value.isEmpty()) {
            throw new IllegalArgumentException(String.format("The value for %s must be non-empty", name));
        }
        if ((value = value.trim()).isEmpty()) {
            throw new IllegalArgumentException(String.format("The value for %s must not contain only whitespace", name));
        }
        return value;
    }

    static {
        UNRETRYABLE_HTTP_CODES.add(400);
        UNRETRYABLE_HTTP_CODES.add(401);
        UNRETRYABLE_HTTP_CODES.add(402);
        UNRETRYABLE_HTTP_CODES.add(403);
        UNRETRYABLE_HTTP_CODES.add(404);
        UNRETRYABLE_HTTP_CODES.add(405);
        UNRETRYABLE_HTTP_CODES.add(406);
        UNRETRYABLE_HTTP_CODES.add(407);
        UNRETRYABLE_HTTP_CODES.add(409);
        UNRETRYABLE_HTTP_CODES.add(410);
        UNRETRYABLE_HTTP_CODES.add(411);
        UNRETRYABLE_HTTP_CODES.add(412);
        UNRETRYABLE_HTTP_CODES.add(413);
        UNRETRYABLE_HTTP_CODES.add(414);
        UNRETRYABLE_HTTP_CODES.add(415);
        UNRETRYABLE_HTTP_CODES.add(501);
        UNRETRYABLE_HTTP_CODES.add(505);
    }
}

