package joyfill

import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.alloc
import kotlinx.cinterop.memScoped
import kotlinx.cinterop.pointed
import kotlinx.cinterop.ptr
import kotlinx.cinterop.refTo
import kotlinx.cinterop.toKString
import platform.posix.SEEK_END
import platform.posix.SEEK_SET
import platform.posix.S_IFDIR
import platform.posix.S_IFMT
import platform.posix.closedir
import platform.posix.fclose
import platform.posix.fopen
import platform.posix.fread
import platform.posix.fseek
import platform.posix.ftell
import platform.posix.getcwd
import platform.posix.opendir
import platform.posix.readdir
import platform.posix.readlink
import platform.posix.stat

/**
 * Linux implementation of ResourceLoader
 */
@OptIn(ExperimentalForeignApi::class)
actual object ResourceLoader {
    // Base directory for resources in Linux
    private val resourcesBasePath: String by lazy {
        // Try to find the executable path
        val exePath = getExecutablePath() ?: ""
        val exeDir = exePath.substringBeforeLast('/', "")
        
        // Resources are typically in a "resources" directory next to the executable
        // or in the current working directory
        val resourcesDir = "$exeDir/resources"
        
        if (directoryExists(resourcesDir)) {
            resourcesDir
        } else {
            // Fallback to current directory + resources
            val currentDir = getCurrentWorkingDirectory() ?: ""
            "$currentDir/resources"
        }
    }
    
    actual fun loadResource(path: String): String? {
        return try {
            val fullPath = "$resourcesBasePath/$path"
            
            // Open the file
            val file = fopen(fullPath, "r") ?: return null
            
            try {
                // Get file size
                fseek(file, 0, SEEK_END)
                val size = ftell(file)
                fseek(file, 0, SEEK_SET)
                
                // Read the file content
                val buffer = ByteArray(size.toInt())
                memScoped {
                    val readSize = fread(buffer.refTo(0), 1.toULong(), size.toULong(), file)
                    if (readSize < size.toULong()) {
                        return null
                    }
                }
                
                // Convert to string
                buffer.decodeToString()
            } finally {
                fclose(file)
            }
        } catch (_: Exception) {
            null
        }
    }

    actual fun filesFrom(path: String): List<String> {
        return try {
            val normalizedPath = path.trimEnd('/')
            val fullPath = "$resourcesBasePath/$normalizedPath"
            
            // Check if directory exists
            if (!directoryExists(fullPath)) {
                return emptyList()
            }
            
            // Open directory
            val dir = opendir(fullPath) ?: return emptyList()
            
            try {
                val result = mutableListOf<String>()
                
                // Read directory entries
                memScoped {
                    var entry = readdir(dir)
                    while (entry != null) {
                        val name = entry.pointed.d_name.toKString()
                        
                        // Skip "." and ".." entries
                        if (name != "." && name != "..") {
                            // Check if it's a file (not a directory)
                            val entryPath = "$fullPath/$name"
                            if (!directoryExists(entryPath)) {
                                result.add(name)
                            }
                        }
                        
                        entry = readdir(dir)
                    }
                }
                
                result
            } finally {
                closedir(dir)
            }
        } catch (_: Exception) {
            emptyList()
        }
    }
    
    // Helper function to check if a directory exists
    @OptIn(ExperimentalForeignApi::class)
    private fun directoryExists(path: String): Boolean {
        memScoped {
            val statBuf = alloc<stat>()
            val result = stat(path, statBuf.ptr)
            
            if (result == 0) {
                return (statBuf.st_mode.toInt() and S_IFMT) == S_IFDIR
            }
            
            return false
        }
    }
    
    // Helper function to get the executable path
    @OptIn(ExperimentalForeignApi::class)
    private fun getExecutablePath(): String? {
        return try {
            memScoped {
                val buffer = ByteArray(4096)
                val size = readlink("/proc/self/exe", buffer.refTo(0), buffer.size.toULong())
                
                if (size > 0) {
                    buffer.decodeToString(0, size.toInt())
                } else {
                    null
                }
            }
        } catch (_: Exception) {
            null
        }
    }
    
    // Helper function to get the current working directory
    @OptIn(ExperimentalForeignApi::class)
    private fun getCurrentWorkingDirectory(): String? {
        return try {
            memScoped {
                val buffer = ByteArray(4096)
                val result = getcwd(buffer.refTo(0), buffer.size.toULong())
                
                if (result != null) {
                    buffer.toKString()
                } else {
                    null
                }
            }
        } catch (_: Exception) {
            null
        }
    }
}