package joyfill.validation

import kotlin.jvm.JvmInline

/**
 * A value class representing a semantic version following the pattern "major.minor.patch[-suffix]".
 * 
 * This class provides functionality to parse version strings and compare versions according to 
 * semantic versioning principles. It implements [Comparable] to support natural ordering of versions.
 * 
 * Examples:
 * - "1.0.0" (basic version)
 * - "2.1.3" (standard version)
 * - "1.0.0-alpha" (version with suffix)
 * 
 * @property raw The original version string as provided
 * 
 * @see Comparable
 */
@JvmInline
value class Version(val raw: String) : Comparable<Version> {
    private val parts: List<String>
        get() = raw.split('.')

    /**
     * The major version number.
     * 
     * This is extracted from the part before the first dot in the version string.
     * If parsing fails, defaults to 1.
     * 
     * Example: For "2.1.3", returns 2
     */
    val major: Int
        get() = raw.substringBefore('.').toIntOrNull() ?: 1

    /**
     * The minor version number.
     * 
     * This is extracted from the part between the first and last dot in the version string.
     * If parsing fails, defaults to 0.
     * 
     * Example: For "2.1.3", returns 1
     */
    val minor: Int
        get() {
            return if (parts.size >= 2) {
                parts[1].toIntOrNull() ?: 0
            } else {
                0
            }
        }

    /**
     * The patch version number.
     * 
     * This is extracted from the part after the last dot, up to any hyphen suffix.
     * If parsing fails, defaults to 0.
     * 
     * Example: For "2.1.3-alpha", returns 3
     */
    val patch: Int
        get() {
            return if (parts.size >= 3) {
                val patchPart = parts[2]
                val endIndex = patchPart.indexOf('-').takeIf { it > 0 } ?: patchPart.length
                patchPart.substring(0, endIndex).toIntOrNull() ?: 0
            } else {
                0
            }
        }

    /**
     * Checks if this version is compatible with another version.
     * 
     * Two versions are considered compatible if they have the same major version number.
     * This follows semantic versioning principles where major version changes indicate 
     * breaking changes.
     * 
     * @param other The version to check compatibility with
     * @return true if both versions have the same major version, false otherwise
     * 
     * Example:
     * ```
     * Version("2.1.0").compatibleWith(Version("2.3.1")) // returns true
     * Version("2.1.0").compatibleWith(Version("3.0.0")) // returns false
     * ```
     */
    fun compatibleWith(other: Version) = major == other.major

    /**
     * Returns the raw version string.
     * 
     * @return The original version string passed to the constructor
     */
    override fun toString() = raw

    /**
     * Compares this version with another version for ordering.
     * 
     * Versions are compared in order of major, minor, then patch version numbers.
     * Higher version numbers are considered greater.
     * 
     * @param other The version to compare with
     * @return Negative integer if this version is less than other, 
     *         zero if equal, positive integer if greater
     * 
     * Example:
     * ```
     * Version("2.1.0").compareTo(Version("2.0.1")) // returns positive (2.1.0 > 2.0.1)
     * Version("1.0.0").compareTo(Version("2.0.0")) // returns negative (1.0.0 < 2.0.0)
     * ```
     */
    override fun compareTo(other: Version): Int {
        if (major != other.major) return major - other.major
        if (minor != other.minor) return minor - other.minor
        if (patch != other.patch) return patch - other.patch
        return 0
    }

    companion object {
        /**
         * The default version used when no specific version is provided.
         * 
         * Set to "1.0.0" following semantic versioning conventions.
         */
        val Default = Version("1.0.0")
    }
}

/**
 * Extension function to convert a string to a [Version] instance.
 * 
 * @receiver The string to convert to a version
 * @return A new [Version] instance wrapping the string
 * 
 * Example:
 * ```
 * val version = "2.1.3".toVersion()
 * ```
 */
fun String.toVersion() = Version(this)