package joyfill.validation

import joyfill.Error
import joyfill.validation.JoyfillSchemaManager.SdkVersion

sealed class SchemaError(
    override val detail: ErrorDetails,
    override val err: String,
    val code: String,
    override val message: String,
) : Error {

    data class Assertion(
        override val detail: ErrorDetails,
        override val err: String,
    ) : SchemaError(
        detail = detail,
        err = err,
        code = "ERROR_SCHEMA_VALIDATION",
        message = "Error detected during schema validation",
    )

    data class Format(
        override val detail: ErrorDetails,
        override val err: String,
    ) : SchemaError(
        detail = detail,
        err = err,
        code = "ERROR_SCHEMA_FORMAT",
        message = "Error detected during schema format validation",
    )

    data class Version(
        override val detail: ErrorDetails,
        override val err: String,
    ) : SchemaError(
        detail = detail,
        err = err,
        code = "ERROR_SCHEMA_VERSION",
        message = "Error detected with targeted schema version",
    )
}

/**
 * Extra details about the schema and SDK versions
 */
data class ErrorDetails(
    val sdkVersion: Version,
    val schemaVersion: Version? = null,
)

internal fun schemaFormatError(message: String) = SchemaError.Format(
    err = message,
    detail = ErrorDetails(
        schemaVersion = null,
        sdkVersion = SdkVersion
    )
)


internal fun validationError(
    message: String,
    schemaVersion: Version? = null
) = SchemaError.Assertion(
    err = message,
    detail = ErrorDetails(
        schemaVersion = schemaVersion,
        sdkVersion = SdkVersion
    )
)

internal fun versionError(
    documentVersion: Version,
    schemaVersion: Version? = null
) = SchemaError.Version(
    err = "Document version $documentVersion is not supported by SDK version $SdkVersion",
    detail = ErrorDetails(
        schemaVersion = schemaVersion,
        sdkVersion = SdkVersion
    )
)
