package joyfill.validation

import io.github.optimumcode.json.schema.JsonSchema
import io.github.optimumcode.json.schema.ValidationError
import io.joyfill.models.BuildKonfig
import joyfill.utils.string
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.jsonObject

/**
 * Utility class for validating JoyDoc documents against JSON schema
 */
object JoyfillSchemaManager {

    val SdkVersion = Version(BuildKonfig.SDK_VERSION)

    /**
     * Validates a JoyDoc document against the JSON schema
     *
     * @param documentJson The JoyDoc document to validate
     * @return SchemaValidationError if validation fails, null if validation passes
     */
    fun validateSchema(documentJson: String): SchemaError? {
        if (documentJson.isBlank()) return validationError(
            message = "Empty/Blank string is not a valid JoyDoc",
            schemaVersion = Version.Default
        )

        val documentJsonObject = try {
            parser.parseToJsonElement(documentJson).jsonObject
        } catch (_: Exception) {
            return validationError(message = "Provided JoyDoc is not a valid json object")
        }

        val errors = mutableListOf<ValidationError>()

        val documentVersion: Version = documentJsonObject.string(DOCUMENT_VERSION_KEY)?.toVersion() ?: Version.Default

        val schema: JsonElement = schemaFor(documentVersion) ?: return versionError(documentVersion)
        val schemaVersion = schema.string(SCHEMA_VERSION_KEY)
            ?.toVersion() ?: error("Should contain: $SCHEMA_VERSION_KEY")

        if (!documentVersion.compatibleWith(schemaVersion)) return versionError(documentVersion, schemaVersion)

        val validator = JsonSchema.fromJsonElement(schema)
        val isValid = validator.validate(documentJsonObject, errors::add)
        if (isValid) return null

        return validationError(
            message = errors.asMessage() ?: "Unknown error occurred",
            schemaVersion = schemaVersion,
        )
    }

    private const val SCHEMA_VERSION_KEY = $$"$joyfillSchemaVersion"
    private const val DOCUMENT_VERSION_KEY = "v"
}

private val parser = Json {
    ignoreUnknownKeys = true
    isLenient = true
}

private fun schemaFor(documentVersion: Version): JsonElement? {
    return try {
        parser.parseToJsonElement(firstJoyfillSchema)
    } catch (_: Exception) {
        null
    }
}

private fun List<ValidationError>.asMessage(): String? {
    val errors = this
    if (isEmpty()) return null

    return buildString {
        errors.forEachIndexed { index, error ->
            append("[$index] Message: ")
            appendLine(error.message)
            append("    Object path: ")
            appendLine(error.objectPath.toString().ifEmpty { "root" })
            append("    Schema path: ")
            appendLine(error.schemaPath.toString())
            val absoluteLocation = error.absoluteLocation?.path?.toString()
            if (!absoluteLocation.isNullOrEmpty()) {
                append("    Absolute location: ")
                appendLine(error.absoluteLocation?.path?.toString())
            }
            if (error.details.isNotEmpty()) {
                append("    Other details: ")
                appendLine(error.details.toString())
            }
        }
    }
}