package joyfill2.utils

import joyfill2.Mappable

class JsonList<T : Mappable>(
    private val wrapped: MutableList<MutableMap<String, Any?>>,
    private val factory: (MutableMap<String, Any?>) -> T
) : AbstractList<T>(), MutableList<T> {

    private val cache = mutableMapOf<Int, T>()

    constructor(
        wrapped: Any?,
        factory: (MutableMap<String, Any?>) -> T
    ) : this(wrapped as MutableList<MutableMap<String, Any?>>, factory)

    override val size: Int get() = wrapped.size

    override fun addAll(elements: Collection<T>): Boolean {
        val result = wrapped.addAll(elements.map { it.toMap() })
        clearCache()
        return result
    }

    override fun addAll(index: Int, elements: Collection<T>): Boolean {
        val result = wrapped.addAll(index, elements.map { it.toMap() })
        clearCache()
        return result
    }

    override fun add(index: Int, element: T) {
        wrapped.add(index, element.toMap())
        shiftCacheIndexes(index, 1)
    }

    override fun add(element: T): Boolean {
        val result = wrapped.add(element.toMap())
        return result
    }

    override fun get(index: Int): T {
        return cache.getOrPut(index) {
            factory(wrapped[index])
        }
    }

    private fun clearCache() {
        cache.clear()
    }

    private fun shiftCacheIndexes(fromIndex: Int, offset: Int) {
        if (offset == 0) return

        val newCache = mutableMapOf<Int, T>()

        cache.forEach { (index, value) ->
            if (index < fromIndex) {
                newCache[index] = value
            } else {
                newCache[index + offset] = value
            }
        }

        cache.clear()
        cache.putAll(newCache)
    }

    override fun iterator(): MutableIterator<T> = CachingIterator()

    override fun listIterator(): MutableListIterator<T> = CachingListIterator(0)

    override fun listIterator(index: Int): MutableListIterator<T> = CachingListIterator(index)

    override fun removeAt(index: Int): T {
        val result = get(index)
        wrapped.removeAt(index)
        shiftCacheIndexes(index + 1, -1)
        cache.remove(index)
        return result
    }

    override fun set(index: Int, element: T): T {
        val oldValue = get(index)
        wrapped.set(index, element.toMap())
        cache[index] = element
        return oldValue
    }

    override fun retainAll(elements: Collection<T>): Boolean {
        val elementMaps = elements.map { it.toMap() }
        val result = wrapped.retainAll { it in elementMaps }
        clearCache()
        return result
    }

    override fun removeAll(elements: Collection<T>): Boolean {
        val elementMaps = elements.map { it.toMap() }
        val result = wrapped.removeAll { it in elementMaps }
        clearCache()
        return result
    }

    override fun remove(element: T): Boolean {
        val index = wrapped.indexOfFirst { it == element.toMap() }
        if (index >= 0) {
            removeAt(index)
            return true
        }
        return false
    }

    override fun subList(fromIndex: Int, toIndex: Int): MutableList<T> {
        val subWrapped = wrapped.subList(fromIndex, toIndex)
        return JsonList(subWrapped, factory)
    }

    override fun clear() {
        wrapped.clear()
        clearCache()
    }

    private inner class CachingIterator : MutableIterator<T> {
        private var index = 0

        override fun hasNext(): Boolean = index < size

        override fun next(): T {
            if (!hasNext()) throw NoSuchElementException()
            return get(index++)
        }

        override fun remove() {
            if (index == 0) throw IllegalStateException()
            removeAt(--index)
        }
    }

    private inner class CachingListIterator(initialIndex: Int) : MutableListIterator<T> {
        private var cursor = initialIndex
        private var lastRet = -1

        override fun hasNext(): Boolean = cursor < size

        override fun hasPrevious(): Boolean = cursor > 0

        override fun next(): T {
            if (!hasNext()) throw NoSuchElementException()
            lastRet = cursor
            return get(cursor++)
        }

        override fun nextIndex(): Int = cursor

        override fun previous(): T {
            if (!hasPrevious()) throw NoSuchElementException()
            lastRet = --cursor
            return get(cursor)
        }

        override fun previousIndex(): Int = cursor - 1

        override fun add(element: T) {
            add(cursor++, element)
            lastRet = -1
        }

        override fun remove() {
            if (lastRet < 0) throw IllegalStateException()
            removeAt(lastRet)
            if (lastRet < cursor) cursor--
            lastRet = -1
        }

        override fun set(element: T) {
            if (lastRet < 0) throw IllegalStateException()
            this@JsonList.set(lastRet, element)
        }
    }
}