package joyfill2.utils

import joyfill2.Mappable

//class JsonList<T : Mappable>(
//    private val wrapped: MutableList<MutableMap<String, Any?>>,
//    private val factory: (MutableMap<String, Any?>) -> T
//) : AbstractList<T>(), MutableList<T> {
//
//    constructor(
//        wrapped: Any?,
//        factory: (MutableMap<String, Any?>) -> T
//    ) : this(wrapped as MutableList<MutableMap<String, Any?>>, factory)
//
//    override val size: Int get() = wrapped.size
//    override fun addAll(elements: Collection<T>) = wrapped.addAll(elements.map { it.toMap() })
//
//    override fun addAll(index: Int, elements: Collection<T>) = wrapped.addAll(index, elements.map { it.toMap() })
//
//    override fun add(index: Int, element: T) = wrapped.add(index, element.toMap())
//
//    override fun add(element: T) = wrapped.add(element.toMap())
//
//    override fun get(index: Int): T = factory(wrapped[index])
//
//    protected fun entries() = wrapped.map { factory(it) }.toMutableList()
//
//    override fun iterator(): MutableIterator<T> = entries().iterator()
//
//    override fun listIterator(): MutableListIterator<T> = entries().listIterator()
//
//    override fun listIterator(index: Int): MutableListIterator<T> = entries().listIterator(index)
//
//    override fun removeAt(index: Int): T = factory(wrapped.removeAt(index))
//
//    override fun set(index: Int, element: T): T = factory(wrapped.set(index, element.toMap()))
//
//    override fun retainAll(elements: Collection<T>): Boolean = wrapped.retainAll(elements.map { it.toMap() })
//
//    override fun removeAll(elements: Collection<T>): Boolean = wrapped.removeAll(elements.map { it.toMap() })
//
//    override fun remove(element: T): Boolean = wrapped.remove(element.toMap())
//
//    override fun subList(fromIndex: Int, toIndex: Int): MutableList<T> =
//        wrapped.subList(fromIndex, toIndex).map { factory(it) }.toMutableList()
//
//    override fun clear() = wrapped.clear()
//}

class JsonList<T : Mappable>(
    private val wrapped: MutableList<MutableMap<String, Any?>>,
    private val factory: (MutableMap<String, Any?>) -> T
) : AbstractList<T>(), MutableList<T> {

    private val objectCache = mutableMapOf<Int, T>()

    constructor(
        wrapped: Any?,
        factory: (MutableMap<String, Any?>) -> T
    ) : this(wrapped as MutableList<MutableMap<String, Any?>>, factory)

    override val size: Int get() = wrapped.size

    override fun addAll(elements: Collection<T>): Boolean {
        val result = wrapped.addAll(elements.map { it.toMap() })
        clearCache()
        return result
    }

    override fun addAll(index: Int, elements: Collection<T>): Boolean {
        val result = wrapped.addAll(index, elements.map { it.toMap() })
        clearCache()
        return result
    }

    override fun add(index: Int, element: T) {
        wrapped.add(index, element.toMap())
        shiftCacheIndexes(index, 1)
    }

    override fun add(element: T): Boolean {
        val result = wrapped.add(element.toMap())
        return result
    }

    override fun get(index: Int): T {
        return objectCache.getOrPut(index) {
            factory(wrapped[index])
        }
    }

    private fun clearCache() {
        objectCache.clear()
    }

    private fun shiftCacheIndexes(fromIndex: Int, offset: Int) {
        if (offset == 0) return

        val newCache = mutableMapOf<Int, T>()

        objectCache.forEach { (index, value) ->
            if (index < fromIndex) {
                newCache[index] = value
            } else {
                newCache[index + offset] = value
            }
        }

        objectCache.clear()
        objectCache.putAll(newCache)
    }

    override fun iterator(): MutableIterator<T> = CachingIterator()

    override fun listIterator(): MutableListIterator<T> = CachingListIterator(0)

    override fun listIterator(index: Int): MutableListIterator<T> = CachingListIterator(index)

    override fun removeAt(index: Int): T {
        val result = get(index)
        wrapped.removeAt(index)
        shiftCacheIndexes(index + 1, -1)
        objectCache.remove(index)
        return result
    }

    override fun set(index: Int, element: T): T {
        val oldValue = get(index)
        wrapped.set(index, element.toMap())
        objectCache[index] = element
        return oldValue
    }

    override fun retainAll(elements: Collection<T>): Boolean {
        val elementMaps = elements.map { it.toMap() }
        val result = wrapped.retainAll { it in elementMaps }
        clearCache()
        return result
    }

    override fun removeAll(elements: Collection<T>): Boolean {
        val elementMaps = elements.map { it.toMap() }
        val result = wrapped.removeAll { it in elementMaps }
        clearCache()
        return result
    }

    override fun remove(element: T): Boolean {
        val index = wrapped.indexOfFirst { it == element.toMap() }
        if (index >= 0) {
            removeAt(index)
            return true
        }
        return false
    }

    override fun subList(fromIndex: Int, toIndex: Int): MutableList<T> {
        val subWrapped = wrapped.subList(fromIndex, toIndex)
        return JsonList(subWrapped, factory)
    }

    override fun clear() {
        wrapped.clear()
        clearCache()
    }

    private inner class CachingIterator : MutableIterator<T> {
        private var index = 0

        override fun hasNext(): Boolean = index < size

        override fun next(): T {
            if (!hasNext()) throw NoSuchElementException()
            return get(index++)
        }

        override fun remove() {
            if (index == 0) throw IllegalStateException()
            removeAt(--index)
        }
    }

    private inner class CachingListIterator(initialIndex: Int) : MutableListIterator<T> {
        private var cursor = initialIndex
        private var lastRet = -1

        override fun hasNext(): Boolean = cursor < size

        override fun hasPrevious(): Boolean = cursor > 0

        override fun next(): T {
            if (!hasNext()) throw NoSuchElementException()
            lastRet = cursor
            return get(cursor++)
        }

        override fun nextIndex(): Int = cursor

        override fun previous(): T {
            if (!hasPrevious()) throw NoSuchElementException()
            lastRet = --cursor
            return get(cursor)
        }

        override fun previousIndex(): Int = cursor - 1

        override fun add(element: T) {
            add(cursor++, element)
            lastRet = -1
        }

        override fun remove() {
            if (lastRet < 0) throw IllegalStateException()
            removeAt(lastRet)
            if (lastRet < cursor) cursor--
            lastRet = -1
        }

        override fun set(element: T) {
            if (lastRet < 0) throw IllegalStateException()
            this@JsonList.set(lastRet, element)
        }
    }
}