package joyfill.table.internal

import joyfill.ChangeEvent
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.editors.internal.TableFieldEditorImpl
import joyfill.fields.Field
import joyfill.fields.table.BarcodeColumn
import joyfill.fields.table.BlockColumn
import joyfill.fields.table.Column
import joyfill.fields.table.DateColumn
import joyfill.fields.table.DropdownColumn
import joyfill.fields.table.ImageColumn
import joyfill.fields.table.MultiselectColumn
import joyfill.fields.table.NumberColumn
import joyfill.fields.table.ProgressColumn
import joyfill.fields.table.Row
import joyfill.fields.table.SignatureColumn
import joyfill.fields.table.TextColumn
import joyfill.table.BarcodeCellEditor
import joyfill.table.BlockCellEditor
import joyfill.table.CellEditor
import joyfill.table.DateCellEditor
import joyfill.table.DropdownCellEditor
import joyfill.table.ImageCellEditor
import joyfill.table.MultiselectCellEditor
import joyfill.table.NumberCellEditor
import joyfill.table.ProgressCellEditor
import joyfill.table.RowEditor
import joyfill.table.SignatureCellEditor
import joyfill.table.TextCellEditor

internal class RowEditorImpl(
    val document: Document,
    val table: TableFieldEditorImpl,
    val identity: IdentityGenerator,
    override val row: Row,
    val onChange: ((ChangeEvent) -> Unit)?
) : RowEditor {

    private val field by lazy { table.field }

    private fun <C : Column> find(key: String?, type: Field.Type) = field.columns.find {
        (it.id == key || it.title == key) && it.type == type
    } as? C

    private fun <C : Column> take(index: Int, type: Field.Type): C? {
        val column = field.columns.getOrNull(index) ?: return null
        if (column.type != type) return null
        return column as? C
    }

    private fun id(column: Column) = "${row.id}:${column.id}"

    override fun text(key: String?): TextCellEditor? {
        val column = find<TextColumn>(key, Field.Type.text)
        column ?: return null
        return cache.getOrPut(id(column)) { TextCellEditorImpl(document, table, column, row, onChange) } as TextCellEditor
    }

    override fun text(index: Int): TextCellEditor? {
        val column = take<TextColumn>(index, Field.Type.text) ?: return null
        return cache.getOrPut(id(column)) { TextCellEditorImpl(document, table, column, row, onChange) } as TextCellEditor
    }

    override fun date(key: String): DateCellEditor? {
        val column = find<DateColumn>(key, Field.Type.date) ?: return null
        return cache.getOrPut(id(column)) { DateCellEditorImpl(document, table, column, row, onChange) } as DateCellEditor
    }

    override fun date(index: Int): DateCellEditor? {
        val column = take<DateColumn>(index, Field.Type.date) ?: return null
        return cache.getOrPut(id(column)) { DateCellEditorImpl(document, table, column, row, onChange) } as DateCellEditor
    }

    override fun blockText(key: String): BlockCellEditor? {
        val column = find<BlockColumn>(key, Field.Type.block) ?: return null
        return cache.getOrPut(id(column)) { BlockCellEditorImpl(document, table, column, row, onChange) } as BlockCellEditor
    }

    override fun blockText(index: Int): BlockCellEditor? {
        val column = take<BlockColumn>(index, Field.Type.text) ?: return null
        return cache.getOrPut(id(column)) { BlockCellEditorImpl(document, table, column, row, onChange) } as BlockCellEditor
    }

    override fun progress(key: String): ProgressCellEditor? {
        val column = find<ProgressColumn>(key, Field.Type.progress) ?: return null
        return cache.getOrPut(id(column)) { ProgressCellEditorImpl(document, table, column, row, onChange) } as ProgressCellEditor
    }

    override fun progress(index: Int): ProgressCellEditor? {
        val column = take<ProgressColumn>(index, Field.Type.progress) ?: return null
        return cache.getOrPut(id(column)) { ProgressCellEditorImpl(document, table, column, row, onChange) } as ProgressCellEditor
    }

    override fun number(key: String): NumberCellEditor? {
        val column = find<NumberColumn>(key, Field.Type.number) ?: return null
        return cache.getOrPut(id(column)) { NumberCellEditorImpl(document, table, column, row, onChange) } as NumberCellEditor
    }

    override fun number(index: Int): NumberCellEditor? {
        val column = take<NumberColumn>(index, Field.Type.number) ?: return null
        return cache.getOrPut(id(column)) { NumberCellEditorImpl(document, table, column, row, onChange) } as NumberCellEditor
    }

    override fun image(key: String?): ImageCellEditor? {
        val column = find<ImageColumn>(key, Field.Type.image) ?: return null
        return cache.getOrPut(id(column)) { ImageCellEditorImpl(document, table, identity, column, row, onChange) } as ImageCellEditor
    }

    override fun image(index: Int): ImageCellEditor? {
        val column = take<ImageColumn>(index, Field.Type.image) ?: return null
        return cache.getOrPut(id(column)) { ImageCellEditorImpl(document, table, identity, column, row, onChange) } as ImageCellEditor
    }

    override fun barcode(key: String?): BarcodeCellEditor? {
        val column = find<BarcodeColumn>(key, Field.Type.barcode) ?: return null
        return cache.getOrPut(id(column)) { BarcodeCellEditorImpl(document, table, column, row, onChange) } as BarcodeCellEditor
    }

    override fun barcode(index: Int): BarcodeCellEditor? {
        val column = take<BarcodeColumn>(index, Field.Type.barcode) ?: return null
        return cache.getOrPut(id(column)) { BarcodeCellEditorImpl(document, table, column, row, onChange) } as BarcodeCellEditor
    }

    override fun signature(key: String?): SignatureCellEditor? {
        val column = find<SignatureColumn>(key, Field.Type.signature) ?: return null
        return cache.getOrPut(id(column)) { SignatureCellEditorImpl(document, table, column, row, onChange) } as SignatureCellEditor
    }

    override fun signature(index: Int): SignatureCellEditor? {
        val column = take<SignatureColumn>(index, Field.Type.signature) ?: return null
        return cache.getOrPut(id(column)) { SignatureCellEditorImpl(document, table, column, row, onChange) } as SignatureCellEditor
    }

    override fun dropdown(key: String?): DropdownCellEditor? {
        val column = find<DropdownColumn>(key, Field.Type.dropdown) ?: return null
        return cache.getOrPut(id(column)) { DropdownCellEditorImpl(document, table, column, row, onChange) } as DropdownCellEditor
    }

    override fun dropdown(index: Int): DropdownCellEditor? {
        val column = take<DropdownColumn>(index, Field.Type.dropdown) ?: return null
        return cache.getOrPut(id(column)) { DropdownCellEditorImpl(document, table, column, row, onChange) } as DropdownCellEditor
    }

    override fun multiSelect(key: String?): MultiselectCellEditor? {
        val column = find<MultiselectColumn>(key, Field.Type.multiSelect) ?: return null
        return cache.getOrPut(id(column)) { MultiselectCellEditorImpl(document, table, column, row, onChange) } as MultiselectCellEditor
    }

    override fun multiSelect(index: Int): MultiselectCellEditor? {
        val column = take<MultiselectColumn>(index, Field.Type.multiSelect) ?: return null
        return cache.getOrPut(id(column)) { MultiselectCellEditorImpl(document, table, column, row, onChange) } as MultiselectCellEditor
    }

    private fun Column.toEditor(): CellEditor? = when (this) {
        is TextColumn -> cache.getOrPut(id(this)) { TextCellEditorImpl(document, table, this, row, onChange) }
        is BlockColumn -> cache.getOrPut(id(this)) { BlockCellEditorImpl(document, table, this, row, onChange) }
        is DateColumn -> cache.getOrPut(id(this)) { DateCellEditorImpl(document, table, this, row, onChange) }
        is NumberColumn -> cache.getOrPut(id(this)) { NumberCellEditorImpl(document, table, this, row, onChange) }
        is ImageColumn -> cache.getOrPut(id(this)) { ImageCellEditorImpl(document, table, identity, this, row, onChange) }
        is BarcodeColumn -> cache.getOrPut(id(this)) { BarcodeCellEditorImpl(document, table, this, row, onChange) }
        is SignatureColumn -> cache.getOrPut(id(this)) { SignatureCellEditorImpl(document, table, this, row, onChange) }
        is DropdownColumn -> cache.getOrPut(id(this)) { DropdownCellEditorImpl(document, table, this, row, onChange) }
        is MultiselectColumn -> cache.getOrPut(id(this)) { MultiselectCellEditorImpl(document, table, this, row, onChange) }
        is ProgressColumn -> cache.getOrPut(id(this)) { ProgressCellEditorImpl(document, table, this, row, onChange) }
        else -> null
    }

    override fun col(key: String?) = field.columns.find {
        it.id == key || it.title == key
    }?.toEditor()

    override fun col(index: Int) = field.columns.getOrNull(index)?.toEditor()

    private fun copyInto(other: RowEditor, column: Column) {
        when (column) {
            is TextColumn -> {
                val value = text(column.id)?.text?.value
                if (value.isNullOrBlank()) return
                other.text(column.id)?.set(value) ?: return
            }

            is DateColumn -> {
                val value = date(column.id)?.date?.value ?: return
                other.date(column.id)?.set(value)
            }

            is BlockColumn -> {
                val value = blockText(column.id)?.blockText?.value ?: return
                other.blockText(column.id)?.set(value)
            }

            is NumberColumn -> {
                val value = number(column.id)?.number?.value ?: return
                other.number(column.id)?.set(value)
            }

            is DropdownColumn -> {
                val selected = dropdown(column.id)?.selected()
                val cell = other.dropdown(column.id) ?: return
                if (selected != null) {
                    cell.select(selected)
                } else {
                    cell.unselect()
                }
            }

            is MultiselectColumn -> {
                val selected = multiSelect(column.id)?.selected()
                val cell = other.multiSelect(column.id) ?: return
                if (selected != null) {
                    cell.selectOptions(selected)
                } else {
                    cell.unselectAll()
                }
            }

            is ImageColumn -> {
                val images = image(column.id)?.value ?: return
                other.image(column.id)?.add(images.map { it.url }) ?: return
            }

            is BarcodeColumn -> {
                val value = barcode(column.id)?.barcode?.value
                if (value.isNullOrBlank()) return
                other.barcode(column.id)?.set(value) ?: return
            }

            is SignatureColumn -> {
                val value = signature(column.id)?.signature?.value
                if (value.isNullOrBlank()) return
                other.signature(column.id)?.set(value) ?: return
            }

            is ProgressColumn -> {
                val value = progress(column.id)?.progress?.value ?: return
                other.progress(column.id)?.set(value)
            }

            else -> {}
        }
    }

    override fun copyInto(other: RowEditor) {
        for (column in field.columns) copyInto(other, column)
    }

    private val cache = mutableMapOf<String, CellEditor>()
}