package joyfill.table.internal.tools

import com.ionspin.kotlin.bignum.decimal.BigDecimal
import joyfill.table.ColumnFilter
import joyfill.table.RowEditor
import joyfill.table.SortingOption

internal fun Collection<RowEditor>.thatMatch(filter: ColumnFilter) = when (filter) {
    is ColumnFilter.DropdownColumnFilter -> thatMatch(filter)
    is ColumnFilter.MultiselectColumnFilter -> thatMatch(filter)
    is ColumnFilter.TextColumnFilter -> thatMatch(filter)
    is ColumnFilter.NumberColumnFilter -> thatMatch(filter)
    is ColumnFilter.BarcodeColumnFilter -> thatMatch(filter)
}

internal fun ColumnFilter.TextColumnFilter.toFunction() = { editor: RowEditor ->
    editor.text(column.id)?.text?.value
}

internal fun Collection<RowEditor>.thatMatch(predicate: ColumnFilter.TextColumnFilter): List<RowEditor> {
    val filtered = filter {
        if (predicate.value == null) return@filter true
        it.text(predicate.column.id)?.text?.value?.contains(predicate.value, ignoreCase = true) == true
    }
    return when (predicate.sort) {
        SortingOption.None -> filtered
        SortingOption.Ascending -> filtered.sortedBy(predicate.toFunction())
        SortingOption.Descending -> filtered.sortedByDescending(predicate.toFunction())
    }
}

internal fun ColumnFilter.NumberColumnFilter.toFunction() = { editor: RowEditor ->
    editor.number(column.id)?.number?.value
}

internal fun Collection<RowEditor>.thatMatch(predicate: ColumnFilter.NumberColumnFilter): List<RowEditor> {
    val filtered = filter {
        if (predicate.value == null) return@filter true
        val  number = it.number(predicate.column.id)?.number?.value
        if (number == null) return@filter false
        BigDecimal.fromDouble(number).toPlainString().startsWith(predicate.value)
    }

    return when (predicate.sort) {
        SortingOption.None -> filtered
        SortingOption.Ascending -> filtered.sortedBy(predicate.toFunction())
        SortingOption.Descending -> filtered.sortedByDescending(predicate.toFunction())
    }
}

internal fun ColumnFilter.BarcodeColumnFilter.toFunction() = { editor: RowEditor ->
    editor.barcode(column.id)?.barcode?.value
}

internal fun Collection<RowEditor>.thatMatch(predicate: ColumnFilter.BarcodeColumnFilter): List<RowEditor> {
    val filtered = filter {
        if (predicate.value == null) return@filter true
        val  value = it.barcode(predicate.column.id)?.barcode?.value?.toString()?: ""
        value.contains(predicate.value, ignoreCase = true)
    }
    return when (predicate.sort) {
        SortingOption.None -> filtered
        SortingOption.Ascending -> filtered.sortedBy(predicate.toFunction())
        SortingOption.Descending -> filtered.sortedByDescending(predicate.toFunction())
    }
}

internal fun ColumnFilter.DropdownColumnFilter.toFunction() = { editor: RowEditor ->
    editor.dropdown(column.id)?.selected()?.value
}

internal fun Collection<RowEditor>.thatMatch(predicate: ColumnFilter.DropdownColumnFilter): List<RowEditor> {
    val filtered = filter {
        if (predicate.value == null) return@filter true
        it.dropdown(predicate.column.id)?.selected()?.id == predicate.value.id
    }
    return when (predicate.sort) {
        SortingOption.None -> filtered
        SortingOption.Ascending -> filtered.sortedBy(predicate.toFunction())
        SortingOption.Descending -> filtered.sortedByDescending(predicate.toFunction())
    }
}

internal fun Collection<RowEditor>.thatMatch(predicate: ColumnFilter.MultiselectColumnFilter): List<RowEditor> {
    val filtered = filter {
        if (predicate.value == null) return@filter true
        it.multiSelect(predicate.column.id)?.selected()?.map { it.id }?.contains( predicate.value.id) == true
    }
    return  filtered //for now sorting is not requirement
}