package joyfill.editors.internal

import joyfill.ChangeEvent
import joyfill.Document
import joyfill.EventDispatcher
import joyfill.collections.PageCollection
import joyfill.editors.MultiSelectFieldEditor
import joyfill.fields.MultiSelectField
import joyfill.utils.Option2
import joyfill.validation.FieldInvalid
import joyfill.validation.FieldValid
import joyfill.validation.FieldValidity

@PublishedApi
internal class MultiSelectFieldEditorImpl(
    document: Document,
    override val field: MultiSelectField,
    pages: PageCollection,
    dispatcher: EventDispatcher,
    onChange: ((ChangeEvent) -> Unit)?
) : AnyFieldEditor<MultiSelectField>(document, field, pages, dispatcher, onChange), MultiSelectFieldEditor {

    override val options: List<Option2> get() = this.field.options

    private fun look(key: String?): Option2? = this.field.options.firstOrNull { it.id == key || it.value == key }

    override fun selected(): List<Option2> {
        val found = mutableListOf<Option2>()
        val value = this.field.value ?: return emptyList()
        for (option in value) found.add(look(option) ?: continue)
        return found
    }

    override fun select(key: String?) {
        val option = look(key) ?: return
        val selected = this.field.value.toMutableSet()
        if (selected.contains(option.id)) return
        if (!field.multi) field.value.clear()
        field.value.add(option.id)
        resolveConditionsAndNotifyChange()
    }

    private fun resolveConditionsAndNotifyChange() {
        resolveConditions()
        notifyChange(field.value)
    }

    override fun set(options: List<Option2>) {
        val pool = field.options.map { it.id }
        val candidates = options.filter { pool.contains(it.id) }.toSet().map { it.id }.toMutableList()
        field.value.clear()
        field.value.addAll(candidates)
        resolveConditionsAndNotifyChange()
    }

    override fun unselect(key: String?) {
        val option = look(key) ?: return
        val selected = this.field.value.toMutableSet()
        if (!selected.contains(option.id)) return
        field.value.remove(option.id)
        resolveConditionsAndNotifyChange()
    }

    override fun unselect(option: Option2?) = unselect(option?.id)

    override fun select(option: Option2?) = select(option?.id)

    override fun validate(): FieldValidity {
        if (hidden.value || !field.required) return FieldValid(field, emptyList())
        if (field.required && field.value.isEmpty()) return FieldInvalid(field, listOf("Field ${field.title} is required"))
        return FieldValid(field, emptyList())
    }
}