package joyfill.table.internal

import cinematic.mutableLiveOf
import joyfill.ChangeEvent
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.editors.internal.TableFieldEditorImpl
import joyfill.fields.TableField
import joyfill.fields.table.BarcodeColumn
import joyfill.fields.table.BlockColumn
import joyfill.fields.table.DateColumn
import joyfill.fields.table.DropdownColumn
import joyfill.fields.table.ImageColumn
import joyfill.fields.table.MultiselectColumn
import joyfill.fields.table.NumberColumn
import joyfill.fields.table.ProgressColumn
import joyfill.fields.table.Row
import joyfill.fields.table.SignatureColumn
import joyfill.fields.table.TextColumn
import joyfill.table.ColumnFilter
import joyfill.table.RowCollection
import joyfill.table.RowEditor
import joyfill.table.RowMover
import joyfill.table.RowResult
import joyfill.table.RowState
import joyfill.table.Selection
import joyfill.table.internal.tools.thatMatch
import joyfill.toRow
import joyfill.utils.ID
import joyfill.utils.RawAttachment
import joyfill.utils.toAttachment
import kotlin.math.max

@PublishedApi
internal class RowCollectionImpl(
    private val document: Document,
    private val field: TableField,
    private val parent: TableFieldEditorImpl,
    private val identity: IdentityGenerator,
    private val onChange: ((ChangeEvent) -> Unit)?
) : RowCollection {

    override val size: Int get() = this.field.value.size

    override val state by lazy {
        val list = all().filter { !it.row.deleted && it.row.id in field.rowOrder }
        mutableLiveOf(RowState(selected = emptyList(), filtered = null, haystack = list))
    }

    override fun all(): List<RowEditor> = field.value.map {
        cache.getOrPut(it.id) { RowEditorImpl(document, parent, identity, it, onChange) }
    }

    override fun create(setDefaults: Boolean): RowEditor {
        val cells = mutableMapOf<String, Any?>()
        for (column in field.columns) when (column) {
            is TextColumn -> cells[column.id] = if(setDefaults) column.value else null
            is DateColumn -> cells[column.id] = if(setDefaults) column.value else null
            is NumberColumn -> cells[column.id] = if(setDefaults) column.value else null
            is DropdownColumn -> cells[column.id] = if(setDefaults) column.value else null
            is ImageColumn -> cells[column.id] = if(setDefaults) column.value.map { it.toMap() }.toMutableList() else null
            is BarcodeColumn -> cells[column.id] = if(setDefaults) column.value else null
            is SignatureColumn -> cells[column.id] = if(setDefaults) column.value else null
            else -> cells[column.id] = null
        }
        val row = mutableMapOf<String, Any?>(
            ID to identity.generate(),
            Row::cells.name to cells
        ).toRow()
        return RowEditorImpl(document, parent, identity, row, null).also { cache[it.row.id] = it }
    }

    private fun create(index: Int): Row {
        val cells = mutableMapOf<String, Any?>()
        for (column in field.columns) when (column) {
            is TextColumn -> cells[column.id] = column.value
            is DateColumn -> cells[column.id] = column.value
            is NumberColumn -> cells[column.id] = column.value
            is DropdownColumn -> cells[column.id] = column.value
            is ImageColumn -> cells[column.id] = column.value.map { it.toMap() }.toMutableList()
            is BarcodeColumn -> cells[column.id] = column.value
            is SignatureColumn -> cells[column.id] = column.value
            else -> cells[column.id] = null
        }
        return put(index, cells)
    }

    private fun put(index: Int, cells: MutableMap<String, Any?>): Row {
        val row = mutableMapOf<String, Any?>(
            ID to identity.generate(),
            Row::cells.name to cells
        ).toRow()
        parent.value.add(index, row)
        field.rowOrder.add(index, row.id)
        return row
    }

    @Deprecated("Ditch index based, In favour of overload with id")
    override fun addAt(index: Int): RowEditor {
        val row = create(index)
        val editor = cache.getOrPut(row.id) { RowEditorImpl(document, parent, identity, row, onChange) }
        val list = state.value.haystack.toMutableList()
        list.add(index, editor)
//        state.value = state.value.copy(selected = emptyList(), haystack = list)
        list.applyFilters()
        return editor
    }


    private fun duplicate(src: Row, index: Int): Row {
        val cells = mutableMapOf<String, Any?>()
        for (column in field.columns) when (column) {
            is TextColumn -> cells[column.id] = src.cells[column.id]
            is DateColumn -> cells[column.id] = src.cells[column.id]
            is NumberColumn -> cells[column.id] = src.cells[column.id]
            is DropdownColumn -> cells[column.id] = src.cells[column.id]
            is ImageColumn -> cells[column.id] = (src.cells[column.id] as? List<MutableMap<String, Any?>>)?.map {
                val at = it.toAttachment()
                RawAttachment(id = at.id, url = at.url, fileName = at.fileName, filePath = at.filePath, download = at.download)
            }?.toMutableList()
            is ProgressColumn -> cells[column.id] = src.cells[column.id]
            is BarcodeColumn -> cells[column.id] = src.cells[column.id]
            is SignatureColumn -> cells[column.id] = src.cells[column.id]

            else -> cells[column.id] = null
        }
        return put(index + 1, cells)
    }

    private fun duplicate(id: String): RowResult<Row> {
        val src = field.value.find { it.id == id } ?: throw IllegalArgumentException("Could not find row with id=$id")
        val cells = mutableMapOf<String, Any?>()
        for (column in field.columns) when (column) {
            is TextColumn -> cells[column.id] = src.cells[column.id]
            is BlockColumn -> cells[column.id] = src.cells[column.id]
            is DateColumn -> cells[column.id] = src.cells[column.id]
            is NumberColumn -> cells[column.id] = src.cells[column.id]
            is DropdownColumn -> cells[column.id] = src.cells[column.id]
            is MultiselectColumn -> cells[column.id] = src.cells[column.id]
            is ImageColumn -> cells[column.id] = (src.cells[column.id] as? List<MutableMap<String, Any?>>)?.map {
                val at = it.toAttachment()
                RawAttachment(id = at.id, url = at.url, fileName = at.fileName, filePath = at.filePath, download = at.download)
            }?.toMutableList()
            is ProgressColumn -> cells[column.id] = src.cells[column.id]
            is BarcodeColumn -> cells[column.id] = src.cells[column.id]
            is SignatureColumn -> cells[column.id] = src.cells[column.id]

            else -> cells[column.id] = null
        }
        val index = field.rowOrder.indexOfFirst { it == id }
        return RowResult(row = put(index + 1, cells), index = index + 1)
    }

    @Deprecated("Ditch index based, In favour of overload with id")
    override fun duplicate(index: Int): Row? {
        val src = get(index)?.row ?: return null
        val row = duplicate(src, index)
        val editor = cache.getOrPut(row.id) { RowEditorImpl(document, parent, identity, row, onChange) }
        val list = state.value.haystack.toMutableList()
        list.add(index + 1, editor)
        state.value = state.value.copy(selected = emptyList(), haystack = list)
        return row
    }

    override fun duplicate(keys: List<String>): List<RowEditor> {
        if (keys.isEmpty()) return emptyList()
        val haystack = state.value.haystack.toMutableList()
        val duplicates = keys.map {
            duplicate(it)
        }.sortedBy {
            it.index
        }.map {
            val editor = cache.getOrPut(it.row.id) { RowEditorImpl(document, parent, identity, it.row, onChange) }
            haystack.add(it.index, editor)
            editor
        }
        haystack.applyFilters()
        return duplicates
    }

    fun List<RowEditor>.applyFilters() {
        var filtered = this
        for (filter in parent.columns.state.value.filters) {
            filtered = filtered.thatMatch(filter)
        }
        state.value = if (filtered.isNotEmpty()) {
            state.value.copy(selected = emptyList(), filtered = filtered, haystack = this)
        } else {
            state.value.copy(selected = emptyList(), filtered = if (parent.columns.areFiltersApplied()) emptyList() else null, haystack = this)
        }
    }

    override fun append(setDefaults: Boolean): RowEditor {
        val editor = create(setDefaults)
        val haystack = state.value.haystack.toMutableList()
        haystack.add(editor)

        for (filter in parent.columns.state.value.filters) when (filter) {
            is ColumnFilter.TextColumnFilter -> editor.text(filter.column.id)?.set(filter.value)
            is ColumnFilter.NumberColumnFilter -> {
                val filterText = filter.value?.replace(">", "")?.replace("<", "")?.replace("=", "")
                val filterNumber = filterText?.toDoubleOrNull()

                if (filterNumber != null) {
                    editor.number(filter.column.id)?.set(filterNumber)
                }
            }

            is ColumnFilter.DropdownColumnFilter -> editor.dropdown(filter.column.id)?.select(filter.value)
            is ColumnFilter.MultiselectColumnFilter -> {
                val opts = if(filter.value != null) listOf(filter.value) else emptyList()
                editor.multiSelect(filter.column.id)?.selectOptions(opts)
            }
            is ColumnFilter.BarcodeColumnFilter -> editor.barcode(filter.column.id)?.set(filter.value)
        }

        for (column in field.columns) when (column) {
            //If block cell value is empty and column value is not take column value
            is BlockColumn -> {
                val cellVal = editor.blockText(column.id)?.blockText?.value
                val columnVal = column.value
                if (cellVal.isNullOrBlank() && !columnVal.isNullOrBlank()) {
                    editor.blockText(column.id)?.set(columnVal)
                }
                break
            }
        }

        parent.value.add(editor.row)
        parent.field.rowOrder.add(editor.row.id)
        haystack.applyFilters()
        return editor
    }

    @Deprecated("Ditch index based, In favour of overload with id")
    override fun addAfter(index: Int) = addAt(index + 1)

    override fun addAfter(id: String): RowEditor {
        val index = state.value.haystack.indexOfFirst { it.row.id == id }
        return addAt(index + 1)
    }

    @Deprecated("Ditch index based, In favour of overload with RowEditor")
    override fun addBefore(index: Int): RowEditor = addAt(max(index - 1, 0))

    override fun addBefore(id: String): RowEditor {
        val index = state.value.selected.indexOfFirst { it.row.id == id }
        return addBefore(index)
    }

    override fun get(index: Int): RowEditor? {
        val editor = state.value.haystack.toList().getOrNull(index) ?: return null
        return cache.getOrPut(editor.row.id) { editor }
    }

    private val cache by lazy { mutableMapOf<String, RowEditor>() }

    override fun get(id: String): RowEditor? = cache.getOrPut(id) {
        state.value.haystack.find { it.row.id == id } ?: return null
    }

    @Deprecated("Ditch index based, In favour of overload with id")
    override fun deleteAt(index: Int): Row? = try {
        val editor = locate(index) ?: error("Not found")
        parent.columns.state.value = parent.columns.state.value.copy(
            validity = parent.columns.state.value.validity.filterNot { it.key.row.id == editor.row.id }
        )
        parent.value.removeAll { it.id == editor.row.id }
        editor.row
    } catch (_: Exception) {
        null
    }

    override fun delete(keys: List<String>): List<Row> {
        val removed = keys.mapNotNull { get(it) }.toSet()
        parent.value.removeAll { it.id in keys }
        field.rowOrder.removeAll { it in keys }
        val rows = state.value
        val filtered = when (val them = rows.filtered) {
            null -> null
            else -> them - removed
        }
        parent.columns.state.value = parent.columns.state.value.copy(
            validity = parent.columns.state.value.validity.filterNot { it.key.row.id in keys }
        )
        state.value = state.value.copy(selected = emptyList(), filtered = filtered, haystack = rows.haystack - removed)
        for (row in removed) cache.remove(row.row.id)
        return removed.map { it.row }
    }

    private fun locate(index: Int): RowEditor? = state.value.displaying().getOrNull(index)

    override fun isSelected(index: Int): Boolean {
        val row = locate(index)
        return isSelected(row)
    }

    override fun isSelected(row: RowEditor?): Boolean {
        if (state.value.haystack.isEmpty()) return false
        if (state.value.selected.size == state.value.haystack.size) return true
        return state.value.selected.contains(row)
    }

    @Deprecated("Ditch index based, In favour of overload with RowEditor")
    override fun select(index: Int): RowEditor? = select(locate(index))

    override fun unselect(index: Int): RowEditor? = unselect(locate(index))

    override fun unselect(row: RowEditor?): RowEditor? {
        row ?: return null
        state.value = state.value.copy(selected = state.value.selected - row)
        return row
    }

    override fun select(row: RowEditor?): RowEditor? {
        row ?: return null
        state.value = state.value.copy(selected = state.value.selected + row)
        return row
    }

    override fun delete() {
        if (state.value.selected.isEmpty()) return
        delete(state.value.selected.map { it.row.id })
    }

    override fun duplicate(): List<RowEditor> {
        val candidates = state.value.selected
        if (candidates.isEmpty()) return emptyList()
        duplicate(candidates.map { it.row.id })
        return candidates
    }

    override fun selectAll() {
        state.value = state.value.copy(selected = state.value.haystack)
    }

    override fun unSelectAll() {
        state.value = state.value.copy(selected = emptyList())
    }

    override fun selection(): Selection = state.value.selection()

    override fun selected(): List<RowEditor> = state.value.selected

    inner class RowMoverImpl : RowMover {
        private fun MutableList<RowEditor>.add(editor: RowEditor, index: Int): RowResult<RowEditor> {
            add(index, editor)
            field.rowOrder.add(index, editor.row.id)
            applyFilters()
            return RowResult(editor, index)
        }

        override fun top(id: String): RowResult<RowEditor> {
            val editor = get(id) ?: throw IllegalArgumentException("Could not find row with id=$id")
            val haystack = state.value.haystack.toMutableList()
            haystack.remove(editor)
            field.rowOrder.remove(editor.row.id)
            return haystack.add(editor = editor, index = 0)
        }

        override fun up(id: String, by: Int): RowResult<RowEditor> {
            val editor = get(id) ?: throw IllegalArgumentException("Could not find row with id=$id")
            val haystack = state.value.haystack.toMutableList()
            val current = haystack.indexOf(editor)
            val next = if (current < by) 0 else current - by
            haystack.remove(editor)
            field.rowOrder.remove(editor.row.id)
            return haystack.add(editor = editor, index = next)
        }

//        override fun up(ids: List<String>): RowResult<List<RowEditor>> {
//            TODO("Not yet implemented")
//        }

        override fun down(id: String, by: Int): RowResult<RowEditor> {
            val editor = get(id) ?: throw IllegalArgumentException("Could not find row with id=$id")
            val haystack = state.value.haystack.toMutableList()
            val size = haystack.size
            val current = haystack.indexOf(editor)
            val next = if (current + by > size) size else current + by
            haystack.remove(editor)
            field.rowOrder.remove(editor.row.id)
            return haystack.add(editor = editor, index = next)
        }

//        override fun down(ids: List<String>): RowResult<RowEditor> {
//            TODO("Not yet implemented")
//        }

        override fun bottom(id: String): RowResult<RowEditor> {
            val editor = get(id) ?: throw IllegalArgumentException("Could not find row with id=$id")
            val haystack = state.value.haystack.toMutableList()
            haystack.remove(editor)
            field.rowOrder.remove(editor.row.id)
            return haystack.add(editor = editor, index = haystack.size)
        }
    }

    override val move by lazy { RowMoverImpl() }
}