package joyfill.table

import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.FieldPosition
import joyfill.Mode
import joyfill.RawDropField
import joyfill.Valid
import joyfill.fields.Field
import joyfill.fields.table.Column
import joyfill.fields.table.MultiselectColumn
import joyfill.image.ImageCountPreview
import joyfill.image.RawImageField
import joyfill.image.UploadLabel

@Composable
internal fun TableCell(
    cell: CellEditor?,
    minHeight: Dp,
    visibleCols: List<Column>,
    rowsState: RowState,
    rowPlacement: RowPlacement?,
    position: FieldPosition?,
    mode: Mode,
    onUpload: (suspend () -> List<String>)?,
    onCapture: (suspend () -> String?)?
) {
    when (cell) {
        is ProgressCellEditor -> {
            val filteredCols =
                visibleCols.filter { it.required && it.type != Field.Type.progress }
            val rowEditor =
                rowsState.haystack.find { it.row.id == rowPlacement?.editor?.row?.id }

            val validCells = filteredCols.map { rowEditor?.col(it.id).also { it?.validate() } }
                .filter { it?.validity?.value is Valid }.size
            val totalCells = filteredCols.size

            JoyProgressCell(totalCells = totalCells, validCells = validCells)
        }

        is TextCellEditor -> JoyTextEditCell(
            value =  cell.text.watchAsState() ?: "",
            readonly = mode == Mode.readonly,
            borders = false,
            minLines = 3,
            modifier = Modifier.fillMaxWidth()
                .heightIn(min = minHeight),
            height = Modifier.height(minHeight),
            onChange = { cell.set(it) },
        )

        is DateCellEditor -> JoyDateTimeCell(
            cellId = cell.id,
            value = cell.date.watchAsState(),
            readonly = mode == Mode.readonly,
            onChange = { cell.set(it) },
            format = position?.columns?.find { it.id == cell.column.id }?.format,
            modifier = Modifier.fillMaxWidth(),
            onSignal = {

            },
        )

        is BlockCellEditor -> JoyTextEditCell(
            value = cell.blockText.watchAsState() ?: "",
            modifier = Modifier.fillMaxWidth()
                .heightIn(min = minHeight),
            height = Modifier.height(minHeight),
            readonly = true,
            borders = false,
            minLines = 3,
            onChange = { cell.set(it) },
        )

        is NumberCellEditor -> JoyNumberCell(
            value = cell.number.watchAsState(),
            readonly = mode == Mode.readonly,
            borders = false,
            minLines = 3,
            onChange = { cell.set(it) },
        )

        is DropdownCellEditor -> RawDropField(
            options = cell.options,
            value = cell.option.watchAsState()?.let { listOf(it) },
            readonly = false,
            borders = false,
            multiple = false,
            onAdd = { cell.select(it) },
            onRemove = { cell.unselect(it) },
        )

        is MultiselectCellEditor -> JoyMultiSelectCell(
            options = cell.options,
            value = cell.value.watchAsState(),
            multi = (cell.column as? MultiselectColumn)?.multi != false,
            onChange = {
                cell.unselectAll()
                cell.selectOptions(it)
            }
        )

        is ImageCellEditor -> RawImageField(
            id = "${cell.row.id}:${cell.column.id}",
            title = cell.column.title,
            readonly = mode == Mode.readonly,
            uploaded = cell.value,
            onUpload = onUpload,
            onDialog = {},
            onAdded = { cell.add(it) },
            onRemoved = { cell.remove(it) },
        ) {
            if (cell.value.isNotEmpty()) {
                ImageCountPreview(
                    count = cell.value.size,
                    modifier = Modifier
                        .fillMaxSize()
                        .padding(4.dp),
                    onClick = it::openModal
                )
            } else {
                UploadLabel(modifier = Modifier.fillMaxWidth(), onClick = it::openModal)
            }
        }

        is BarcodeCellEditor -> JoyBarcodeCell(
            value = cell.barcode.watchAsState(),
            readonly = mode == Mode.readonly,
            borders = false,
            onChange = { cell.set(it) },
            onCapture = onCapture
        )

        is SignatureCellEditor -> JoySignatureCell(
            id = "${cell.row.id}:${cell.column.id}",
            value = cell.signature.watchAsState() ?: "",
            readonly = mode == Mode.readonly,
            onChange = {
                cell.set(it)
            },
        )

        else -> Text("-", color = Color.Yellow)
    }
}