package joyfill.table

import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.text.BasicTextField
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.SolidColor
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.VisualTransformation
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp

@Composable
@OptIn(ExperimentalMaterial3Api::class)
fun JoyTextEditCell(
    value: String,
    onChange: (String) -> Unit,
    modifier: Modifier = Modifier,
    enabled: Boolean = true,
    readonly: Boolean = false,
    borders: Boolean = true,
    maxLines: Int = Int.MAX_VALUE,
    minLines: Int = 1,
    height: Modifier = Modifier
) {
    val interactionSource = remember { MutableInteractionSource() }
    val textStyle = MaterialTheme.typography.bodyLarge.copy(
        textAlign = TextAlign.Start,
        color = MaterialTheme.colorScheme.onSurface
    )
    Box(
        contentAlignment = Alignment.Center,
        modifier = Modifier.fillMaxWidth().then(height)
    ) {
        BasicTextField(
            value = value,
            onValueChange = onChange,
            modifier = modifier,
            textStyle = textStyle,
            interactionSource = interactionSource,
            enabled = enabled,
            readOnly = readonly,
            minLines = minLines,
            maxLines = maxLines,
            singleLine = minLines == 1 && maxLines == 1,
            keyboardOptions = KeyboardOptions.Default.copy(keyboardType = KeyboardType.Text),
            cursorBrush = SolidColor(MaterialTheme.colorScheme.primary),
        ) { innerTextField ->
            OutlinedTextFieldDefaults.DecorationBox(
                value = value,
                innerTextField = innerTextField,
                singleLine = minLines == 1 && maxLines == 1,
                enabled = enabled,
                interactionSource = interactionSource,
                contentPadding = PaddingValues(vertical = 1.dp, horizontal = 4.dp),
                visualTransformation = VisualTransformation.None,
                colors = if (borders) {
                    TextFieldDefaults.colors(
                        focusedContainerColor = Color.Transparent,
                        unfocusedContainerColor = Color.Transparent,
                    )
                } else {
                    OutlinedTextFieldDefaults.colors(
                        focusedBorderColor = Color.Transparent,
                        unfocusedBorderColor = Color.Transparent,
                        focusedContainerColor = Color.Transparent,
                        unfocusedContainerColor = Color.Transparent,
                    )
                }
            )
        }
    }
}