package joyfill.table

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Add
import androidx.compose.material3.Button
import androidx.compose.material3.Icon
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import cinematic.watchAsState
import joyfill.FieldPosition
import joyfill.Mode
import joyfill.editors.TableFieldEditor
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

@Composable
internal fun FullGroupView(
    editor: TableFieldEditor,
    position: FieldPosition?,
    mode: Mode,
    showUnsupportedColumns: Boolean,
    onClose: () -> Unit,
    uploadHandler: (suspend () -> List<String>)?,
    barcodeHandler: (suspend () -> String?)?
) = Column(modifier = Modifier.fillMaxWidth().padding(8.dp)) {
    val body = rememberLazyListState()
    val columns = remember(editor) { editor.columns }
    val columnsState = columns.state.watchAsState()
    val scope = rememberCoroutineScope()
    val state = editor.state.watchAsState()

    FullTableHeaderButtons(editor, onClose = onClose)
    FilterRow(
        state = columnsState,
        onTextChange = columns::filter,
        onOptionChange = columns::filter,
        onSort = columns::sort,
        onClearFilter = columns::clearFilter
    )

    FullTableBody(editor,  position, body, showUnsupportedColumns, mode, uploadHandler, barcodeHandler)
    if (!editor.field.disabled && mode != Mode.readonly) Row(modifier = Modifier.fillMaxWidth()) {
        Button(
            shape = RoundedCornerShape(8.dp),
            onClick = {
                editor.rows.append()
                scope.launch {
                    delay(100) // wait for the row to be added
                    body.scrollToItem(editor.rows.state.value.selected.size)
                }
            }) {
            Icon(Icons.Outlined.Add, "Add row")
            Text("Add Row")
        }
    }

    val form = state.form
    if (form is RowEditor) Dialog(
        onDismissRequest = { editor.hideRowForm() },
        properties = DialogProperties(usePlatformDefaultWidth = false)
    ) {
        Surface(modifier = Modifier.fillMaxSize(0.8f)) {
            RowForm(
                title = editor.field.title,
                editor = form,
                columns = editor.field.columns,
                position = position,
                mode = mode,
                onUpload = uploadHandler,
                onCapture = barcodeHandler,
                onSubmit = { editor.submit() },
                onClose = { editor.hideRowForm() },
            )
        }
    }
}