@file:OptIn(ExperimentalMaterial3Api::class, ExperimentalMaterial3Api::class)

package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.interaction.PressInteraction
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.CalendarMonth
import androidx.compose.material3.Button
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DisplayMode
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.SelectableDates
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TimePicker
import androidx.compose.material3.TimePickerState
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import cinematic.watchAsState
import joyfill.editors.DateFieldEditor
import joyfill.utils.Picking
import joyfill.utils.replaceHolders
import joyfill.utils.requiresDate
import joyfill.utils.requiresTime
import kotlinx.coroutines.launch
import kotlinx.datetime.TimeZone
import kotlinx.datetime.offsetAt
import krono.Instant
import krono.PureDateTimeFormatter

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun JoyDateTimeField(
    editor: DateFieldEditor,
    format: String?,
    mode: Mode,
    onSignal: (Signal<Long?>) -> Unit,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    val field = remember(editor) { editor.field }
    var value by remember(field) { mutableStateOf(field.value?.let { Instant(it) }) }
    var dialog by remember { mutableStateOf(false) }
    val readonly = remember(field, mode) { field.disabled || mode == Mode.readonly }

    var offsetMills by remember(field) { mutableStateOf(field.value?.let {
        val instant = kotlinx.datetime.Instant.fromEpochSeconds(it)
        TimeZone.currentSystemDefault().offsetAt(instant).totalSeconds * 1000
    } ?: 0) }

    val pattern = remember {
        format?.replaceHolders() ?: "{YYYY}-{MM}-{DD} {hh}:{mm}"
    }

    val initialPicking = remember(pattern) { if (pattern.requiresDate()) Picking.Date else Picking.Time }

    var picking by remember(initialPicking) { mutableStateOf(initialPicking) }

    val date = rememberDatePickerState(
        initialSelectedDateMillis = field.value?.plus(offsetMills),
        selectableDates = object : SelectableDates {
            override fun isSelectableDate(utcTimeMillis: Long): Boolean = mode == Mode.fill
            override fun isSelectableYear(year: Int): Boolean = mode == Mode.fill
        },
        initialDisplayMode = if (mode == Mode.readonly) DisplayMode.Input else DisplayMode.Picker
    )

    val time = remember(value, date.selectedDateMillis) {
        val t = value?.atSystemZone()?.time
        TimePickerState(
            initialHour = t?.hour ?: 0,
            initialMinute = t?.minute ?: 0,
            is24Hour = true
        )
    }

    val interaction = remember { MutableInteractionSource() }

    LaunchedEffect(interaction, readonly) {
        if (!readonly) launch {
            interaction.interactions.collect {
                if (it is PressInteraction.Release) {
                    picking = initialPicking
                    dialog = true
                    onSignal(Signal.Focus)
                }
            }
        }
    }

    Column(modifier = Modifier.testTag(field.id).fillMaxWidth()) {
        JoyFieldHead(field)
        Box(modifier = Modifier.fillMaxWidth()) {
            OutlinedTextField(
                value = pattern.format(date.selectedDateMillis, time.hour, time.minute),
                onValueChange = {},
                interactionSource = interaction,
                modifier = Modifier.testTag("${field.id}-body-output").fillMaxWidth(),
                readOnly = true
            )
            val modifier = Modifier.align(Alignment.CenterEnd).padding(end = 8.dp)
            Icon(Icons.Outlined.CalendarMonth, "calendar", modifier = modifier.clickable {
                picking = initialPicking
                dialog = true
                onSignal(Signal.Focus)
            })
        }
    }

    if (dialog) Dialog(
        onDismissRequest = {
            dialog = false
            date.selectedDateMillis = field.value
            value = field.value?.let { Instant(it) }
            onSignal(Signal.Blur(field.value))
        },
        properties = DialogProperties(usePlatformDefaultWidth = false)
    ) {
        Surface(modifier = Modifier.fillMaxWidth(0.95f)) {
            Column(modifier = Modifier.padding(8.dp).fillMaxWidth()) {
                when (picking) {
                    Picking.Date -> DatePicker(
                        state = date,
                        title = null,
                        headline = null,
                        modifier = Modifier.testTag("${field.id}-input-date").padding(14.dp),
                    )

                    Picking.Time -> TimePicker(
                        state = time,
                        modifier = Modifier.testTag("${field.id}-input-time")
                    )
                }

                Row(horizontalArrangement = Arrangement.End, modifier = Modifier.fillMaxWidth().padding(top = 12.dp)) {
                    OutlinedButton(
                        modifier = Modifier.testTag("${field.id}-input-cancel"),
                        onClick = {
                            dialog = false
                            date.selectedDateMillis = field.value?.plus(offsetMills)
                            value = field.value?.let { Instant(it) }
                            onSignal(Signal.Blur(field.value))
                        },
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Cancel")
                    }
                    Spacer(modifier = Modifier.width(8.dp))
                    Button(
                        modifier = Modifier.testTag("${field.id}-input-submit"),
                        onClick = {
                            when (picking) {
                                Picking.Date -> {
                                    if (pattern.requiresTime()) {
                                        picking = Picking.Time
                                    } else {
                                        val millis = date.selectedDateMillis
                                        onSignal(Signal.Change(millis))
                                        dialog = false
                                        onSignal(Signal.Blur(millis))
                                        editor.value = millis
                                    }
                                }

                                Picking.Time -> {
                                    val minutes = time.minute + (time.hour * 60)
                                    val millis = (date.selectedDateMillis ?: 0L) + (minutes * 60 * 1000L)
                                    onSignal(Signal.Change(millis))
                                    dialog = false
                                    onSignal(Signal.Blur(millis))
                                    editor.value = millis
                                }
                            }
                        },
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Submit")
                    }
                }
            }
        }
    }
}

private fun String.format(date: Long?, hr: Int?, min: Int?): String {
    val formatter = PureDateTimeFormatter(this)
    if (date != null) {
        val zdt = Instant(date).atZone("UTC")
        return formatter.formatDateTime(zdt.year, zdt.monthNumber, zdt.dayOfMonth, hr ?: 0, min ?: 0, 0)
    }
    if (requiresDate()) {
        return ""
    }

    if (requiresTime() && hr == null) {
        return ""
    }

    return formatter.formatDateTime(0, 1, 1, hr ?: 0, min ?: 0, 0)
}