package joyfill

import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.interaction.PressInteraction
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.CalendarMonth
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.common.DateTimeDialog
import joyfill.table.DateCellEditor
import joyfill.utils.Picking
import joyfill.utils.format
import joyfill.utils.replaceHolders
import joyfill.utils.requiresDate
import kotlinx.coroutines.launch

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun DateInputField(
    cell: DateCellEditor,
    format: String? = null,
    modifier: Modifier = Modifier,
    mode: Mode = Mode.readonly,
    onChange: (Long?) -> Unit,
    onSignal: (Signal<Long?>) -> Unit,
) {
    val v = cell.date.watchAsState()
    var dialog by remember { mutableStateOf(false) }

    val pattern = remember { format?.replaceHolders() ?: "{YYYY}-{MM}-{DD} {hh}:{mm}" }

    val initialPicking = remember(pattern) { if (pattern.requiresDate()) Picking.Date else Picking.Time }
    var picking by remember(initialPicking) { mutableStateOf(initialPicking) }

    var date by remember { mutableStateOf(cell.date.value) }
    var hour by remember { mutableStateOf(0) }
    var minute by remember { mutableStateOf(0) }

    val interaction = remember { MutableInteractionSource() }

    LaunchedEffect(interaction, mode) {
        if (mode != Mode.readonly) launch {
            interaction.interactions.collect {
                if (it is PressInteraction.Release) {
                    picking = initialPicking
                    dialog = true
                    onSignal(Signal.Focus)
                }
            }
        }
    }

    Box(modifier = modifier.padding(end = 1.dp)) {
        TextField(
            value = pattern.format(date, hour, minute),
            onValueChange = {},
            interactionSource = interaction,
            modifier = Modifier.testTag("${cell.id}-body-output").fillMaxWidth(),
            colors = TextFieldDefaults.colors(
                focusedIndicatorColor = Color.Transparent,
                unfocusedIndicatorColor = Color.Transparent,
                disabledIndicatorColor = Color.Transparent
            ),
            readOnly = true,
        )

        if(v== null || v <= 0){
            val modifier = Modifier.align(Alignment.CenterEnd).padding(end = 8.dp)

            Icon(Icons.Outlined.CalendarMonth, "calendar", modifier = modifier.clickable {
                picking = initialPicking
                dialog = true
                onSignal(Signal.Focus)
            })
        }
    }

    if (dialog) {
        DateTimeDialog(
            pickingWhat = picking,
            onSuccess = { dt, hr, min ->
                onSignal(Signal.Change(dt))
                onSignal(Signal.Blur(dt))
                onChange(dt/1000)
                dialog = false

                date = dt
                hour = hr
                minute = min
            },
            onDismissRequest = { dialog = false },
            pattern = pattern,
            initialValue = v,
            mode = mode,
            tagId = cell.id,
            is24Hour = !pattern.contains("a", ignoreCase = true)
        )
    }
}