package joyfill.table

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material3.Button
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.FieldPosition
import joyfill.JoyTitle
import joyfill.Mode
import joyfill.RawDropField
import joyfill.fields.Field
import joyfill.fields.table.BarcodeColumn
import joyfill.fields.table.Column
import joyfill.fields.table.DateColumn
import joyfill.fields.table.DropdownColumn
import joyfill.fields.table.ImageColumn
import joyfill.fields.table.MultiselectColumn
import joyfill.fields.table.NumberColumn
import joyfill.fields.table.SignatureColumn
import joyfill.fields.table.TextColumn
import joyfill.image.FirstImagePreview
import joyfill.image.RawImageField

@Composable
internal fun RowForm(
    title: String,
    editor: RowEditor,
    columns: List<Column>,
    position: FieldPosition?,
    mode: Mode,
    onUpload: (suspend () -> List<String>)?,
    onCapture: (suspend () -> String?)?,
    onSubmit: () -> Unit,
    onClose: () -> Unit,
) = Column(modifier = Modifier.fillMaxWidth().padding(8.dp).verticalScroll(rememberScrollState())) {
    Row(
        modifier = Modifier.fillMaxWidth().padding(bottom = 8.dp), horizontalArrangement = Arrangement.SpaceBetween
    ) {
        JoyTitle("$title entry", modifier = Modifier.padding(bottom = 4.dp))
        Icon(Icons.Filled.Close, "close", modifier = Modifier.testTag("${title}-capture-close").clickable { onClose() })
    }

    val ignoredColumTypes = listOf(Field.Type.block, Field.Type.progress)

    columns.forEach { column ->
        val hidden = position?.columns?.find { it.id == column.id }?.hidden == true
        if(hidden) return@forEach

        JoyTitle(column.title, modifier = Modifier.padding(top = 8.dp))
        when (column) {
            is TextColumn -> {
                val e = remember(editor, column.id) { editor.text(column.id) }
                JoyTextEditCell(
                    value = e?.text?.watchAsState() ?: "",
                    readonly = mode == Mode.readonly,
                    borders = true,
                    minLines = 3,
                    onChange = { e?.set(it) },
                    modifier = Modifier.fillMaxWidth()
                )
            }

            is DateColumn -> {
                val cell = remember(editor, column.id) { editor.date(column.id) }
                JoyDateTimeCell(
                    cellId = cell?.id,
                    value = cell?.date?.watchAsState(),
                    readonly = mode == Mode.readonly,
                    borders = true,
                    format = position?.columns?.find { it.id == cell?.column?.id }?.format,
                    onChange = { cell?.set(it) },
                    onSignal = {},
                    modifier = Modifier.fillMaxWidth()
                )
            }

            is NumberColumn -> {
                val e = remember(editor, column.id) { editor.number(column.id) }
                JoyNumberCell(
                    value = e?.number?.watchAsState(),
                    readonly = mode == Mode.readonly,
                    borders = true,
                    minLines = 3,
                    onChange = { e?.set(it) },
                    modifier = Modifier.fillMaxWidth()
                )
            }

            is DropdownColumn -> {
                val e = remember(editor, column.id) { editor.dropdown(column.id) }
                RawDropField(
                    options = e?.options ?: emptyList(),
                    value = listOf(e?.option?.watchAsState()).mapNotNull { it },
                    readonly = false,
                    borders = true,
                    multiple = false,
                    onAdd = { e?.select(it) },
                    onRemove = { e?.unselect(it) }
                )
            }

            is MultiselectColumn -> {
                val cell = remember(editor, column.id) { editor.multiSelect(column.id) }
                JoyMultiSelectCell(
                    options = cell?.options ?: emptyList(),
                    value = cell?.value?.watchAsState() ?: emptyList(),
                    borders = true,
                    onChange = {
                        cell?.unselectAll()
                        cell?.selectOptions(it)
                    }
                )
            }

            is ImageColumn -> {
                val e = remember(editor, column.id) { editor.image(column.id) }
                RawImageField(
                    id = column.id,
                    title = column.title,
                    readonly = mode == Mode.readonly,
                    onDialog = {},
                    uploaded = emptyList(),
                    onUpload = onUpload,
                    onAdded = { e?.add(it) },
                    onRemoved = { e?.remove(it) }
                ) {
                    FirstImagePreview(id = "new-row", params = it, onFocus = {}, onRemove = {})
                }
            }

            is BarcodeColumn -> {
                val cell = remember(editor, column.id) { editor.barcode(column.id) }
                JoyBarcodeCell(
                    value = cell?.barcode?.watchAsState(),
                    readonly = mode == Mode.readonly,
                    borders = true,
                    onChange = { cell?.set(it) },
                    onCapture = onCapture,
                    modifier = Modifier.fillMaxWidth()
                )
            }

            is SignatureColumn -> {
                val cell = remember(editor, column.id) { editor.signature(column.id) }
                JoySignatureCell(
                    id = "${cell?.row?.id}:${cell?.column?.id}",
                    value = cell?.signature?.watchAsState() ?: "",
                    readonly = mode == Mode.readonly,
                    onChange = { cell?.set(it) },
                    borders = true
                )
            }

            else -> {
                if (column.type !in ignoredColumTypes) {
                    Text("Unsupported table column type '${column.type}'")
                }
            }
        }
    }

    Row(horizontalArrangement = Arrangement.End, modifier = Modifier.fillMaxWidth().padding(top = 12.dp)) {
        OutlinedButton(
            modifier = Modifier.testTag("${title}-capture-cancel"),
            onClick = onClose,
            shape = RoundedCornerShape(8.dp),
        ) {
            Text("Cancel")
        }
        Spacer(modifier = Modifier.width(8.dp))
        Button(
            modifier = Modifier.testTag("${title}-capture-submit"),
            onClick = onSubmit,
            shape = RoundedCornerShape(8.dp),
        ) {
            Text("Submit")
        }
    }
}