package joyfill.table

import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material3.AssistChip
import androidx.compose.material3.AssistChipDefaults
import androidx.compose.material3.Button
import androidx.compose.material3.Card
import androidx.compose.material3.Checkbox
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import joyfill.common.toColor
import joyfill.utils.Option2

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun JoyMultiSelectCell(
    options: List<Option2>,
    value: List<Option2>,
    modifier: Modifier = Modifier,
    borders: Boolean = false,
    multi: Boolean = true,
    onChange: ((List<Option2>) -> Unit)? = null,
) {
    var openWidget = remember { mutableStateOf(false) }

    val selected = remember(value, options) {
        val v = value.mapNotNull { vl -> options.find { vl.id == it.id } }
        mutableStateListOf(*v.toTypedArray())
    }

    val colors = if (borders) {
        OutlinedTextFieldDefaults.colors()
    } else {
        OutlinedTextFieldDefaults.colors(
            focusedBorderColor = Color.Transparent,
            unfocusedBorderColor = Color.Transparent,
        )
    }
    Box(
        modifier = Modifier.fillMaxWidth()
            .height(OutlinedTextFieldDefaults.MinHeight)
            .border(
                width = 0.8.dp, color = if(borders) colors.unfocusedIndicatorColor else Color.Transparent,
                shape = OutlinedTextFieldDefaults.shape
            ),
    ){
        val optCount = value.size
        val firstItem = value.firstOrNull()

        if (firstItem != null) {
            val text = firstItem.value + if (optCount > 1) " +${optCount - 1}" else ""
            AssistChip(
                label = { Text(text) },
                onClick = { openWidget.value = !openWidget.value },
                modifier = Modifier.padding(8.dp).clickable { openWidget.value = !openWidget.value },
                colors = AssistChipDefaults.assistChipColors().copy(
                    containerColor = firstItem.color?.toColor() ?: Color.Transparent
                )
            )
        } else {
            val modifier = Modifier.fillMaxSize().clickable { openWidget.value = !openWidget.value }
            Text("", modifier = modifier)
        }
    }

    if (openWidget.value) {
        Dialog(
            onDismissRequest = { openWidget.value = false },
            properties = DialogProperties(usePlatformDefaultWidth = false),
        ) {
            Surface(
                modifier = modifier
                    .padding(vertical = 16.dp, horizontal = 8.dp)
                    .fillMaxWidth()
                    .heightIn(min = 200.dp)
            ) {
                Column(modifier = Modifier.fillMaxSize()) {
                    Row(
                        verticalAlignment = Alignment.CenterVertically,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Text(if(multi) "Multiselect" else "Single Select", modifier = Modifier.weight(1f))
                        Button(onClick = {
                            onChange?.invoke(selected)
                            openWidget.value = false
                        }) {
                            Text("Apply")
                        }
                        Button(
                            modifier = Modifier.padding(8.dp),
                            onClick = {
                                selected.clear()
                                selected.addAll(value)
                                onChange?.invoke(value)
                                openWidget.value = false
                            }
                        ) {
                            Icon(Icons.Default.Close, "Close")
                        }
                    }

                    Card(modifier = Modifier.fillMaxWidth()){
                        LazyColumn(modifier = Modifier.fillMaxSize()){
                            items(options.size){ index ->
                                val option = options[index]
                                Row(
                                    verticalAlignment = Alignment.CenterVertically,
                                    modifier = Modifier.fillMaxWidth()
                                        .background(option.color?.toColor() ?: Color.Transparent)
                                ) {
                                    Checkbox(
                                        checked = option.id in selected.map { it.id },
                                        onCheckedChange = {
                                            if (it) {
                                                if(!multi){
                                                    selected.clear()
                                                }
                                                selected.add(option)
                                            } else {
                                                selected.remove(option)
                                            }
                                        }
                                    )
                                    Text(option.value, modifier = Modifier.weight(1f))
                                }

                                HorizontalDivider()
                            }
                        }
                    }
                }
            }
        }
    }
}