package joyfill.image

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.staggeredgrid.LazyVerticalStaggeredGrid
import androidx.compose.foundation.lazy.staggeredgrid.StaggeredGridCells
import androidx.compose.foundation.lazy.staggeredgrid.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.outlined.Close
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.SolidColor
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import joyfill.utils.Attachment

@Composable
internal fun RawImageField(
    id: String,
    title: String,
    uploaded: List<Attachment>,
    readonly: Boolean,
    onUpload: (suspend () -> List<String>)? = null,
    onDialog: (opened: Boolean) -> Unit,
    onAdded: (List<String>) -> Unit,
    onRemoved: (List<String>) -> Unit,
    preview: @Composable (params: PreviewParams) -> Unit,
) {
    val value = remember { mutableStateListOf(*uploaded.map { it.url }.toTypedArray()) }
    val selected = remember { mutableStateListOf<String>() }
    var isDialogOpened by remember { mutableStateOf(false) }
    val scope = rememberCoroutineScope()

    val params by remember(value, selected, isDialogOpened) {
        derivedStateOf {
            PreviewParams(
                value = value,
                selected = selected,
                setModalState = {
                    isDialogOpened = it
                    onDialog(it)
                },
                readonly = readonly,
                onUpload = onUpload,
                scope = scope,
                onAdded = onAdded
            )
        }
    }

    preview(params)

    if (isDialogOpened) Dialog(
        onDismissRequest = params::closeModal,
        properties = DialogProperties(
            usePlatformDefaultWidth = false
        )
    ) {
        Surface(modifier = Modifier.fillMaxSize(0.95f)) {
            Box(modifier = Modifier.fillMaxSize().padding(8.dp)) {
                Column(modifier = Modifier.padding(start = 4.dp)) {
                    Text(title, modifier = Modifier.padding(8.dp))
                    if (!readonly) Row(modifier = Modifier.fillMaxWidth().padding(bottom = 12.dp)) {
                        UploadLabel(
                            onClick = params::onUploadHandler,
                            modifier = Modifier.testTag("$id-body-upload").padding(8.dp)
                        )
                        Spacer(modifier = Modifier.width(8.dp))
                        val color = Color.Red.copy(alpha = 0.6f)
                        OutlinedButton(
                            onClick = {
                                value.removeAll(selected)
                                onRemoved(selected)
                                selected.clear()
                                isDialogOpened = value.size > 1
                                if (!isDialogOpened) onDialog(false)
                            },
                            enabled = selected.isNotEmpty(),
                            border = ButtonDefaults.outlinedButtonBorder.copy(
                                brush = SolidColor(color)
                            ),
                            colors = ButtonDefaults.outlinedButtonColors(contentColor = color),
                            shape = RoundedCornerShape(8.dp),
                            contentPadding = PaddingValues(horizontal = 8.dp, vertical = 0.dp),
                        ) {
                            Text("Delete")
                            Spacer(modifier = Modifier.width(8.dp))
                            Icon(Icons.Filled.Delete, "Delete Selected")
                        }
                    }
                }
                LazyVerticalStaggeredGrid(
                    columns = StaggeredGridCells.Adaptive(150.dp),
                    modifier = Modifier.padding(top = if (readonly) 0.dp else 100.dp)
                ) {
//                    items(value, key = { it }) {
                    items(value) {
                        val marked = selected.contains(it)
                        Picture(
                            url = it,
                            selected = marked,
                            margin = 4.dp,
                            onSelectorClicked = { url: String ->
                                if (marked) selected.remove(url) else selected.add(url)
                                Unit
                            }.takeIf { !readonly },
                            modifier = Modifier.testTag("$id-body-image-$it")
                        )
                    }
                }
                val modifier = Modifier.align(Alignment.TopEnd).clickable { params.closeModal() }
                Icon(Icons.Outlined.Close, "Close Model", modifier)
            }
        }
    }
}