package joyfill.common

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Button
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DisplayMode
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.SelectableDates
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TimePicker
import androidx.compose.material3.TimePickerState
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import joyfill.Mode
import joyfill.utils.Picking
import joyfill.utils.requiresTime
import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun DateTimeDialog(
    pickingWhat: Picking,
    onSuccess: (Long, Int, Int) -> Unit,
    onDismissRequest: () -> Unit,
    pattern: String,
    initialValue: Long?,
    mode: Mode,
    tagId: String,
    modifier: Modifier = Modifier.padding(8.dp).fillMaxWidth(),
    is24Hour: Boolean = false
) {
    val now = remember {  if(initialValue == null) Clock.System.now() else Instant.fromEpochSeconds(initialValue) }
    val localDate = remember(now){ now.toLocalDateTime(TimeZone.currentSystemDefault()) }

    var picking by remember(pickingWhat) { mutableStateOf(pickingWhat) }

    val date = rememberDatePickerState(
        initialSelectedDateMillis = now.toEpochMilliseconds(),
        selectableDates = object : SelectableDates {
            override fun isSelectableDate(utcTimeMillis: Long): Boolean = mode == Mode.fill
            override fun isSelectableYear(year: Int): Boolean = mode == Mode.fill
        },
        initialDisplayMode = if (mode == Mode.readonly) DisplayMode.Input else DisplayMode.Picker
    )

    val time = remember(date.selectedDateMillis) {
        TimePickerState(
            initialHour = localDate.hour,
            initialMinute = localDate.minute,
            is24Hour = is24Hour
        )
    }

    Dialog(
        onDismissRequest = { onDismissRequest() },
        properties = DialogProperties(usePlatformDefaultWidth = false)
    ) {
        Surface(modifier = modifier) {
            Column(modifier = Modifier.padding(8.dp).fillMaxWidth()) {
                when (picking) {
                    Picking.Date -> DatePicker(
                        state = date,
                        title = null,
                        headline = null,
                        modifier = Modifier.testTag("${tagId}-input-date").padding(14.dp),
                    )

                    Picking.Time -> TimePicker(
                        state = time,
                        modifier = Modifier.testTag("${tagId}-input-time")
                    )
                }

                Row(
                    horizontalArrangement = Arrangement.End,
                    modifier = Modifier.fillMaxWidth().padding(top = 12.dp)
                ) {
                    OutlinedButton(
                        modifier = Modifier.testTag("${tagId}-input-cancel"),
                        onClick = {
                           onDismissRequest()
                        },
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Cancel")
                    }
                    Spacer(modifier = Modifier.width(8.dp))
                    Button(
                        modifier = Modifier.testTag("${tagId}-input-submit"),
                        onClick = {
                            when (picking) {
                                Picking.Date -> {
                                    if (pattern.requiresTime()) {
                                        picking = Picking.Time
                                    } else {
                                        val millis = date.selectedDateMillis
                                        onSuccess(millis?:0, 0, 0)
                                    }
                                }

                                Picking.Time -> {
                                    val minutes = time.minute + (time.hour * 60)
                                    val millis = (date.selectedDateMillis ?: 0L) + (minutes * 60 * 1000L)
                                    onSuccess(millis, time.hour, time.minute)
                                }
                            }
                        },
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Submit")
                    }
                }
            }
        }
    }
}