package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Checkbox
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.RadioButton
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.editors.MultiSelectFieldEditor

@Composable
internal fun JoySelectField(
    editor: MultiSelectFieldEditor,
    mode: Mode,
    onSignal: (Signal<List<String>>) -> Unit
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    val field = remember(editor) { editor.field }
    val options = remember(editor) { editor.options.filter { !it.deleted } }

    val values = remember(editor) {
        mutableStateListOf(*editor.selected().map { it.id }.toTypedArray())
    }

    val focus = remember(onSignal) { FocusManager(onSignal) { } }

    Column(Modifier.fillMaxWidth()) {
//        JoyTitle(field, modifier = Modifier.testTag("${field.id}-title"))
        JoyFieldHead(field)
        Surface(
            color = Color.Transparent,
            shape = RoundedCornerShape(8.dp),
            border = BorderStroke(2.dp, LocalContentColor.current.copy(0.1f)),
            modifier = Modifier.fillMaxWidth().testTag("${field.id}-body").onFocusChanged(focus.handler)
        ) {
            Column(modifier = Modifier.padding(vertical = 7.dp)) {
                val size = options.size
                val multiple = field.multi
                options.forEachIndexed { index, option ->
                    JoyOption(
                        label = option.value,
                        selected = option.id in values,
                        square = multiple,
                        modifier = Modifier.semantics { contentDescription = option.value },
                        onClick = {
                            if (field.disabled || mode == Mode.readonly) return@JoyOption
                            if (option.id in values) {
                                editor.unselect(option)
                                values.remove(option.id)
                            } else {
                                editor.select(option)
                                if (!multiple) values.clear()
                                values.add(option.id)
                            }
                            onSignal(Signal.Change(values.toList()))
                        }
                    )
                    if (index != size - 1) Spacer(
                        modifier = Modifier
                            .fillMaxWidth()
                            .height(1.dp)
                            .background(LocalContentColor.current.copy(0.1f))
                    )
                }
            }
        }
    }
}

@Composable
private fun JoyOption(
    label: String,
    selected: Boolean,
    square: Boolean,
    onClick: () -> Unit,
    modifier: Modifier = Modifier,
) = Row(
    verticalAlignment = Alignment.CenterVertically,
    modifier = modifier
        .fillMaxWidth()
        .clickable { onClick() }
        .padding(horizontal = 14.dp, vertical = 7.dp)
) {
    val prefix = remember(selected) { (if (selected) "" else "un") + "selected" }
    if (square) {
        Checkbox(selected, null, modifier = Modifier.semantics { contentDescription = "$prefix $label checkbox" })
    } else {
        RadioButton(selected, null, modifier = Modifier.semantics { contentDescription = "$prefix $label radio" })
    }
    Spacer(modifier = Modifier.width(7.dp))
    Text(label, modifier = Modifier.semantics { contentDescription = "$prefix $label value" })
}