@file:OptIn(ExperimentalMaterial3Api::class)

package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Check
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import cinematic.watchAsState
import joyfill.editors.DropdownFieldEditor
import joyfill.utils.Option2

@Composable
internal fun JoyDropField(
    editor: DropdownFieldEditor,
    mode: Mode,
    multiple: Boolean,
    onSignal: (Signal<Option2?>) -> Unit,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    Column(Modifier.fillMaxWidth()) {

        val field = remember(editor) { editor.field }

        val selected by remember(editor) { mutableStateOf(editor.selected()) }

        val focus = remember(onSignal) { FocusManager(onSignal) { } }

        val options = remember(editor) { editor.options.filter { !it.deleted } }

        JoyTitle(field, modifier = Modifier.testTag("${field.id}-title"))
        RawDropField(
            options = options,
            value = selected?.let { listOf(it) } ?: emptyList(),
            readonly = field.disabled || mode == Mode.readonly,
            multiple = multiple,
            borders = true,
            onAdd = {
                editor.select(it)
                onSignal(Signal.Change(it))
            },
            onRemove = {
                editor.unselect(it)
                onSignal(Signal.Change(null))
            },
            modifier = Modifier.testTag("${field.id}-body").onFocusChanged(focus.handler)
        )
    }
}

@Composable
internal fun RawDropField(
    options: List<Option2>,
    value: List<Option2>?,
    readonly: Boolean,
    multiple: Boolean,
    borders: Boolean,
    modifier: Modifier = Modifier,
    onClick: ((Option2) -> Unit)? = null,
    onAdd: ((Option2) -> Unit)? = null,
    onRemove: ((Option2) -> Unit)? = null
) {
    var expanded by remember { mutableStateOf(false) }
    val values = remember(value, options) {
        val v = value?.mapNotNull { vl ->
            options.find { vl.id == it.id }
        } ?: emptyList()
        mutableStateListOf(*v.toTypedArray())
    }

    ExposedDropdownMenuBox(
        expanded = !readonly && expanded,
        onExpandedChange = { expanded = !readonly && it },
        modifier = modifier
    ) {
        OutlinedTextField(
            value = values.joinToString(", ") { it.value },
            maxLines = 4,
            readOnly = true,
            onValueChange = {},
            trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded) },
            colors = if (borders) {
                OutlinedTextFieldDefaults.colors()
            } else {
                OutlinedTextFieldDefaults.colors(
                    focusedBorderColor = Color.Transparent,
                    unfocusedBorderColor = Color.Transparent,
                )
            },
            modifier = Modifier.fillMaxWidth().menuAnchor().clickable { expanded = !readonly }
        )

        ExposedDropdownMenu(
            expanded = expanded,
            onDismissRequest = { expanded = false },
            modifier = Modifier.fillMaxWidth()
        ) {
            for (option in options) DropdownMenuItem(
                modifier = Modifier.semantics { contentDescription = option.value },
                text = { Text(option.value) },
                leadingIcon = {
                    if (option in values && multiple) {
                        Icon(Icons.Default.Check, "selected")
                    }
                },
                onClick = {
                    onClick?.invoke(option)
                    if (option in values) {
                        onRemove?.invoke(option)
                        values.remove(option)
                    } else {
                        if (!multiple) values.clear()
                        onAdd?.invoke(option)
                        values.add(option)
                    }
                    if (!multiple) {
                        expanded = false
                    }
                },
            )
        }
    }
}