package joyfill.table

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Check
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.MenuAnchorType
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.unit.dp
import joyfill.utils.Option2

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun SearchDropDown(
    options: List<Option2>,
    value: List<Option2>?,
    readonly: Boolean,
    multiple: Boolean,
    modifier: Modifier = Modifier,
    onClick: ((Option2) -> Unit)? = null,
    onAdd: ((Option2) -> Unit)? = null,
    onRemove: ((Option2) -> Unit)? = null
) {
    var expanded by remember { mutableStateOf(false) }
    val values = remember(value, options) {
        val v = value?.mapNotNull { vl ->
            options.find { vl.id == it.id }
        } ?: emptyList()
        mutableStateListOf(*v.toTypedArray())
    }

    ExposedDropdownMenuBox(
        expanded = !readonly && expanded,
        onExpandedChange = { expanded = !readonly && it },
        modifier = modifier
    ) {
        Row(
            modifier = Modifier.fillMaxSize()
                .padding(horizontal = 4.dp)
                .menuAnchor(type = MenuAnchorType.PrimaryEditable).clickable(enabled = false) { expanded = !readonly },
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = if (values.isNotEmpty()) Arrangement.SpaceBetween else Arrangement.End
        ) {
            Text(values.joinToString(", ") { it.value })
            ExposedDropdownMenuDefaults.TrailingIcon(expanded)
        }

        ExposedDropdownMenu(
            expanded = expanded,
            onDismissRequest = { expanded = false },
            modifier = Modifier.fillMaxWidth()
        ) {
            for (option in options) DropdownMenuItem(
                modifier = Modifier.semantics { contentDescription = option.value },
                text = { Text(option.value) },
                leadingIcon = {
                    if (option in values && multiple) {
                        Icon(Icons.Default.Check, "selected")
                    }
                },
                onClick = {
                    onClick?.invoke(option)
                    if (option in values) {
                        onRemove?.invoke(option)
                        values.remove(option)
                    } else {
                        if (!multiple) values.clear()
                        onAdd?.invoke(option)
                        values.add(option)
                    }
                    if (!multiple) {
                        expanded = false
                    }
                },
            )
        }
    }
}