package joyfill.table

import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.ArrowRight
import androidx.compose.material3.AssistChip
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.FieldPosition
import joyfill.JoyTitle
import joyfill.RawNumberField
import joyfill.Valid
import joyfill.editors.TableFieldEditor
import joyfill.fields.Field

@Composable
internal fun Preview(
    editor: TableFieldEditor,
    position: FieldPosition?,
    rows: Int,
    totalRows: Int,
    onClick: () -> Unit,
) = Column(modifier = Modifier.fillMaxWidth()) {
    val field = remember(editor) { editor.field }
    while (editor.rows.size < rows) editor.rows.append()
    Box(modifier = Modifier.fillMaxWidth()) {
        JoyTitle(field, modifier = Modifier.align(Alignment.CenterStart))
        Row(modifier = Modifier.align(Alignment.CenterEnd)) {
            TextButton(onClick = onClick, contentPadding = PaddingValues(0.dp), shape = RoundedCornerShape(2.dp)) {
                Text("View")
                Icon(Icons.Outlined.ArrowRight, "View table")
                if (totalRows != 0) Text("+$totalRows", modifier = Modifier.padding(start = 4.dp))
            }
        }
    }
    Column(
        modifier = Modifier.border(
            1.dp,
            color = LocalContentColor.current.copy(alpha = 0.6f),
            shape = RoundedCornerShape(8.dp)
        ).clip(RoundedCornerShape(8.dp))
    ) {
        RowArrayTable(
            cols = field.columns.size,
            rows = rows,
            modifier = Modifier.testTag("${field.id}-small").fillMaxWidth().clickable { onClick() },
        ) { row, col ->
            val width = if (col < 0) 30.dp else colWidth(field.columns[col].type, preview = true)
            val modifier = Modifier.width(width).padding(horizontal = 8.dp)
            val rowEditor =  editor.rows.get(row)
            val cell = editor.rows.get(row)?.col(col)

            val hidden = position?.columns?.find { it.id == cell?.column?.id }?.hidden == true
            if (hidden) return@RowArrayTable

            when {
                col == -1 && row == -1 -> Box(modifier = modifier)
                col == -1 && row != -1 -> Text(text = "${row + 1}", modifier = modifier)
                col != -1 && row == -1 -> {
                    val hidden = position?.columns?.find { it.id ==  field.columns[col].id }?.hidden == true
                    if(!hidden){
                        Text(text = field.columns[col].title, modifier = modifier)
                    }
                }
                else -> {
                    when (cell) {
                        is TextCellEditor -> Text(text = cell.text.watchAsState() ?: "", modifier = modifier)
                        is BlockCellEditor -> Text(text = cell.blockText.watchAsState() ?: "", modifier = modifier)
                        is DropdownCellEditor -> Text(text = cell.option.watchAsState()?.value ?: "", modifier = modifier)
                        is MultiselectCellEditor -> {
                            val optCount = cell.value.watchAsState().size
                            if(optCount == 0) {
                                Text("", modifier = modifier)
                            } else {
                                val firstItem = cell.value.watchAsState().firstOrNull()
                                val text = firstItem?.value + if (optCount > 1) " +${optCount - 1}" else ""
                                AssistChip(
                                    label = { Text(text) },
                                    onClick = { },
                                    modifier = Modifier.clickable { }
                                )
                            }
                        }
                        is NumberCellEditor -> RawNumberField(value = cell.number.watchAsState(), readonly = true, borders = false, modifier = modifier)
                        is DateCellEditor -> {
                            val format = position?.columns?.find { it.id == cell.column.id }?.format
                            JoyDateTimeCell(cellId = cell.id, value = cell.date.watchAsState(),  readonly = true, format = format, modifier = modifier)
                        }
                        is ProgressCellEditor -> {
                            val  visibleCols =  field.columns.filter { col -> position?.columns?.find { it.id == col.id }?.hidden != true }
                            val requiredColumns = visibleCols.filter { it.required && it.type != Field.Type.progress }
                            val validCells = requiredColumns.map { rowEditor?.col(it.id).also { it?.validate() } }.filter { it?.validity?.value is Valid }.size
                            val totalCells = requiredColumns.size
                            JoyProgressCell(validCells = validCells, totalCells = totalCells, modifier = modifier)
                        }
                        is ImageCellEditor -> Text(text = "${cell.value.size}", modifier = modifier)
                        is BarcodeCellEditor -> Text(text = cell.barcode.value ?: "", modifier = modifier)
                        else -> {
                            val text = buildAnnotatedString { withStyle(SpanStyle(color = Color.Yellow)) { append("!") } }
                            Text(text = text, modifier = Modifier.weight(1f))
                        }
                    }
                }
            }
        }
    }
}