package joyfill.table

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.JoyTitle
import joyfill.RawDropField
import joyfill.editors.TableFieldEditor
import joyfill.utils.Option

@Composable
internal fun FullTableHeaderButtons(
    editor: TableFieldEditor,
    onClose: () -> Unit,
) = Box(modifier = Modifier.fillMaxWidth().height(60.dp)) {
    val field = editor.field
    val state = editor.rows.state.watchAsState()
    val measurer = rememberJoyfillTextMeasurer()
    val density = LocalDensity.current

    JoyTitle(
        field = field,
        modifier = Modifier.padding(bottom = 4.dp).align(Alignment.CenterStart)
    )

    Row(
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.End,
        modifier = Modifier.align(Alignment.CenterEnd)
    ) {
        AnimatedVisibility(editor.rows.selection() !is Selection.None) {
            val width = with(density) { measurer.measure("More").size.width.toDp() + 50.dp }
            Box(
                modifier = Modifier.width(width).height(40.dp),
                contentAlignment = Alignment.CenterStart
            ) {
                val trail = if (state.selected.size == 1) "1 Row" else "${state.selected.size} Rows"
                val dup = "Duplicate $trail"
                val edit = "Edit $trail"
                val delete = "Delete $trail"
                val moveUp = "Move Up"
                val moveTop = "Move to the top"
                val moveDown = "Move Down"
                val moveBottom = "Move to the bottom"
                val insertBelow = "Insert below"
                val options = buildList {
                    add(dup)
                    add(edit)
                    if (state.selected.size == 1) {
                        if (!editor.columns.areFiltersApplied()) {
                            add(moveUp)
                            add(moveDown)
                        }
                        add(insertBelow)
                    }
                    add(delete)
                }.map { Option(it, it, false) }

                RawDropField(
                    options = options,
                    value = emptyList(),
                    readonly = false,
                    multiple = false,
                    borders = true,
                    onClick = {
                        when (it.value) {
                            dup -> editor.rows.duplicate()
                            edit -> editor.showRowForm()
                            moveTop -> editor.rows.move.top(state.selected.first().row.id)
                            moveUp -> editor.rows.move.up(state.selected.first().row.id)
                            moveDown -> editor.rows.move.down(state.selected.first().row.id)
                            moveBottom -> editor.rows.move.bottom(state.selected.first().row.id)
                            insertBelow -> editor.rows.addAfter(state.selected.first().row.id)
                            delete -> editor.rows.delete()
                            else -> {}
                        }
                    },
                )
                Text("More", modifier = Modifier.padding(start = 8.dp))
            }
        }
        Icon(
            imageVector = Icons.Filled.Close,
            contentDescription = "close",
            modifier = Modifier.testTag("${field.id}-large-close").clickable(onClick = onClose)
        )
    }
}