package joyfill.shared

import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Delete
import androidx.compose.material.icons.outlined.Redo
import androidx.compose.material.icons.outlined.Undo
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.Path
import androidx.compose.ui.graphics.drawscope.CanvasDrawScope
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalFontFamilyResolver
import androidx.compose.ui.platform.LocalLayoutDirection
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.TextMeasurer
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.drawText
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import joyfill.Image
import joyfill.JoyRawTitle
import joyfill.utils.onDrawing
import joyfill.utils.toByteArray
import kotlin.io.encoding.Base64
import kotlin.io.encoding.ExperimentalEncodingApi

@OptIn(ExperimentalEncodingApi::class)
@Composable
fun RawSignatureInput(
    onCaptured: (String?) -> Unit,
    onCanceled: () -> Unit,
    title: String,
    required: Boolean,
    modifier: Modifier = Modifier,
    url: String? = null,
    testTagId: String = ""
) {
    var value by remember { mutableStateOf(url) }
    val points = remember { mutableStateListOf<Offset>() }
    var drawing by remember { mutableStateOf(false) }
    var cleared by remember { mutableStateOf(false) }

    val paths = remember { mutableStateListOf<Path>() }

    var text by remember { mutableStateOf("") }

    val measurer = TextMeasurer(
        defaultFontFamilyResolver = LocalFontFamilyResolver.current,
        defaultDensity = LocalDensity.current,
        defaultLayoutDirection = LocalLayoutDirection.current
    )
    val density = LocalDensity.current

    var size by remember { mutableStateOf<Size?>(null) }

    val color = Color.Black
    val style = LocalTextStyle.current.copy(
        fontSize = 60.sp,
        fontFamily = FontFamily.Cursive,
        color = color
    )

    Dialog(
        onDismissRequest = onCanceled,
        properties = DialogProperties(usePlatformDefaultWidth = false)
    ) {
        Surface(
            modifier = modifier
                .testTag("${testTagId}-capture")
                .padding(
                    vertical = 16.dp,
                    horizontal = 8.dp
                )
                .fillMaxSize()
        ) {
            Column(modifier = Modifier.fillMaxSize()
                .padding(8.dp)) {
                val v = value

                Row(
                    modifier = Modifier.fillMaxWidth().height(60.dp),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Row(horizontalArrangement = Arrangement.Start) { JoyRawTitle(title, required) }
                    Row(
                        horizontalArrangement = Arrangement.End,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        val redo = remember { mutableStateListOf<Path>() }
                        var deleting by remember { mutableStateOf(false) }

                        if (text.isEmpty() && paths.isNotEmpty() && !deleting) IconButton(onClick = {
                            val path = paths.last()
                            redo.add(path)
                            paths.remove(path)
                        }) {
                            Icon(Icons.Outlined.Undo, "Undo")
                        }

                        if (text.isEmpty() && redo.isNotEmpty() && !deleting) IconButton(onClick = {
                            val path = redo.last()
                            paths.add(path)
                            redo.remove(path)
                        }) {
                            Icon(Icons.Outlined.Redo, "redo")
                        }


                        if (v != null || paths.isNotEmpty() || text.isNotEmpty()) DeleteOption(
                            testTagId = testTagId,
                            deleting = deleting,
                            onDelete = { deleting = true },
                            onConfirm = {
                                value = null
                                paths.clear()
                                text = ""
                                cleared = true
                                deleting = false
                            },
                            onCanceled = { deleting = false }
                        )
                    }
                }

                Box(
                    modifier = Modifier
                        .onDrawing(
                            started = { drawing = true },
                            progressing = {
                                if (drawing) points.add(it)
                            },
                            ended = {
                                drawing = false
                                paths.add(points.toPath())
                                points.clear()
                            }
                        )
                        .fillMaxWidth()
                        .height(200.dp)
                        .border(
                            width = 1.dp,
                            color = Color.White,
                            shape = RoundedCornerShape(4.dp)
                        )
                ) {
                    when {
                        v != null && !drawing && paths.isEmpty() && text.isEmpty() -> Image(
                            v,
                            v,
                            modifier = Modifier.fillMaxWidth().heightIn(200.dp, 400.dp)
                                .clip(RoundedCornerShape(8.dp)).background(Color.White)
                        )
                        else -> Canvas(modifier = Modifier.fillMaxWidth().height(200.dp).background(Color.White)) {
                            if (size == null) {
                                size = this.size
                            }
                            drawSignature(measurer, style, paths, points, text, color)
                        }
                    }
                }

                Row(
                    horizontalArrangement = Arrangement.End,
                    modifier = Modifier.fillMaxWidth().padding(top = 12.dp)
                ) {
                    OutlinedButton(
                        modifier = Modifier.testTag("${testTagId}-capture-cancel"),
                        onClick = onCanceled,
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Cancel")
                    }
                    Spacer(modifier = Modifier.width(8.dp))
                    Button(
                        modifier = Modifier.testTag("${testTagId}-capture-submit"),
                        onClick = {
                            if (text.isEmpty() && paths.isEmpty()) {
                                if (cleared) {
                                    return@Button onCaptured(null)
                                } else {
                                    return@Button
                                }
                            }
                            val width = size?.width?.toInt() ?: return@Button
                            val height = size?.height?.toInt() ?: return@Button

                            val scope = CanvasDrawScope()
                            val bitmap = ImageBitmap(width, height)
                            val canvas = androidx.compose.ui.graphics.Canvas(bitmap)
                            scope.draw(
                                density = density,
                                layoutDirection = LayoutDirection.Ltr,
                                canvas = canvas,
                                size = Size(width.toFloat(), height.toFloat())
                            ) {
                                drawSignature(measurer, style, paths, points, text, Color.Black)
                            }
                            onCaptured("data:image/png;base64," + Base64.encode(bitmap.toByteArray()))
                        },
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Submit")
                    }
                }
            }
        }
    }
}


@Composable
private fun DeleteOption(
    deleting: Boolean,
    onDelete: () -> Unit,
    onCanceled: () -> Unit,
    onConfirm: () -> Unit,
    testTagId: String,
) {
    if (!deleting) IconButton(
        modifier = Modifier.testTag("${testTagId}-capture-delete"),
        onClick = onDelete
    ) {
        Icon(Icons.Outlined.Delete, "Delete", tint = Color.Red)
    } else {
        OutlinedButton(
            modifier = Modifier.testTag("${testTagId}-capture-delete-cancel"),
            onClick = onCanceled,
            shape = RoundedCornerShape(8.dp),
        ) {
            Text("Cancel")
        }
        Spacer(modifier = Modifier.width(8.dp))
        OutlinedButton(
            modifier = Modifier.testTag("${testTagId}-capture-delete-confirm"),
            onClick = onConfirm,
            shape = RoundedCornerShape(8.dp),
            colors = ButtonDefaults.outlinedButtonColors(contentColor = Color.Red)
        ) {
            Text("Confirm")
        }
    }
}

private fun List<Offset>.toPath() = Path().apply {
    forEachIndexed { index, point ->
        if (index == 0) {
            moveTo(point.x, point.y)
        } else {
            lineTo(point.x, point.y)
        }
    }
}

private fun DrawScope.drawSignature(
    measurer: TextMeasurer,
    style: TextStyle,
    paths: List<Path>,
    points: List<Offset>,
    text: String,
    color: Color,
) {
    if (text.isNotEmpty()) {
        val dimension = measurer.measure(text, style).size
        val pos = Offset(size.width - dimension.width, size.height - dimension.height) / 2f
        drawText(textMeasurer = measurer, topLeft = pos, text = text, style = style)
    } else {
        for (p in paths + points.toPath()) drawPath(
            p,
            color = color,
            alpha = 0.5f,
            style = Stroke(width = 4f)
        )
    }
}


