package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import cinematic.watchAsState
import joyfill.editors.RichTextFieldEditor
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonPrimitive

@Composable
internal fun JoyRichTextField(
    editor: RichTextFieldEditor,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    val field = editor.field
    Column(modifier = Modifier.testTag(field.id).fillMaxWidth()) {
        DraftJsRichText(field.value)
        Spacer(modifier = Modifier.height(8.dp))
    }
}

@Composable
private fun DraftJsRichText(value: String?) {
    val codec = remember { Json { ignoreUnknownKeys = true } }
    val v = try {
        codec.decodeFromString(Value.serializer(), value ?: "")
    } catch (error: Throwable) {
        null
    }

    v?.blocks?.forEach { BlockView(it) }
}

@Composable
private fun BlockView(block: Block) {
    val text = buildAnnotatedString {
        append(block.text)
        for (range in block.inlineStyleRanges) {
            val style = when {
                range.style.startsWith("fontsize-") -> {
                    val size = range.style.removePrefix("fontsize-").toIntOrNull() ?: 16
                    SpanStyle(fontSize = size.sp)
                }

                range.style.startsWith("BOLD") -> {
                    SpanStyle(fontWeight = FontWeight.Bold)
                }

                range.style.startsWith("color-") -> {
                    range.style.toColor()?.let { SpanStyle(color = it) }
                }

                else -> null
            }
            if (style == null) continue
            addStyle(style, start = range.offset, end = range.offset + range.length)
        }
    }

    val align = when (block.data["text-align"]?.jsonPrimitive?.content) {
        "center" -> TextAlign.Center
        "right" -> TextAlign.Right
        else -> TextAlign.Left
    }
    Text(text, textAlign = align, modifier = Modifier.fillMaxWidth())
}

@Serializable
private data class Value(
    val blocks: List<Block> = emptyList()
)

@Serializable
private data class Block(
    val text: String = "",
    val inlineStyleRanges: List<StyleRange> = emptyList(),
    val data: JsonObject = JsonObject(mapOf())
)

@Serializable
private data class StyleRange(
    val offset: Int = 0,
    val length: Int = 0,
    val style: String,
)

private fun String.toColor(): Color? {
    if (!startsWith("color-")) {
        return null
    }

    val raw = substringAfter("color-")
    if (raw.startsWith("rgb(")) {
        val rgb = substringAfter("rgb(").substringBefore(")").split(",")
        val r = rgb.getOrNull(0)?.toIntOrNull() ?: return null
        val g = rgb.getOrNull(1)?.toIntOrNull() ?: return null
        val b = rgb.getOrNull(2)?.toIntOrNull() ?: return null
        return Color(r, g, b)
    }

    return null
}