package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ColumnScope
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.StackedLineChart
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Rect
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Path
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.graphics.drawscope.rotate
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.drawText
import androidx.compose.ui.text.rememberTextMeasurer
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import cinematic.watchAsState
import joyfill.chart.LineEditor
import joyfill.chart.PointEditor
import joyfill.editors.AxesEditor
import joyfill.editors.AxisEditor
import joyfill.editors.ChartFieldEditor
import joyfill.fields.ChartField
import joyfill.fields.chart.Line
import joyfill.fields.chart.Point
import joyfill.utils.chartColors
import kotlin.math.round

@Composable
internal fun JoyChartField(
    editor: ChartFieldEditor,
    mode: Mode,
    onSignal: (Signal<Any?>) -> Unit,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    val field = remember(editor) { editor.field }
    var capturing by remember { mutableStateOf(false) }
    val lines = editor.lines.current.watchAsState()

    val readonly = field.disabled || mode == Mode.readonly

    Column(
        modifier = Modifier.testTag(editor.field.id)
            .clickable {
                capturing = true
                onSignal(Signal.Focus)
            }
            .fillMaxWidth()
    ) {
        JoyTitle(field.title, modifier = Modifier.testTag("${field.id}-preview-title"))
        Spacer(modifier = Modifier.height(8.dp))
//        ChartPreview(
//            editor = editor,
//            modifier = Modifier.testTag("${field.id}-preview-button").fillMaxWidth().height(180.dp),
//            xMarkings = 5,
//            yMarkings = 4
//        )
        Box(
            modifier = Modifier.fillMaxWidth()
                .border(width = 2.dp, color = LocalContentColor.current.copy(alpha = 0.4f), shape = RoundedCornerShape(4.dp))
                .height(50.dp)
        ) {
            Icon(
                imageVector = Icons.Filled.StackedLineChart,
                contentDescription = "View Chart",
                modifier = Modifier.align(Alignment.Center)
            )
        }

        if (capturing) Dialog(onDismissRequest = {
            capturing = false
            onSignal(Signal.Blur(Unit))
        }, properties = DialogProperties(usePlatformDefaultWidth = false)) {
            Surface(
                modifier = Modifier.testTag("${field.id}-capture").padding(
                    vertical = 16.dp, horizontal = 8.dp
                ).fillMaxSize(0.96f)
            ) {
                Column(modifier = Modifier.fillMaxWidth().padding(8.dp)) {
                    Row(
                        modifier = Modifier.fillMaxWidth().padding(8.dp),
                        horizontalArrangement = Arrangement.SpaceBetween,
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        JoyTitle(field.title, modifier = Modifier.testTag("${field.id}-capture-title"))
                        Icon(
                            imageVector = Icons.Filled.Close,
                            contentDescription = "${field.id}-capture-close",
                            modifier = Modifier.clickable {
                                capturing = false
                                onSignal(Signal.Blur(Unit))
                            }.padding(8.dp)
                        )
                    }
                    Spacer(modifier = Modifier.height(8.dp))
                    val last = lines.lastOrNull()

//                    ChartPreview(
//                        editor = editor,
//                        modifier = Modifier.testTag("${field.id}-view-button").fillMaxWidth().height(260.dp),
//                        xMarkings = 5,
//                        yMarkings = 5
//                    )

                    CordinatesEditor(editor.axes)

                    Column(modifier = Modifier.fillMaxWidth().verticalScroll(rememberScrollState())) {
                        for (line in lines) Column(modifier = Modifier.fillMaxWidth()) {
                            LineEditor(
                                field = field,
                                editor = line,
                                onDelete = { editor.lines.remove(line.line.id) },
                                readonly = readonly,
                            )
                            if (line != last) Spacer(modifier = Modifier.height(12.dp))
                        }

                        if (!readonly) {
                            Spacer(modifier = Modifier.height(8.dp))
                            OutlinedButton(
                                onClick = { editor.lines.add(title = "Line ${lines.size + 1}") },
                                shape = RoundedCornerShape(8.dp),
                                modifier = Modifier.fillMaxWidth()
                                    .testTag("${field.id}-line-add")
                                    .height(40.dp)
                            ) {
                                Text("Add Chart Line")
                            }
                        }
                    }
                }
            }
        }
    }
}

@Composable
private fun CordinatesEditor(
    editor: AxesEditor
) {
    var editingCoordinates by remember { mutableStateOf(false) }
    OutlinedButton(
        onClick = { editingCoordinates = !editingCoordinates },
        shape = RoundedCornerShape(8.dp),
        modifier = Modifier.fillMaxWidth().height(40.dp)
    ) {
        if (!editingCoordinates) Text("Edit Coordinates") else Text("Hide Coordinates")
    }

    AnimatedVisibility(editingCoordinates) {
        Column(
            modifier = Modifier.padding(4.dp).border(
                width = 2.dp,
                shape = RoundedCornerShape(8.dp),
                color = LocalContentColor.current.copy(alpha = 0.5f)
            ).padding(8.dp)
        ) {
            AxisEditorForm(
                label = "Vertical(Y)",
                editor = editor.yAxis
            )
            Spacer(modifier = Modifier.height(8.dp))
            AxisEditorForm(
                label = "Horizontal(X)",
                editor = editor.xAxis
            )
        }
    }
}

@Composable
private fun AxisEditorForm(
    editor: AxisEditor,
    label: String
) {
    val name = editor.label.watchAsState()
    Text(label)
    Spacer(modifier = Modifier.height(8.dp))
    RawTextField(
        value = name,
        onChange = { editor.label(it) },
        borders = true,
        modifier = Modifier.fillMaxWidth(),
        maxLines = 1,
        readonly = false,
        onFocusChanged = {}
    )

    Row(
        modifier = Modifier.fillMaxWidth(),
        horizontalArrangement = Arrangement.SpaceBetween,
        verticalAlignment = Alignment.Bottom
    ) {
        val min = editor.min.watchAsState()
        NumberEditor(
            label = "Min",
            number = min,
            onChange = { editor.min(it) },
            readonly = false,
            modifier = Modifier.weight(1f).padding(end = 4.dp)
        )

        val max = editor.max.watchAsState()
        NumberEditor(
            label = "Max",
            number = max,
            onChange = { editor.max(it) },
            readonly = false,
            modifier = Modifier.weight(1f).padding(start = 4.dp)
        )
    }
}

@Composable
private fun LineEditor(
    field: ChartField,
    editor: LineEditor,
    onDelete: () -> Unit,
    readonly: Boolean,
    modifier: Modifier = Modifier
) = Deletable(readonly = readonly, onDelete = onDelete, modifier = modifier) {
    val line = remember(editor) { editor.line }
    var name by remember(line) { mutableStateOf(line.title) }
    var description by remember(line) { mutableStateOf(line.description) }
    val points = remember(line.points) { mutableStateListOf(*line.points.toTypedArray()) }

    Text("Title")
    RawTextField(
        value = name,
        onChange = {
            name = it
            editor.title(it)
        },
        borders = true,
        modifier = Modifier.fillMaxWidth().padding(bottom = 8.dp),
        maxLines = 1,
        readonly = readonly,
        onFocusChanged = {}
    )

    Text("Description")
    RawTextField(
        value = description,
        onChange = {
            description = it
            editor.description(it)
        },
        borders = true,
        modifier = Modifier.fillMaxWidth().padding(bottom = 8.dp),
        maxLines = 1,
        readonly = readonly,
        onFocusChanged = {}
    )

    if (points.isNotEmpty()) {
        Text("Points")
        Spacer(modifier = Modifier.height(8.dp))
    }
    for (point in points) {
        PointEditor(
            editor = editor.point(point),
            field = field,
            onDelete = {
                points.remove(point)
                editor.remove(point.id)
            },
            readonly = readonly
        )
        Spacer(modifier = Modifier.height(8.dp))
    }

    if (!readonly) {
        Spacer(modifier = Modifier.height(8.dp))
        OutlinedButton(
            onClick = {
                val point = editor.add()
                points.add(point.point)
            },
            shape = RoundedCornerShape(8.dp),
            modifier = Modifier.fillMaxWidth().height(40.dp)
        ) {
            Text("Add Point")
        }
    }
}

@Composable
private fun PointEditor(
    field: ChartField,
    editor: PointEditor,
    onDelete: () -> Unit,
    readonly: Boolean = false,
    modifier: Modifier = Modifier
) = Deletable(readonly, onDelete, modifier = modifier) {

    val point = remember(editor) { editor.point }
    var label by mutableStateOf(point.label)
    Text("Label")
    Spacer(modifier = Modifier.height(8.dp))
    RawTextField(
        value = label,
        onChange = {
            label = it
            editor.label(it)
        },
        borders = true,
        modifier = Modifier.fillMaxWidth(),
        maxLines = 1,
        readonly = readonly,
        onFocusChanged = {}
    )

    Row(
        modifier = Modifier.fillMaxWidth(),
        horizontalArrangement = Arrangement.SpaceBetween,
        verticalAlignment = Alignment.Bottom
    ) {
        NumberEditor(
            label = field.x.label,
            number = point.x,
            onChange = { editor.x(it) },
            readonly = readonly,
            modifier = Modifier.weight(1f).padding(end = 4.dp)
        )
        NumberEditor(
            label = field.y.label,
            number = point.y,
            onChange = { editor.y(it) },
            readonly = readonly,
            modifier = Modifier.weight(1f).padding(start = 4.dp)
        )
    }
}

@Composable
private fun NumberEditor(
    label: String,
    number: Double,
    onChange: (Double) -> Unit,
    readonly: Boolean = false,
    modifier: Modifier = Modifier
) = Column(modifier) {
    var value by mutableStateOf(number.toString())
    Text(label)
    Spacer(modifier = Modifier.height(8.dp))
    RawTextField(
        value = value,
        onChange = {
            val num = it.toDoubleOrNull() ?: 0.0
            value = num.toString()
            onChange(num)
        },
        borders = true,
        modifier = Modifier.fillMaxWidth(),
        maxLines = 1,
        readonly = readonly,
        onFocusChanged = {}
    )
}

@Composable
private fun Deletable(
    readonly: Boolean,
    onDelete: () -> Unit,
    modifier: Modifier = Modifier,
    content: @Composable ColumnScope.() -> Unit
) = Box(modifier = modifier.fillMaxWidth()) {
    Column(
        modifier = Modifier.padding(4.dp).border(
            width = 2.dp,
            shape = RoundedCornerShape(8.dp),
            color = LocalContentColor.current.copy(alpha = 0.5f)
        ).padding(8.dp),
        content = content
    )
    if (!readonly) Icon(
        imageVector = Icons.Filled.Delete,
        contentDescription = "Delete point",
        tint = LocalContentColor.current,
        modifier = Modifier.align(Alignment.TopEnd)
            .clip(CircleShape)
            .background(color = MaterialTheme.colorScheme.background.copy(alpha = 0.8f))
            .clickable { onDelete() }
            .padding(2.dp)
    )
}

private fun Double.stringify(): String {
    val res = (round(this * 100) / 100).toString()
    if (res.endsWith(".0")) {
        return res.substringBefore(".0")
    }
    return res
}

@Composable
private fun ChartPreview(
    editor: ChartFieldEditor,
    axisWidth: Float = 3f,
    axisColor: Color = LocalContentColor.current,
    gridY: Dp = 40.dp,
    gridX: Dp = 80.dp,
    xMarkings: Int,
    yMarkings: Int,
    markLength: Int = 10,
    modifier: Modifier = Modifier
) {
    val field = remember(editor) { editor.field }
    val lines = editor.lines.current.watchAsState()

    val measure = rememberTextMeasurer()
    val style = remember { TextStyle.Default.copy(color = axisColor) }
    val density = LocalDensity.current
    val ySpacing = remember(gridY) { with(density) { gridY.toPx() } }
    val xSpacing = remember(gridX) { with(density) { gridX.toPx() } }

    Canvas(modifier) {
        val yAxisDim = measure.measure(field.y.label).size
        val xAxisDim = measure.measure(field.x.label).size

        val maxYAxisMarkings = if (ySpacing == 0f) 2 else run {
//            val res = (size.height / ySpacing).toInt()
//            if (res < 2) 2 else res
            yMarkings
        }

        val maxXAxisMarkings = if (xSpacing == 0f) 2 else run {
//            val res = (size.width / xSpacing).toInt()
//            if (res < 2) 2 else res
            xMarkings
        }

        var originX = yAxisDim.height.toFloat()

        repeat(maxYAxisMarkings) { mark ->
            val t = (mark.toFloat() + 1.0) / maxYAxisMarkings
            val value = lerp(field.y.min, field.y.max, t)
            val text = value.stringify()
            val s = measure.measure(text).size
            val textOffsetX = yAxisDim.height.toFloat()
            val lineOffsetX = yAxisDim.height.toFloat() + s.width
            val textOffsetY = (1f - ((mark + 1f) / maxYAxisMarkings)) * (size.height - xAxisDim.height)

            val chartMarginX = lineOffsetX + markLength
            if (chartMarginX > originX) originX = chartMarginX

            val markPos = Offset(textOffsetX, textOffsetY)
            drawText(measure, text, topLeft = Offset(textOffsetX, markPos.y - s.height / 2), style = style)
            drawLine(color = axisColor, start = Offset(lineOffsetX, markPos.y), end = Offset(lineOffsetX + markLength, markPos.y), strokeWidth = axisWidth / 2)
        }

        var originY = size.height - xAxisDim.height

        repeat(maxXAxisMarkings) { mark ->
            val t = (mark.toFloat() + 1.0) / maxXAxisMarkings
            val value = lerp(field.x.min, field.x.max, t)
            val text = value.stringify()
            val s = measure.measure(text).size
            val lineOffsetX = lerp(originX, size.width, t).toFloat()
            val lineOffsetY = size.height - xAxisDim.height - markLength - s.height
            val textOffsetX = lineOffsetX - (s.width / 2)

            val chartMarginY = lineOffsetY
            if (chartMarginY < originY) originY = chartMarginY

            val markPos = Offset(lineOffsetX, lineOffsetY)
            val topLeft = Offset(textOffsetX - (s.width / 2) - 1, lineOffsetY + markLength)
            drawText(measure, text, topLeft, style = style)
            drawLine(color = axisColor, start = markPos, end = Offset(lineOffsetX, lineOffsetY + markLength), strokeWidth = axisWidth / 2)
        }

        val origin = Offset(originX, originY)
        val cords = Size(size.width - yAxisDim.width, size.height - xAxisDim.height)
        val bounds = Rect(offset = origin.copy(y = cords.height - origin.y), cords)
        val yPos = Offset(-(cords.height + yAxisDim.width) / 2, -yAxisDim.height.toFloat() / 2)
        val xPos = Offset(origin.x + (cords.width - xAxisDim.width) / 2, size.height - xAxisDim.height)
        rotate(-90f, pivot = Offset.Zero) {
            drawText(measure, field.y.label, topLeft = yPos, style = style)
        }
        drawText(measure, field.x.label, topLeft = xPos, style = style)
        drawLine(color = axisColor, start = Offset(origin.x, 0f), end = Offset(origin.x, origin.y), strokeWidth = axisWidth)
        drawLine(color = axisColor, start = Offset(origin.x, origin.y), end = Offset(size.width, origin.y), strokeWidth = axisWidth)
        drawCircle(color = axisColor, center = Offset(origin.x, origin.y), radius = axisWidth / 2)

        for (index in lines.indices) {
            val line = lines[index].line
            val color = chartColors[index % chartColors.size]
            draw(field, line, bounds = bounds, color = color, strokeWidth = 2f, pointRadius = 4f)
        }
    }
}

private fun DrawScope.draw(
    field: ChartField,
    line: Line,
    bounds: Rect,
    color: Color,
    strokeWidth: Float,
    pointRadius: Float
) {
    val points = line.points
    for (point in points) draw(field, point, bounds, color, pointRadius)
    val path = line.toCurvedPath(field, bounds)
    drawPath(path, color = color, style = Stroke(width = strokeWidth))
    path.reset()
}

private fun DrawScope.draw(
    field: ChartField,
    point: Point,
    bounds: Rect,
    color: Color,
    radius: Float
) = drawCircle(color = color, center = point.translate(field, bounds), radius = radius)

private fun Line.toCurvedPath(field: ChartField, bounds: Rect): Path {
    val path = Path()
    val points = points.map { it.translate(field, bounds) }
    if (points.isEmpty()) return path
    val first = points.first()
    path.moveTo(first.x, first.y)
    for (index in 0..(points.size - 2)) {
        val curr = points[index]
        val next = points[index + 1]
        val mid = (curr + next) / 2f
        val cp1 = (curr + mid) / 2f
        val cp2 = (mid + next) / 2f
        path.quadraticBezierTo(cp1.x, curr.y, mid.x, mid.y)
        path.quadraticBezierTo(cp2.x, next.y, next.x, next.y)
    }
    return path
}

private fun lerp(min: Float, max: Float, t: Double) = min + (max - min) * t
private fun lerp(min: Double, max: Double, t: Double) = min + (max - min) * t

private fun Point.translate(field: ChartField, bounds: Rect): Offset {
    val tx = (x - field.x.min) / (field.x.max - field.x.min)
    val ty = (y - field.y.min) / (field.y.max - field.y.min)

    val px = lerp(bounds.left, bounds.right, tx)
    val py = lerp(bounds.bottom, bounds.top, ty) - bounds.top
    return Offset(px.toFloat(), py.toFloat())
}