package joyfill.table

import androidx.compose.foundation.clickable
import androidx.compose.foundation.focusable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.BasicTextField
import androidx.compose.foundation.text.input.TextFieldState
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.SolidColor
import androidx.compose.ui.input.key.onKeyEvent
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import joyfill.legacy_compose.generated.resources.Res
import joyfill.legacy_compose.generated.resources.barcode_scanner
import kotlinx.coroutines.launch
import org.jetbrains.compose.resources.painterResource

@Composable
internal fun SearchBarcodeInput(
    value: String?,
    modifier: Modifier = Modifier,
    onChange: (String) -> Unit,
    onCapture: (suspend () -> String?)?,
) {
    val color = LocalContentColor.current
    val state = remember(value) { TextFieldState(initialText = value ?: "") }
    val scope = rememberCoroutineScope()

    BasicTextField(
        state = state,
        modifier = modifier.onKeyEvent {
            onChange(state.text.toString())
            false
        },
        cursorBrush = SolidColor(color),
        textStyle = TextStyle(color, fontSize = 18.sp, fontWeight = FontWeight.W300),
        decorator = {
            Box(
                modifier = Modifier.padding(8.dp),
                contentAlignment = Alignment.CenterStart
            ) {
                if (state.text.isEmpty()) Text("Search...", color = color.copy(alpha = 0.6f))
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.SpaceAround
                ) {
                    it()

                    if (onCapture != null) {
                        Icon(
                            painter = painterResource(Res.drawable.barcode_scanner),
                            contentDescription = "Scan",
                            modifier = Modifier.focusable(false)
                                .clickable {
                                    scope.launch {
                                        val barcode = onCapture.invoke()
                                        onChange(barcode ?: "")
                                    }
                                }
                        )
                    }
                }
            }
        }
    )
}