package joyfill.table

import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.interaction.PressInteraction
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.CalendarMonth
import androidx.compose.material.icons.outlined.RemoveCircleOutline
import androidx.compose.material3.Button
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DisplayMode
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.SelectableDates
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.material3.TimePicker
import androidx.compose.material3.TimePickerState
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import joyfill.Signal
import joyfill.utils.Picking
import joyfill.utils.format
import joyfill.utils.requiresDate
import joyfill.utils.requiresTime
import kotlinx.coroutines.launch
import kotlinx.datetime.Clock
import kotlinx.datetime.TimeZone
import kotlinx.datetime.offsetAt
import kotlinx.datetime.toLocalDateTime
import krono.Instant

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun JoyDateTimeCell(
    value: Long?,
    format: String? = null,
    modifier: Modifier = Modifier,
    readonly: Boolean = false,
    onChange: (Long?) -> Unit = {},
    onSignal: (Signal<Long?>) -> Unit = {},
    borders:Boolean = false,
    cellId: String? = null,
) {
    val now = Clock.System.now().epochSeconds

    var initValue by remember(value) { mutableStateOf<Instant>(Instant((value?: now))) }
    var dialog by remember { mutableStateOf(false) }

    var offsetSecs by remember(value) { mutableStateOf(value?.let {
        val instant = kotlinx.datetime.Instant.fromEpochSeconds(it)
        TimeZone.currentSystemDefault().offsetAt(instant).totalSeconds
    } ?: 0) }

    val pattern = remember {
        format?.replace("YYYY", "{YYYY}")
            ?.replace("MM", "{MM}")
            ?.replace("DD", "{DD}")
            ?.replace("hh", "{hh}")
            ?.replace("mm", "{mm}")
            ?.replace("a", "{ampm}")
            ?: "{YYYY}-{MM}-{DD} {hh}:{mm}"
    }

    val initialPicking = remember(pattern) { if (pattern.requiresDate()) Picking.Date else Picking.Time }
    var picking by remember(initialPicking) { mutableStateOf(initialPicking) }

    val date = rememberDatePickerState(
        initialSelectedDateMillis = value?.plus(offsetSecs)?.times(1000),
        selectableDates = object : SelectableDates {
            override fun isSelectableDate(utcTimeMillis: Long): Boolean = !readonly
            override fun isSelectableYear(year: Int): Boolean = !readonly
        },
        initialDisplayMode = if (readonly) DisplayMode.Input else DisplayMode.Picker
    )

    val time = remember(value, initValue, date.selectedDateMillis) {
        val t = kotlinx.datetime.Instant.fromEpochSeconds(now)
                .toLocalDateTime(TimeZone.currentSystemDefault())
        TimePickerState(
            initialHour = t.hour,
            initialMinute = t.minute,
            is24Hour = true
        )
    }

    val interaction = remember { MutableInteractionSource() }

    LaunchedEffect(interaction, readonly) {
        if (!readonly) launch {
            interaction.interactions.collect {
                if (it is PressInteraction.Release) {
                    picking = initialPicking
                    dialog = true
                    onSignal(Signal.Focus)
                }
            }
        }
    }

    Box(modifier = modifier.fillMaxWidth()) {
        val selectedDate =  value?.plus(offsetSecs)?.times(1000) ?: date.selectedDateMillis //when time not set use this
        OutlinedTextField(
            value = if(value == null) "" else pattern.format(selectedDate, time.hour, time.minute),
            onValueChange = {},
            interactionSource = interaction,
            modifier = Modifier.testTag("${cellId}-body-output").fillMaxWidth(),
            readOnly = true,
            colors = if(borders)
                TextFieldDefaults.colors(
                    focusedContainerColor = Color.Transparent,
                    unfocusedContainerColor = Color.Transparent
                )
            else TextFieldDefaults.colors(
                focusedIndicatorColor = Color.Transparent,
                unfocusedIndicatorColor = Color.Transparent,
                disabledIndicatorColor = Color.Transparent,
                focusedContainerColor = Color.Transparent,
                unfocusedContainerColor = Color.Transparent,
                disabledContainerColor = Color.Transparent,
            ),
        )

        if((value == null || value <= 0) &&  !readonly){
            val modifier = Modifier.align(Alignment.CenterEnd).padding(end = 8.dp)
            Icon(Icons.Outlined.CalendarMonth, "calendar", modifier = modifier.clickable {
                picking = initialPicking
                dialog = true
                onSignal(Signal.Focus)
            })
        }else if(!readonly && value != null && value > 0){
            val modifier = Modifier.align(Alignment.CenterEnd).padding(end = 8.dp)
            Icon(Icons.Outlined.RemoveCircleOutline, null, modifier = modifier.clickable{
                onChange(null)
            })
        }
    }

    if (dialog) Dialog(
        onDismissRequest = {
            dialog = false
            date.selectedDateMillis = value
            initValue = Instant(value ?: now)
            onSignal(Signal.Blur(value))
        },
        properties = DialogProperties(usePlatformDefaultWidth = false)
    ) {
        Surface(modifier = Modifier.fillMaxWidth(0.95f)) {
            Column(modifier = Modifier.padding(8.dp).fillMaxWidth()) {
                when (picking) {
                    Picking.Date -> DatePicker(
                        state = date,
                        title = null,
                        headline = null,
                        modifier = Modifier.testTag("${cellId}-input-date").padding(14.dp),
                    )

                    Picking.Time -> TimePicker(
                        state = time,
                        modifier = Modifier.testTag("${cellId}-input-time")
                    )
                }

                Row(horizontalArrangement = Arrangement.End, modifier = Modifier.fillMaxWidth().padding(top = 12.dp)) {
                    OutlinedButton(
                        modifier = Modifier.testTag("${cellId}-input-cancel"),
                        onClick = {
                            dialog = false
                            date.selectedDateMillis = value?.plus(offsetSecs)
                            initValue = Instant(value ?: now)
                            onSignal(Signal.Blur(value))
                        },
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Cancel")
                    }
                    Spacer(modifier = Modifier.width(8.dp))
                    Button(
                        modifier = Modifier.testTag("${cellId}-input-submit"),
                        onClick = {
                            when (picking) {
                                Picking.Date -> {
                                    if (pattern.requiresTime()) {
                                        picking = Picking.Time
                                    } else {
                                        val secs = date.selectedDateMillis?.div(1000)
                                        onSignal(Signal.Change(secs))
                                        dialog = false
                                        onSignal(Signal.Blur(secs))
                                        onChange(secs)
                                    }
                                }

                                Picking.Time -> {
                                    val minutes = time.minute + (time.hour * 60)
                                    val millis = (date.selectedDateMillis ?: 0L) + (minutes * 60 * 1000L)
                                    val secs = millis.div(1000)
                                    onSignal(Signal.Change(secs))
                                    dialog = false
                                    onSignal(Signal.Blur(secs))
                                    onChange(secs)
                                }
                            }
                        },
                        shape = RoundedCornerShape(8.dp),
                    ) {
                        Text("Submit")
                    }
                }
            }
        }
    }
}