package joyfill.table

import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.requiredHeight
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.FilterAlt
import androidx.compose.material3.Checkbox
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.FieldPosition
import joyfill.Invalid
import joyfill.Mode
import joyfill.editors.TableFieldEditor
import joyfill.fields.Field
import joyfill.fields.table.UnknownColumn

private val unsupportedFilters = listOf(Field.Type.date, Field.Type.image, Field.Type.progress, Field.Type.block, Field.Type.signature, Field.Type.unknown)

@Composable
internal fun FullTableBody(
    editor: TableFieldEditor,
    position: FieldPosition?,
    state: LazyListState,
    showUnsupportedColumns: Boolean,
    mode: Mode,
    uploadHandler: (suspend () -> List<String>)?,
    barcodeHandler: (suspend () -> String?)?
) {
    val rowsState = editor.rows.state.watchAsState()
    val columnsState = editor.columns.state.watchAsState()
    val rows = when (val filtered = rowsState.filtered) {
        null -> rowsState.haystack
        else -> filtered
    }

    val columns = remember(showUnsupportedColumns, editor) {
        val them = editor.field.columns
        when (showUnsupportedColumns) {
            true -> them
            else -> them.filter { it.type != Field.Type.unknown }
        }
    }
    val  visibleCols = columns.filter { col -> position?.columns?.find { it.id == col.id }?.hidden != true }

    LazyRowArrayTable(
        state = state,
        columns = visibleCols,
        rows = rows,
        modifier = Modifier.fillMaxWidth().fillMaxHeight(0.9f)
    ) { placed, column ->
        val measurer = rememberJoyfillTextMeasurer()
        val density = LocalDensity.current

        val width = if (column == null) with(density) {
            measurer.measure("${rows.size}").size.width.toDp() + 70.dp
        } else colWidth(column.type, preview = false, measurer)

        val selected = column != null && columnsState.current?.column?.id == column.id

        val wrapperModifier = if (selected && placed == null) {
            Modifier.border(width = 2.dp, color = LocalContentColor.current)
        } else {
            Modifier
        }.width(width).fillMaxHeight().clickable { editor.columns.select(column) }

        val typography = MaterialTheme.typography.bodyLarge
        val lineHeight = typography.lineHeight
        val height = with(density) { lineHeight.toDp().times(3) }

        Box(wrapperModifier.heightIn(min = height), contentAlignment = Alignment.Center) {
            when {
                column == null && placed == null -> Row(
                    horizontalArrangement = Arrangement.End,
                    verticalAlignment = Alignment.CenterVertically,
                    modifier = Modifier.fillMaxWidth().requiredHeight(height)
                ) {
                    Checkbox(
                        checked = editor.rows.selection() is Selection.All,
                        onCheckedChange = { checked -> if (checked) editor.rows.selectAll() else editor.rows.unSelectAll() }
                    )
                }

                column == null && placed != null -> Row(
                    horizontalArrangement = Arrangement.SpaceBetween,
                    verticalAlignment = Alignment.CenterVertically,
                    modifier = Modifier.fillMaxWidth().requiredHeight(height).padding(start = 8.dp)
                ) {
                    val rx = placed.editor
                    Text(text = "${placed.number}", color = LocalContentColor.current.copy(alpha = 0.6f))
                    Checkbox(checked = editor.rows.isSelected(rx), onCheckedChange = { checked ->
                        if (checked) editor.rows.select(rx) else editor.rows.unselect(rx)
                    })
                }

                column != null && placed == null -> Row(
                    horizontalArrangement = Arrangement.SpaceBetween,
                    verticalAlignment = Alignment.CenterVertically,
                    modifier = Modifier.width(width).padding(horizontal = 8.dp)
                ) {
                    val text = buildAnnotatedString {
                        append("${column.title} ")
                        when {
                            column is UnknownColumn -> withStyle(SpanStyle(color = Color.Yellow)) { append("!") }
                            columnsState.validity(column) is Invalid -> withStyle(SpanStyle(color = Color.Red)) { append("*") }
                        }
                    }
                    Text(text = text, modifier = Modifier.weight(1f))
                    val alpha = if (columnsState.isFiltered(column)) 1.0f else 0.6f
                    val color = LocalContentColor.current.copy(alpha = alpha)

                    if (column.type !in unsupportedFilters) Icon(
                        imageVector = Icons.Outlined.FilterAlt,
                        contentDescription = "Filter Icon",
                        tint = color
                    )
                }

                else -> {
                    val cell = remember(column?.id, placed?.editor) { placed?.editor?.col(column?.id)?.also { it.validate() } }
                    TableCell(
                        cell = cell,
                        minHeight = height,
                        visibleCols = visibleCols,
                        rowsState = rowsState,
                        rowPlacement = placed,
                        position = position,
                        mode = mode,
                        onUpload = uploadHandler,
                        onCapture = barcodeHandler,
                    )
                }
            }
        }
    }
}