package joyfill.table

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.slideIn
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.Surface
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.key
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import cinematic.watchAsState
import joyfill.FieldEvent
import joyfill.Mode
import joyfill.Page
import joyfill.editors.TableFieldEditor

@Composable
internal fun JoyTableField(
    editor: TableFieldEditor,
    page: Page,
    previewRows: Int,
    mode: Mode,
    onUpload: (suspend (FieldEvent) -> List<String>)?,
    onCapture: (suspend (FieldEvent) -> String?)?,
    showUnsupportedColumns: Boolean,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    var view by remember { mutableStateOf(UIView.small) }
    val state = editor.rows.state.watchAsState()
    val rows = state.selected
    val position = page.positions.find { it.field == editor.field.id }

    key(view){ //reload preview data on closing dialog (to show correct progress)
        Preview(editor, position, previewRows, totalRows = rows.size, onClick = { view = UIView.large })
    }

    val uploadHandler = if (onUpload != null) {
        suspend {
            val event = FieldEvent(
                field = editor.field,
                page = page
            )
            onUpload(event)
        }
    } else null

    val barcodeHandler = if (onCapture != null) {
        suspend {
            val event = FieldEvent(
                field = editor.field,
                page = page
            )
            onCapture(event)
        }
    } else null

    val display = position?.displayType

    AnimatedVisibility(
        visible = view == UIView.large,
        enter = slideIn { IntOffset.Zero },
        modifier = Modifier.testTag("${editor.field.id}-large")
    ) {
        Dialog(
            onDismissRequest = { view = UIView.small },
            properties = DialogProperties(usePlatformDefaultWidth = false)
        ) {
            Surface(modifier = Modifier.fillMaxSize(0.95f)) {
                when (display) {
                    "inputGroup" -> FullGroupView(editor, position, mode, showUnsupportedColumns, onClose = { view = UIView.small }, uploadHandler, barcodeHandler)
                    else -> FullTabularView(editor, position, mode, showUnsupportedColumns, onClose = { view = UIView.small }, uploadHandler, barcodeHandler)
                }
            }
        }
    }
}