package joyfill.table

import androidx.compose.foundation.clickable
import androidx.compose.foundation.focusable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.input.KeyboardType
import joyfill.legacy_compose.generated.resources.Res
import joyfill.legacy_compose.generated.resources.barcode_scanner
import kotlinx.coroutines.launch
import org.jetbrains.compose.resources.painterResource

@Composable
internal fun JoyBarcodeCell(
    value: String?,
    placeholder: String? = null,
    borders: Boolean = true,
    onChange: (String?) -> Unit,
    modifier: Modifier = Modifier,
    onCapture: (suspend () -> String?)?,
    maxLines: Int = Int.MAX_VALUE,
    readonly: Boolean = false,
    keyboard: KeyboardOptions = KeyboardOptions.Default.copy(keyboardType = KeyboardType.Text),
    minLines: Int = 1,
) {
    var v by remember(value) { mutableStateOf(value ?: "") }
    val scope = rememberCoroutineScope()

    Box(modifier = modifier) {
        OutlinedTextField(
            value = v,
            onValueChange = {
                v = it
                onChange(it)
            },
            readOnly = readonly,
            modifier = Modifier.fillMaxWidth(),
            singleLine = maxLines == 1,
            maxLines = maxLines,
            placeholder = {
                if (placeholder != null) Text(
                    placeholder,
                    color = LocalContentColor.current.copy(alpha = 0.6f)
                )
            },
            keyboardOptions = keyboard,
            minLines = minLines,
            colors = if (borders) {
                OutlinedTextFieldDefaults.colors()
            } else {
                OutlinedTextFieldDefaults.colors(
                    focusedBorderColor = Color.Companion.Transparent,
                    unfocusedBorderColor = Color.Companion.Transparent,
                )
            },
            trailingIcon = {
                Icon(
                    painter = painterResource(Res.drawable.barcode_scanner),
                    contentDescription = "Scan",
                    modifier = Modifier.focusable(false).clickable {
                        scope.launch {
                            onCapture?.invoke()?.let {
                                v = it
                                onChange(it)
                            }
                        }
                    }
                )
            }
        )
    }
}